/*
 * #%L
 * BOW UI
 * %%
 * Copyright (C) 2010 - 2015 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow.action.group;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.bow.BowGroup;
import org.chorem.bow.BowGroupImpl;
import org.chorem.bow.BowProxy;
import org.chorem.bow.BowSession;
import org.chorem.bow.BowUser;
import org.chorem.bow.BowUtils;
import org.chorem.bow.action.BowBaseAction;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.entities.WikittyUserHelper;
import org.nuiton.wikitty.query.WikittyQuery;
import org.nuiton.wikitty.query.WikittyQueryMaker;
import org.nuiton.wikitty.query.WikittyQueryResult;
import org.nuiton.wikitty.services.WikittyServiceAuthorisation;

/**
 * Handles RSS flux demande
 * - tagLine
 * - fullTextLine
 * - count: number of bookmark to retrieve
 *
 * @author poussin
 * @version $Revision$
 *          <p/>
 *          Last update: $Date$
 *          by : $Author$
 */
public class GroupEditAction extends BowBaseAction {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(GroupEditAction.class);

    private static final long serialVersionUID = 1L;

    protected String groupId;
    protected BowGroup group;

    public String getGroupId() {
        return groupId;
    }

    public void setGroupId(String groupId) {
        this.groupId = groupId;
    }

    public BowGroup getGroup() {
        if (group == null) {
            loadGroup();
        }
        return group;
    }
    
    public boolean isPublic() {
        return BowUtils.isPublicGroup(getGroup());
    }

    public void setPublic(boolean pub) {
        String reader;
        if (pub && getConfig().isAllowPublicGroup()) {
            reader = WikittyServiceAuthorisation.PUBLIC;
        } else {
            reader = getGroup().getWikittyId();
        }
        getGroup().setReader(Collections.singleton(reader));
    }

    public void loadGroup() {
        if (log.isDebugEnabled()) {
            log.debug("group load " + group);
        }
        BowProxy proxy = getBowProxy();
        if (StringUtils.isBlank(groupId)) {
            BowSession session = getBowSession();
            BowUser user = session.getUser();

            group = new BowGroupImpl();
            // add Wikitty to force add in preload too
            group.addMembers(proxy.castTo(Wikitty.class, user));
            setPublic(false);
        } else {
            group = proxy.restore(BowGroup.class, groupId, "WikittyGroup.members");
        }
        if (log.isDebugEnabled()) {
            log.debug("group loaded " + group);
        }
    }

    public String load() throws Exception {
        return SUCCESS;
    }

    public String removeMe() throws Exception {
        String result = SUCCESS;
        try {
            BowSession session = getBowSession();

            BowUser user = session.getUser();
            BowProxy proxy = session.getProxy();

            // group is loaded by prepare method
            getGroup().removeMembers(user.getWikittyId());
            if (CollectionUtils.isEmpty(getGroup().getMembers())) {
                addActionMessage(t("bow.group.delete.no.member"));
                proxy.delete(group);
            } else {
                proxy.store(group);
            }

        } catch (Exception eee) {
            addActionError(t("bow.error.internal"));
            log.error(eee.getMessage(), eee);
            result = ERROR;
        }
        return result;
    }

    public String save() throws Exception {
        if (log.isDebugEnabled()) {
            log.debug("group save " + group);
        }
        String result = SUCCESS;
        try {
            BowSession session = getBowSession();
            BowProxy proxy = session.getProxy();

            if (StringUtils.isNotBlank(getGroup().getName())) {
                Map<String, String> logins = new HashMap<String, String>();
                // convert user login to id
                for (String login : getGroup().getMembers()) {
                    if (StringUtils.startsWith(login, "add ")) {
                        logins.put(StringUtils.substringAfter(login, "add "), login);
                    }
                }
                if (!logins.isEmpty()) {
                    WikittyQuery criteria = new WikittyQueryMaker().containsOne(
                            BowUser.ELEMENT_FIELD_WIKITTYUSER_LOGIN, logins.keySet()).end();
                    WikittyQueryResult<Wikitty> users = proxy.findAllByQuery(Wikitty.class, criteria);
                    getGroup().removeMembers(logins.values().toArray(new String[logins.size()]));
                    getGroup().addMembers(users.getAll().toArray(new Wikitty[users.size()]));

                    if (users.size() != logins.size()) {
                        for (Wikitty w : users) {
                            logins.remove(WikittyUserHelper.getLogin(w));
                        }
                        addActionError(t("bow.group.some.user.not.found",
                                StringUtils.join(logins.keySet(), ", ")));
                        result = ERROR;
                    }

                }
                if (result == SUCCESS && CollectionUtils.isEmpty(getGroup().getMembers())) {
                    addActionMessage(t("bow.group.delete.no.member"));
                    proxy.delete(group);
                } else {
                    // force admin to group all time
                    getGroup().setAdmin(Collections.singleton(getGroup().getWikittyId()));

                    proxy.store(group);
                }
            } else {
                addActionError(t("bow.group.name.blank"));
                result = ERROR;
            }

        } catch (Exception eee) {
            addActionError(t("bow.error.internal"));
            log.error(eee.getMessage(), eee);
            result = ERROR;
        }
        return result;
    }

    public String delete() throws Exception {
        if (log.isDebugEnabled()) {
            log.debug("group delete " + group);
        }
        String result = SUCCESS;
        try {
            BowSession session = getBowSession();
            BowProxy proxy = session.getProxy();
            proxy.delete(group);

        } catch (Exception eee) {
            addActionError(t("bow.error.internal"));
            log.error(eee.getMessage(), eee);
            result = ERROR;
        }
        return result;
    }

}
