/*
 * #%L
 * BOW UI
 * %%
 * Copyright (C) 2010 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow.action.bookmark;

import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.bow.BookmarkUtils;
import org.chorem.bow.BowBookmark;
import org.chorem.bow.BowBookmarkImpl;
import org.chorem.bow.BowConfig;
import org.chorem.bow.BowGroup;
import org.chorem.bow.BowProxy;
import org.chorem.bow.BowUser;
import org.chorem.bow.BowUtils;
import org.chorem.bow.action.BowBaseAction;
import org.nuiton.wikitty.query.WikittyQuery;
import org.nuiton.wikitty.query.WikittyQueryMaker;
import org.nuiton.wikitty.query.WikittyQueryResult;

/**
 * Ajoute ou modifie un bookmark, soit par le formulaire,
 * soit par un appel javascript
 *
 * @author poussin
 */
public class AddOrUpdateAction extends BowBaseAction {

    private static final long serialVersionUID = 3389170166034184139L;

    private static Log log = LogFactory.getLog(AddOrUpdateAction.class);
    
    protected String bookmarkId;

    protected String link;

    protected String name;

    protected String privateAlias;

    protected String publicAlias;

    protected String tags;

    protected String nameAndTags;

    protected String redirectTo;

    public String getBookmarkId() {
        return bookmarkId;
    }

    public void setBookmarkId(String bookmarkId) {
        this.bookmarkId = bookmarkId;
    }

    /** @return the link */
    public String getLink() {
        return link;
    }

    /** @param link the link to set */
    public void setLink(String link) {
        this.link = link;
    }

    /** @return the description */
    public String getName() {
        return name;
    }

    /** @param name the description to set */
    public void setName(String name) {
        this.name = name;
    }

    /** @return the privateAlias */
    public String getPrivateAlias() {
        return privateAlias;
    }

    /** @param privateAlias the privateAlias to set */
    public void setPrivateAlias(String privateAlias) {
        this.privateAlias = privateAlias;
    }

    public String getPublicAlias() {
        return publicAlias;
    }

    public void setPublicAlias(String publicAlias) {
        this.publicAlias = publicAlias;
    }

    /** @return the tags */
    public String getTags() {
        return tags;
    }

    /** @param tags the tags to set */
    public void setTags(String tags) {
        this.tags = tags;
    }

    /** @return the nameAndTags */
    public String getNameAndTags() {
        return nameAndTags;
    }

    /** @param nameAndTags the nameAndTags to set */
    public void setNameAndTags(String nameAndTags) {
        this.nameAndTags = nameAndTags;
    }

    /** @return the redirectTo */
    public String getRedirectTo() {
        return redirectTo;
    }

    @Override
    public String execute() {
        String result = SUCCESS;
        try {
            boolean isScriptlet = false;
            BowProxy proxy = getBowProxy();
            BowBookmark bookmark = null;

            // if we don't have link, link is last entry in nameAndTags
            if (StringUtils.isBlank(link)) {
                if (StringUtils.contains(nameAndTags, "|")) {
                    link = StringUtils.substringAfterLast(nameAndTags, "|");
                    nameAndTags = StringUtils.substringBeforeLast(nameAndTags, "|");
                } else {
                    link = nameAndTags;
                    nameAndTags = "";
                }
            }

            // gere le cas d'une mise a jour ou d'un ajout via le formulaire
            if (name == null) {
                // on est dans le cas ou l'ajout a ete fait par le scriptlet qui
                // utilise une seul chaine pour name et tags, on decoupe
                // et on continue normalement
                isScriptlet = true;
                nameAndTags = StringUtils.trimToEmpty(nameAndTags);
                name = StringUtils.substringBeforeLast(nameAndTags, "|");
                tags = StringUtils.substringAfterLast(nameAndTags, "|");
            }


            // on met toutes les valeurs dans le bon format
            name = BowUtils.normalizeString(name);
            link = BowUtils.normalizeUrl(link);
            tags = BowUtils.normalizeString(tags);
            privateAlias = BowUtils.normalizeString(privateAlias);
            publicAlias = BowUtils.normalizeString(publicAlias);


            BowUser user = getBowSession().getUser();
            String userId = user.getWikittyId();
            if (StringUtils.isNotBlank(bookmarkId)) {
                // on modifie un bookmark existant et on a deja son id
                bookmark = proxy.restore(BowBookmark.class, bookmarkId);
                if (log.isDebugEnabled()) {
                    log.debug("Change old wikitty: " + bookmark);
                }
            } else {
                WikittyQuery criteria = new WikittyQueryMaker().and()
                        .eq(BowBookmark.FQ_FIELD_BOWBOOKMARK_LINK, link)
                        .eq(BowBookmark.FQ_FIELD_WIKITTYAUTHORISATION_OWNER, userId)
                        .end();
                bookmark =
                        proxy.findByQuery(BowBookmark.class, criteria);
                if (bookmark != null) {
                    // on modifie le bookmark qui a la meme url que celle a ajouter
                    // on fusionne les descriptions
                    String description = bookmark.getDescription();
                    name = description +"\n"
                            + BowUtils.formatDate(new Date()) + ": " + name;
                    // on fusionne les tags
                    tags += " " + BookmarkUtils.getBookmarkTagsString(bookmark);
                }
            }

            // on a pas retrouve de bookmark a modifier, on en cree un nouveau
            if (bookmark == null) {
                bookmark = new BowBookmarkImpl();
                bookmark.setClick(0);
                bookmark.setOwner(user);
                bookmark.addReader(user.getWikittyId()); // only owner can read it
                bookmark.setCreationDate(new Date());

                if (user.getScreenshot()) {
                    // on essaie de prendre l'image du site que l'on ajoute
                    byte[] screenshot = BookmarkUtils.getScreenshot(link);
                    bookmark.setScreenshot(screenshot);
                }

                if (user.getFavicon()) {
                    // on essaie de recuperer le favicone
                    byte[] favicone = BookmarkUtils.getFavicon(link);
                    bookmark.setFavicon(favicone);
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Change old bookmark: " + bookmark);
                }
            }

            // on met a jour les infos
            bookmark.setDescription(name);
            bookmark.setLink(link);
            Set<String> tagsWord = BowUtils.getWords(tags);
            bookmark.setLabels(tagsWord);

            // add in WikittyAuthorisation.parent all groups (tag start with @)
            // where current user is member
            Set<String> groupsName = BowUtils.getGroups(tagsWord);
            if (CollectionUtils.isNotEmpty(groupsName)) {
                WikittyQuery groupsQuery = new WikittyQueryMaker().and()
                        .containsOne(BowGroup.ELEMENT_FIELD_WIKITTYGROUP_NAME, groupsName)
                        .eq(BowGroup.ELEMENT_FIELD_WIKITTYGROUP_MEMBERS, userId)
                        .end();
                WikittyQueryResult<String> groupsId = proxy.findAllByQuery(groupsQuery);
                bookmark.setParent(new HashSet<String>(groupsId.getAll()));

                if (groupsName.size() != groupsId.size()) {
                    List<BowGroup> groups = proxy.restore(BowGroup.class, groupsId.getAll());
                    for (BowGroup g : groups) {
                        groupsName.remove(g.getName());
                    }
                    for (String groupName : groupsName) {
                        bookmark.removeLabels(BowConfig.GROUP_MARK + groupName);
                    }
                    addActionMessage(t("bow.group.not.members", StringUtils.join(groupsName, ", ")));
                }
            }

            // Si l'alias prive souhaite est deja utilise on ne l'accept pas
            // Si l'alias public souhaite est deja utilise on ne l'accept pas
            if (StringUtils.isBlank(privateAlias)) {
                bookmark.setPrivateAlias("");
            } else {
                WikittyQuery privateAliasCriteria = new WikittyQueryMaker().and()
                        .eq(BowBookmark.FQ_FIELD_BOWBOOKMARK_PRIVATEALIAS, privateAlias)
                        .eq(BowBookmark.FQ_FIELD_WIKITTYAUTHORISATION_OWNER, userId)
                        .end().setLimit(1);
                WikittyQueryResult<String> aliasResult = proxy.findAllByQuery(
                        privateAliasCriteria);
                if (aliasResult.getTotalResult() == 0 ||
                        bookmark.getWikittyId().equals(aliasResult.get(0))) {
                    bookmark.setPrivateAlias(privateAlias);
                } else {
                    addActionMessage(t("bow.alias.already.exists", privateAlias));
                }
            }

            if (StringUtils.isBlank(publicAlias)) {
                bookmark.setPublicAlias("");
            } else {
                WikittyQuery publicAliasCriteria = new WikittyQueryMaker()
                        .eq(BowBookmark.FQ_FIELD_BOWBOOKMARK_PUBLICALIAS, publicAlias)
                        .end().setLimit(1);
                WikittyQueryResult<String> aliasResult = proxy.findAllByQuery(
                        publicAliasCriteria);
                if (aliasResult.getTotalResult() == 0 ||
                        bookmark.getWikittyId().equals(aliasResult.get(0))) {
                    bookmark.setPublicAlias(publicAlias);
                } else {
                    addActionMessage(t("bow.alias.already.exists", publicAlias));
                }
            }

            proxy.store(bookmark); //Stores the bookmark if everything is ok
            addActionMessage(t("bow.bookmark.add.successful"));
            if (log.isDebugEnabled()) {
                log.debug("Adding URL");
            }

            if (isScriptlet && StringUtils.isBlank(bookmark.getDescription())
                    && CollectionUtils.isEmpty(bookmark.getLabels())) {
                // le bookmark est trop peu renseigne, on renvoie vers l'edition
                redirectTo = String.format("editBookmark.action?id=%s", bookmark.getWikittyId());
            } else {
                // on essai d'afficher l'entree ajoutee/modifiee
                if (StringUtils.isEmpty(tagLine) && StringUtils.isEmpty(fullTextLine)) {
                    // il n'y avait pas de filtre on filtre sur l'element qui vient d'etre modifie
                    listId = Collections.singletonList(bookmark.getWikittyId());
                    redirectTo = BowUtils.redirectTo(listId, tagLine, fullTextLine);
                } else {
                    redirectTo = BowUtils.redirectTo(tagLine, fullTextLine);
                }
            }
        } catch (Exception eee) {
            result = ERROR;
            addActionError(t("bow.error.internal"));
            log.error(eee.getMessage(), eee);
        }
        return result;
    }

}
