/*
 * #%L
 * BOW UI
 * %%
 * Copyright (C) 2010 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow.action;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.bow.BowBookmark;
import org.chorem.bow.BowProxy;
import org.chorem.bow.BowUser;
import org.nuiton.wikitty.query.WikittyQuery;
import org.nuiton.wikitty.query.WikittyQueryMaker;

/**
 * Handles only public aliases. Private aliases are handled by open search.
 *
 * @author poussin
 * @version $Revision$
 *          <p/>
 *          Last update: $Date$
 *          by : $Author$
 */
public class AliasAction extends BowBaseAction {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(AliasAction.class);

    private static final long serialVersionUID = 1L;

    protected String redirectTo = "/";

    protected String alias;

    protected String bookmarkId;

    public String getRedirectTo() {
        return redirectTo;
    }

    public void setAlias(String alias) {
        this.alias = alias;
    }

    public void setBookmarkId(String bookmarkId) {
        this.bookmarkId = bookmarkId;
    }

    @Override
    public String execute() throws Exception {
        if (log.isDebugEnabled()) {
            log.debug("Redirect alias " + alias);
        }
        String result = SUCCESS;
        try {
            if (StringUtils.isNotEmpty(alias)) {
                BowProxy proxy = getBowProxy();

                // HACK: poussin 20150720, security access is check for read/write
                // with wikitty security. No logged user can't restore not readable by him
                // without security exception. We use wikitty security problem
                // with select query that permit to access all field (but not object)
                // to support public alias. But we must find better way.
                // with this hack, we can't count bookmark access via public alias
                WikittyQuery criteria = new WikittyQueryMaker()
                        .select(BowBookmark.ELEMENT_FIELD_BOWBOOKMARK_LINK).where().or()
                        .ideq(alias)
                        .eq(BowBookmark.FQ_FIELD_BOWBOOKMARK_PUBLICALIAS, alias)
                        .end();

                String link = proxy.findByQuery(criteria);
                if (link == null) {
                    // not id or public alias, looking for private if user is logged
                    BowUser user = getBowSession().getUser();
                    if (user != null) {
                        criteria = new WikittyQueryMaker().and()
                                .eq(BowBookmark.FQ_FIELD_WIKITTYAUTHORISATION_OWNER, user.getWikittyId())
                                .eq(BowBookmark.FQ_FIELD_BOWBOOKMARK_PRIVATEALIAS, alias)
                                .end();
                        BowBookmark bookmark = proxy.findByQuery(BowBookmark.class, criteria);
                        link = bookmark.getLink();
                        int clicks = bookmark.getClick() + 1;
                        bookmark.setClick(clicks);
                        proxy.store(bookmark);
                    }
                }
                if (link != null) {
                    redirectTo = link;
                } else {
                    log.error(String.format("No bookmark with alias or id '%s'", alias));
                    addActionError(t("bow.alias.bookmarkId.unknown", alias));
                    result = ERROR;
                }
            }
        } catch (Exception eee) {
            addActionError(t("bow.error.internal"));
            log.error(eee.getMessage(), eee);
            result = ERROR;
        }
        return result;
    }

    public String create() throws Exception {
        String result = SUCCESS;
        try {
            if (!StringUtils.isEmpty(alias)) {
                BowProxy proxy = getBowProxy();

                WikittyQuery criteria = new WikittyQueryMaker().eq(
                        BowBookmark.FQ_FIELD_BOWBOOKMARK_PUBLICALIAS, alias).end();

                BowBookmark bookmark = proxy.findByQuery(BowBookmark.class, criteria);
                if (bookmark != null) {
                    addActionError(t("bow.alias.already.exists", alias));
                    log.error("A bookmark already exists with alias " + alias);
                    result = ERROR;

                } else {
                    bookmark = proxy.restore(BowBookmark.class, bookmarkId);
                    if (bookmark != null) {
                        bookmark.setPublicAlias(alias);
                        proxy.store(bookmark);
                    } else {
                        addActionError(t("bow.alias.bookmarkId.unknown", bookmarkId));
                        log.debug("No bookmark with id " + bookmarkId);
                        result = ERROR;
                    }
                }
            }
        } catch (Exception eee) {
            addActionError(t("bow.error.internal"));
            log.error(eee.getMessage(), eee);
            result = ERROR;
        }

        return result;
    }

    public String delete() throws Exception {
        log.debug("delete alias for id " + bookmarkId);
        String result = SUCCESS;
        try {
            BowProxy proxy = getBowProxy();

            BowBookmark bookmark = proxy.restore(BowBookmark.class, bookmarkId);
            if (bookmark != null) {
                bookmark.setPublicAlias(null);
                proxy.store(bookmark);
            } else {
                addActionError(t("bow.alias.bookmarkId.unknown", bookmarkId));
                log.error("No bookmark with id " + bookmarkId);
                result = ERROR;
            }
        } catch (Exception eee) {
            addActionError(t("bow.error.internal"));
            log.error(eee.getMessage(), eee);
            result = ERROR;
        }

        return result;
    }

}
