/*
 * #%L
 * BOW UI
 * %%
 * Copyright (C) 2010 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.nuiton.util.StringUtil;

import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.collections.CollectionUtils;
import org.nuiton.wikitty.query.WikittyQuery;
import org.nuiton.wikitty.query.WikittyQueryMaker;
import org.nuiton.wikitty.services.WikittyServiceAuthorisation;

public class BowUtils {

    public static final String BOW_DATE_SHORT_PATTERN = "yyyy/MM/dd";

    /**
     * public to use easily in jsp
     */
    public BowUtils() {
    }

    public static String toBase64(byte[] b) {
        return Base64.encodeBase64String(b);
    }

    /**
     * Retourne toujours une date valide. Si date est null, on prend la date
     * du jour
     *
     * @param date null or valide date
     * @return date if date not null, or today date
     */
    public static Date normalizeDate(Date date) {
        Date result = date;
        if (result == null) {
            result = new Date();
        }
        return result;
    }

    /**
     * Supprime les blancs de debut et de fin, et retourne une chaine vide si s
     * est null
     *
     * @param s null or string
     * @return empty string or string without blank at begin and end
     */
    public static String normalizeString(String s) {
        String result = StringUtils.trimToEmpty(s);
        return result;
    }

    /**
     * Normalise l'url, si elle est null ou vide retourne une chaine vide.
     * Si elle ne commence pas par un protocol (ex: ftp://) alors on ajoute
     * automatiquement http://
     *
     * @param url l'url a normaliser
     * @return l'url normalisee
     */
    public static String normalizeUrl(String url) {
        String result = "";
        if (StringUtils.isNotBlank(url)) {
            result = url.trim();
            if (!result.matches("[a-z]+://.+")) {
                result = "http://" + result;
            }
        }
        return result;
    }

    /**
     * Donne la liste des mots de la chaine passee en parametre.
     * Si tags est null ou vide, le 'set' retourne sera vide.
     * <p/>
     * La liste retournee ne doit jamais contenir de chaine null ou vide.
     *
     * @param tags
     * @return
     */
    public static Set<String> getWords(String tags) {
        String[] words = StringUtils.split(tags);
        words = ArrayUtils.nullToEmpty(words);

        Set<String> result = new HashSet<String>(Arrays.asList(words));

        // on s'assure que la liste ne retourne jamais de chaine null ou vide.
        result.remove("");
        result.remove(null);

        return result;
    }

    /**
     * Recupere la liste des noms des groupes parmi les tags. Un groupe dans
     * les tags commence par "@" mais son nom ne contient pas ce '@'. Il est
     * donc retire avant d'etre retourne.
     * @param tags
     * @return
     */
    public static Set<String> getGroups(Set<String> tags) {
        Set<String> result = new HashSet<String>();
        if (CollectionUtils.isNotEmpty(tags)) {
            for (String t : tags) {
                if (StringUtils.startsWith(t, BowConfig.GROUP_MARK)) {
                    result.add(StringUtils.substringAfter(t, BowConfig.GROUP_MARK));
                }
            }
        }
        return result;
    }

    public static boolean isPublicGroup(BowGroup g) {
        Set<String> reader = g.getReader();
        boolean result = org.apache.commons.collections.CollectionUtils.isNotEmpty(reader)
                && reader.contains(WikittyServiceAuthorisation.PUBLIC);
        return result;
    }

    public static String formatDate(Date date) {
        String result = DateFormatUtils.format(date, BOW_DATE_SHORT_PATTERN);
        return result;
    }

    /**
     * Generates a random token
     *
     * @return the generated token
     */
    public static String generateToken() {
        UUID uuid = UUID.randomUUID();
        String tokenGenerated = uuid.toString();
        tokenGenerated = StringUtil.encodeMD5(tokenGenerated); // encode the token in MD5
        return tokenGenerated; // return the generated token
    }

    /**
     * Redirects the user either on the home page or on the search page
     *
     * @param tagLine      search line
     * @param fullTextLine full text search
     * @return the page where the user will be redirected
     */
    public static String redirectTo(String tagLine, String fullTextLine) {
        String result = redirectTo(null, tagLine, fullTextLine);
        return result;
    }

    /**
     * Redirects the user either on the home page or on the search page
     *
     * @param listId list of bookmark id
     * @param tagLine      search line
     * @param fullTextLine full text search
     * @return the page where the user will be redirected
     */
    public static String redirectTo(List<String> listId, String tagLine, String fullTextLine) {
        String result = "home.action";

        boolean listIdNotEmpty = CollectionUtils.isNotEmpty(listId);
        boolean tagLineNotBlank = StringUtils.isNotBlank(tagLine);
        boolean fullTextLineNotBlank = StringUtils.isNotBlank(fullTextLine);

        if (listIdNotEmpty || tagLineNotBlank || fullTextLineNotBlank) {
            result += "?";
            String sep = "";
            if (listIdNotEmpty) {
                for (String id : listId) {
                    result += sep + "listId=" + id;
                    sep = "&";
                }
            }
            if (tagLineNotBlank) {
                result += sep + "tagLine=" + tagLine;
                sep = "&";
            }
            if (fullTextLineNotBlank) {
                result += sep + "fullTextLine=" + fullTextLine;
            }
        }
        return result;
    }

    /**
     * Retrieves the user by the temporary or the permanent token
     *
     * @param session the session
     * @param token   either the temporary or the permanent token
     * @return the user if the token is valid, null if the token doesn't exist
     */
    public static BowUser checkToken(BowSession session, String token) {
        BowUser result = null;
        if (checkTemporaryToken(session, token) || checkPermanentToken(session, token)) {
            result = session.getUser();
        }
        return result;
    }

    /**
     * check if token exist, if true, put User associated with it, in BowSession
     *
     * @param session bow session
     * @param token   which contains the MD5 encoding token
     * @return User the token owner, null if the token doesn't exist
     */
    protected static boolean checkPermanentToken(BowSession session, String token) {
        BowProxy proxy = session.getProxy();
        boolean result = false;
        if (token != null) {
            WikittyQuery criteria = new WikittyQueryMaker().eq(
                    BowUser.FQ_FIELD_BOWUSER_PERMANENTTOKEN, token).end();
            BowUser user = proxy.findByQuery(BowUser.class, criteria);
            if (user != null) {
                session.setUser(user);
                result = true;
            }
        }
        return result;
    }

    /**
     * Check token with temporary token
     *
     * @param session bow session
     * @param token   which contains the MD5 encoding token
     * @return user the token owner, null if the token doesn't exist
     */
    protected static boolean checkTemporaryToken(BowSession session, String token) {
        String temporaryToken = session.getTemporaryToken();
        boolean result = temporaryToken != null && temporaryToken.equals(token);
        return result;
    }
}
