/*
 * #%L
 * BOW UI
 * %%
 * Copyright (C) 2010 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.nuiton.wikitty.entities.WikittyGroup;
import org.nuiton.wikitty.entities.WikittyUser;
import org.nuiton.wikitty.query.WikittyQuery;
import org.nuiton.wikitty.query.WikittyQueryMaker;
import org.nuiton.wikitty.query.WikittyQueryResult;
import org.nuiton.wikitty.services.WikittySecurityUtil;

/**
 * Classe utilisee pour stocker les objets utils en session utilisateur
 *
 * @author poussin
 * @version $Revision$
 *          <p/>
 *          Last update: $Date$
 *          by : $Author$
 */
public class BowSession implements Serializable {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(BowSession.class);

    static final private String BOW_SESSION_KEY = BowSession.class.getSimpleName();

    private static final long serialVersionUID = 1L;

    protected BowProxy proxy;

    // user herite de preference, on ne stocke que user
    protected BowUser user;

    protected String temporaryToken;

    protected boolean admin;

    public BowSession() {
        proxy = BowProxy.getInstance(null);
    }

    public static void invalidate(Map<String, Object> session) {
        session.remove(BOW_SESSION_KEY);
        // it's not necessary to logout user on proxy. Wikitty reuse old token
        // if necessary, no database leak with many token
        // not necessary, because when user use scriptlet add, auto-login is done
        // with no logout
    }

    public static BowSession getBowSession(HttpServletRequest request) {
        HttpSession session = request.getSession();
        BowSession result = getBowSession(session);
        return result;
    }

    public static BowSession getBowSession(HttpSession httpSession) {
        BowSession result = (BowSession) httpSession.getAttribute(BOW_SESSION_KEY);
        if (result == null) {
            result = new BowSession();
            httpSession.setAttribute(BOW_SESSION_KEY, result);
        }
        return result;
    }

    public static BowSession getBowSession(Map<String, Object> session) {
        BowSession result = (BowSession) session.get(BOW_SESSION_KEY);
        if (result == null) {
            result = new BowSession();
            session.put(BOW_SESSION_KEY, result);
        }
        return result;
    }

    public BowProxy getProxy() {
        return proxy;
    }

    public BowUser getUser() {
        return user;
    }

    public List<BowSearchPrefix> getSearchPrefix() {
        BowUser user = getUser();
        
        WikittyQuery criteria = new WikittyQueryMaker().and()
                .exteq(BowSearchPrefix.EXT_BOWSEARCHPREFIX)
                .eq(BowSearchPrefix.FQ_FIELD_BOWSEARCHPREFIX_BOWUSER, user.getWikittyId()).end()
                .setSortAscending(BowSearchPrefix.ELEMENT_FIELD_BOWSEARCHPREFIX_PREFIX);
        WikittyQueryResult<BowSearchPrefix> result = getProxy().findAllByQuery(BowSearchPrefix.class, criteria);
        List<BowSearchPrefix> prefix = new ArrayList<BowSearchPrefix>(result.getAll());
        log.debug("####################### prefix number ################### " + prefix.size());
        if (prefix.isEmpty()) {
            // no prefix found, create default prefix for this request
            String userId = user.getWikittyId();
            String prefixSeparator = BowConfig.getPrefixSeparator();

            BowSearchPrefixImpl pWeb = new BowSearchPrefixImpl();
            pWeb.setBowUser(userId);
            pWeb.setPrefix(BowConfig.getWebSearchPrefix() + prefixSeparator);
            pWeb.setSearch(BowConfig.getSearchEngine());
            // TODO poussin 20130916 add default value in config for suggestion

            BowSearchPrefixImpl pTag = new BowSearchPrefixImpl();
            pTag.setBowUser(userId);
            pTag.setPrefix(BowConfig.getTagSearchPrefix() + prefixSeparator);
            pTag.setSearch("search.tag");
            pTag.setSuggestion("suggestion.tag");

            BowSearchPrefixImpl pFullText = new BowSearchPrefixImpl();
            pFullText.setBowUser(userId);
            pFullText.setPrefix(BowConfig.getFullTextSearchPrefix() + prefixSeparator);
            pFullText.setSearch("search.fulltext");
            pFullText.setSuggestion("suggestion.fulltext");

            BowSearchPrefixImpl pAlias = new BowSearchPrefixImpl();
            pAlias.setBowUser(userId);
            pAlias.setPrefix(BowConfig.getAliasPrefix() + prefixSeparator);
            pAlias.setSearch("search.alias");
            pAlias.setSuggestion("suggestion.alias");

            BowSearchPrefixImpl pAdd = new BowSearchPrefixImpl();
            pAdd.setBowUser(userId);
            pAdd.setPrefix(BowConfig.getAddPrefix() + prefixSeparator);
            pAdd.setSearch("search.add");
            pAdd.setSuggestion("suggestion.add");

            BowSearchPrefixImpl pDefault = new BowSearchPrefixImpl();
            pDefault.getWikitty().replaceWith(pWeb.getWikitty(), true);
            pDefault.setPrefix("");

            BowSearchPrefixImpl pDefaultPrefix = new BowSearchPrefixImpl();
            pDefaultPrefix.getWikitty().replaceWith(pTag.getWikitty(), true);
            pDefaultPrefix.setPrefix(prefixSeparator);

            proxy.store(pWeb, pTag, pFullText, pAlias, pAdd, pDefault, pDefaultPrefix);

            prefix.add(pWeb);
            prefix.add(pTag);
            prefix.add(pFullText);
            prefix.add(pAlias);
            prefix.add(pDefault);
            prefix.add(pDefaultPrefix);

        }
        return prefix;
    }

    public void setUser(BowUser user) {
        // check if this user is admin
        String login = user.getLogin();
        String[] admins = BowConfig.getAdmins();
        boolean isAdmin = ArrayUtils.contains(admins, login);
        setAdmin(isAdmin);

        checkPreference(user);

        // si c'est le meme user, on ne regenere pas le temporaryToken
        if (!ObjectUtils.equals(this.user, user)) {
            // generate temporary token
            String temporaryToken = BowUtils.generateToken();
            setTemporaryToken(temporaryToken);

            String wikittyToken = proxy.getSecurityToken(user.getWikittyId());
            proxy.setSecurityToken(wikittyToken);

            if (isAdmin) {
                checkAppAdminGroup(user, admins);
            }
        }
        this.user = user;
    }

    /**
     * Verifie que le group AppAdminGroup exist et est a jour
     * @param user
     * @param adminsLogin other admin identified by login
     */
    protected void checkAppAdminGroup(BowUser user, String ... adminsLogin) {
        boolean mustSave = false;
        WikittyGroup adminGroup = WikittySecurityUtil.getAppAdminGroup(proxy);
        if (adminGroup == null) {
            adminGroup = WikittySecurityUtil.createAppAdminGroup(user);
            mustSave = true;
        }
        if (ArrayUtils.isNotEmpty(adminsLogin)) {
            List<String> admins = Arrays.asList(adminsLogin);
            WikittyQuery criteria = new WikittyQueryMaker()
                    .containsOne(WikittyUser.ELEMENT_FIELD_WIKITTYUSER_LOGIN, admins).end();
            List<String> adminsId = proxy.findAllByQuery(criteria).getAll();
            if (adminGroup.getMembers().size() != adminsId.size() || !adminGroup.getMembers().containsAll(adminsId)) {
                adminGroup.setMembers(new HashSet<String>(adminsId));
                mustSave = true;
            }
        }

        if (mustSave) {
            proxy.store(adminGroup);
        }
    }


    protected void checkPreference(BowPreference preference) {
        if (preference.getBookmarks() <= 0) {
            preference.setBookmarks(100); // set default to 100
        }
        if (preference.getTags() <= 0) {
            preference.setTags(100); // set default to 100
        }
        if (preference.getColors() == null) {
            preference.setColors("");
        }
//        if (preference.getSearchEngineUrlResults() == null) {
//            preference.setSearchEngineUrlResults(BowConfig.getSearchEngine());
//        }
//        if (preference.getSearchEngineUrlSuggestions() == null) {
//            preference.setSearchEngineUrlSuggestions(""); // TODO add default value in config
//        }
//        if (StringUtils.isBlank(preference.getPrefixSeparator())) {
//            preference.setPrefixSeparator(BowConfig.getPrefixSeparator());
//        }
//        if (StringUtils.isBlank(preference.getTagSearchPrefix())) {
//            preference.setTagSearchPrefix(BowConfig.getTagSearchPrefix());
//        }
//        if (StringUtils.isBlank(preference.getFullTextSearchPrefix())) {
//            preference.setFullTextSearchPrefix(BowConfig.getFullTextSearchPrefix());
//        }
//        if (StringUtils.isBlank(preference.getWebSearchPrefix())) {
//            preference.setWebSearchPrefix(BowConfig.getWebSearchPrefix());
//        }
//        if (StringUtils.isBlank(preference.getAliasPrefix())) {
//            preference.setAliasPrefix(BowConfig.getAliasPrefix());
//        }
//        if (StringUtils.isBlank(preference.getDefaultPrefix())) {
//            preference.setDefaultPrefix(BowConfig.getDefaultPrefix());
//        }
//        if (StringUtils.isBlank(preference.getDefaultAction())) {
//            preference.setDefaultAction(BowConfig.getDefaultAction());
//        }
    }

    public String getPermanentToken() {
        return getUser().getPermanentToken();
    }

    public String getTemporaryToken() {
        return temporaryToken;
    }

    public void setTemporaryToken(String temporaryToken) {
        this.temporaryToken = temporaryToken;
    }

    public boolean isAdmin() {
        return admin;
    }

    public void setAdmin(boolean admin) {
        this.admin = admin;
    }

}
