/*
 * #%L
 * BOW UI
 * %%
 * Copyright (C) 2010 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow.action.preference;

import org.apache.commons.lang3.StringUtils;
import org.chorem.bow.BookmarkUtils;
import org.chorem.bow.BowBookmark;
import org.chorem.bow.BowImportHelper;
import org.chorem.bow.BowUser;
import org.htmlparser.Node;
import org.htmlparser.Parser;
import org.htmlparser.util.NodeList;
import org.htmlparser.util.ParserException;
import org.htmlparser.util.SimpleNodeIterator;
import org.nuiton.wikitty.entities.Wikitty;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.bow.BowProxy;

/**
 * Import un bookmark HTML
 *
 * @author poussin
 */
public class ImportBookmarksAction extends PreferenceBaseAction {

    private static final long serialVersionUID = -5962680416570797028L;

    private static Log log = LogFactory.getLog(ImportBookmarksAction.class);

    protected File upfile;

    protected String upfileContentType;

    protected String upfileFileName;

    /** @return the upfile */
    public File getUpfile() {
        return upfile;
    }

    /** @param upfile the upfile to set */
    public void setUpfile(File upfile) {
        this.upfile = upfile;
    }

    /** @return the upfileContentType */
    public String getUpfileContentType() {
        return upfileContentType;
    }

    /** @param upfileContentType the upfileContentType to set */
    public void setUpfileContentType(String upfileContentType) {
        this.upfileContentType = upfileContentType;
    }

    /** @return the upfileFileName */
    public String getUpfileFileName() {
        return upfileFileName;
    }

    /** @param upfileFileName the upfileFileName to set */
    public void setUpfileFileName(String upfileFileName) {
        this.upfileFileName = upfileFileName;
    }

    /**
     * @param html String html
     * @param name String name
     * @param user User user
     * @return Bookmark the bookmark filled
     */
    protected BowBookmark createBookmarkFromHtml(String html, String name, BowUser user) {
        String link = getUrlFromHtml(html);
        String tags = getTagsFromHtml(html);
        Date date = getDateFromHtml(html);
        BowBookmark bookmark = BookmarkUtils.createBookmark(link,
                                                            name,
                                                            tags,
                                                            user,
                                                            null,
                                                            null,
                                                            date);
        return bookmark;
    }

    /**
     * @param html String html
     * @return Date     from the html
     */
    protected Date getDateFromHtml(String html) {
        Date date = null;
        if (html != null) {
            Pattern p = Pattern.compile("ADD_DATE=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                String str = m.group(1);
                if (str != null && !str.isEmpty()) {
                    long time = Long.valueOf(str) * 1000; // the date in ms
                    date = new Date(time);
                }
            }
        }
        return date;
    }

    /**
     * @param html String html
     * @return url
     */
    protected String getUrlFromHtml(String html) {
        String url = "";
        if (html != null) {
            Pattern p = Pattern.compile("A HREF=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                url = m.group(1);
            }
        }
        return url;
    }

    protected String getTagsFromHtml(String html) {
        String tags = "";
        if (html != null) {
            Pattern p = Pattern.compile("SHORTCUTURL=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                tags = m.group(1);
            }
        }
        return tags;
    }

    protected void parseHtmlToBookmarks(NodeList list,
                                        BowUser user,
                                        Date date,
                                        List<BowBookmark> bookmarks,
                                        Set<String> tagList)
            throws ParserException {

        SimpleNodeIterator it = list.elements();

        while (it.hasMoreNodes()) {
            Node node = it.nextNode();
            String plainText = node.toPlainTextString(); // The text between
            // two heads ==>
            // <toto>plainText</toto>
            String text = node.getText(); // The text in the head ==>
            // <text></toto>

            if (StringUtils.startsWithIgnoreCase(text, "h3")) { // H3 = folder
                if (StringUtils.isNotBlank(plainText)) {
                    tagList.add(plainText); // Adds the folder name to the
                    // tagList
                }
            } else if (StringUtils.startsWithIgnoreCase(text, "a href")) { // HREF
                // = new
                // bookmarks
                BowBookmark bookmark = createBookmarkFromHtml(text, plainText, user);
                Wikitty w = getBowProxy().castTo(Wikitty.class, bookmark);
                BowImportHelper.addExtension(w);
                BowImportHelper.setImportDate(w, date);

                bookmark.addAllLabels(tagList);

                bookmarks.add(bookmark);
            }
            NodeList children = node.getChildren();

            if (children != null) {
                parseHtmlToBookmarks(children, user, date, bookmarks,
                                     // On utilise une nouvelle liste pour les enfants
                                     // pour eviter que les tags de tous les enfants se
                                     // retrouve ensemble
                                     new HashSet<String>(tagList)); // If there is an under node = recursion
            }
        }
    }

    @Override
    public String execute() {
        String result = SUCCESS;
        log.info("Importing boormarks : " + upfile);
        try {
            BowUser user = getBowSession().getUser();

            // Struts limite to html files, is null if not one
            if (upfile != null) {
                BowProxy proxy = getBowProxy();

                try {
                    URL url = upfile.toURI().toURL();
                    Parser parser = new Parser(url.openConnection());
                    NodeList list = parser.parse(null);

                    List<BowBookmark> bookmarks = new ArrayList<BowBookmark>();
                    parseHtmlToBookmarks(list, user, new Date(), bookmarks,
                                         new HashSet<String>());
                    proxy.store(bookmarks);

                    addActionMessage(_("bow.bookmark.import.successful"));
                } catch (ParserException eee) {
                    result = ERROR;
                    log.error(String.format("Can't parse imported file '%s'",
                                            getUpfileFileName()), eee);
                    addActionError(_("bow.bookmark.badFileFormat"));
                } catch (IOException eee) {
                    result = ERROR;
                    log.error(String.format("Can't read imported file '%s'",
                                            getUpfileFileName()), eee);
                    addActionError(_("bow.error.internal"));
                }
            } else {
                addActionError(_("bow.bookmark.badFileFormat"));
                result = ERROR;
            }
        } catch (Exception eee) {
            result = ERROR;
            addActionError(_("bow.error.internal"));
            log.error(eee.getMessage(), eee);
        } finally {
            // on recharge les data apres l'action pour l'affichage
            load();
        }
        return result;
    }
}
