/*
 * #%L
 * BOW UI
 * %%
 * Copyright (C) 2010 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow.action.login;

import org.apache.commons.lang3.StringUtils;
import org.chorem.bow.BowMail;
import org.chorem.bow.BowUser;
import org.chorem.bow.BowUserImpl;
import org.chorem.bow.BowUtils;
import org.chorem.bow.action.BowBaseAction;
import org.nuiton.util.StringUtil;

import javax.mail.MessagingException;
import javax.mail.internet.AddressException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.bow.BowProxy;
import org.nuiton.wikitty.query.WikittyQuery;
import org.nuiton.wikitty.query.WikittyQueryMaker;

/**
 * Enregistre un nouvel utilisateur
 *
 * @author poussin
 */
public class RegisterAction extends BowBaseAction {

    private static final long serialVersionUID = 2204772861770399542L;

    private static Log log = LogFactory.getLog(RegisterAction.class);

    protected String email;

    protected String password;

    protected String repeatPassword;

    /** @return the email */
    public String getEmail() {
        return email;
    }

    /** @param email the email to set */
    public void setEmail(String email) {
        this.email = email;
    }

    /** @return the password */
    public String getPassword() {
        return password;
    }

    /** @param password the password to set */
    public void setPassword(String password) {
        this.password = password;
    }

    /** @return the repeatPassword */
    public String getRepeatPassword() {
        return repeatPassword;
    }

    /** @param repeatPassword the repeatPassword to set */
    public void setRepeatPassword(String repeatPassword) {
        this.repeatPassword = repeatPassword;
    }

    /**
     * @param email non null and non blank email
     * @return false if the email address isn't already registered, true
     *         otherwise
     */
    protected boolean alreadyRegistered(String email) {
        boolean result;

        BowProxy proxy = getBowProxy();
        //Retrieves user by user name (email)
        WikittyQuery criteria = new WikittyQueryMaker()
                .eq(BowUser.FQ_FIELD_WIKITTYUSER_LOGIN, email).end();

        result = proxy.findByQuery(BowUser.class, criteria) != null;
        //If the user doesn't already exist
        if (result) {
            //The email address is already used by someone
            addFieldError("email", _("bow.register.emailAldyUsed"));
        }
        return result;
    }

    /** Registers the new user and sends an email to confirm registration */
    @Override
    public String execute() {
        String result = INPUT;

        try {
            if (StringUtils.isNotBlank(email)) {
                email = email.trim();
                if (StringUtils.isNotBlank(password)) {
                    if (!password.equals(repeatPassword)) {
                        addActionError(_("bow.register.pwdDontMatch"));
                    } else {
                        String md5 = StringUtil.encodeMD5(password);

                        //If the email address isn't already used
                        if (!alreadyRegistered(email)) {
                            BowProxy proxy = getBowProxy();
                            BowUserImpl newUser = new BowUserImpl();
                            newUser.setPassword(md5);
                            newUser.setLogin(email);
                            String permanentToken = BowUtils.generateToken();
                            newUser.setPermanentToken(permanentToken);
                            BowUser login = proxy.store(newUser); //Stores the new user

                            //If there was an error when storing the user
                            if (login == null) {
                                addFieldError("email", _("bow.register.invalidLogin"));
                            } else {
                                getBowSession().setUser(login);
                                try {
                                    String mailContent;

                                    mailContent =
                                            _("bow.register.mailHi") + ",\n\n" +
                                            _("bow.register.mailPwd") + ": " + password + "\n\n" +
                                            _("bow.register.mailEmail") + ": " + email + "\n\n";
                                    BowMail.sendMail(email, _("bow.register.mailSubject"), mailContent);
                                } catch (AddressException eee) {
                                    //If the email address is invalid
                                    addActionError(_("bow.mail.badFormat"));
                                    log.error(String.format("Can't send message email to '%s'",
                                                            email), eee);
                                } catch (MessagingException eee) {
                                    addActionError("bow.mail.sendError");
                                    log.error(String.format("Can't send message email to '%s'",
                                                            email), eee);
                                }
                                result = SUCCESS;
                            }
                        }
                    }
                }
            }
        } catch (Exception eee) {
            addActionError(_("bow.error.internal"));
            log.error(eee.getMessage(), eee);
        }
        return result;
    }
}
