/*
 * #%L
 * BOW UI
 * %%
 * Copyright (C) 2010 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow;

import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.StringUtil;

import java.util.Date;
import java.util.Set;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.nuiton.wikitty.entities.Element;
import org.nuiton.wikitty.query.WikittyQuery;
import org.nuiton.wikitty.query.WikittyQueryMaker;

public class BookmarkUtils {

    /** Logger. */
    public static Log log = LogFactory.getLog(BookmarkUtils.class);

    /**
     * Ajoute a la recherche tout ce qui est necessaire pour connaitre
     * toutes les informations que l'utilisateur peut voir, meme s'il n'est
     * pas le createur de ce bookmark
     *
     * @param search
     * @param userId
     * @return
     */
    public static WikittyQueryMaker addEqUser(WikittyQueryMaker search, String userId) {
        search.or();
          search.eq(BowBookmark.FQ_FIELD_WIKITTYAUTHORISATION_OWNER, userId);
          search.eq(BowBookmark.FQ_FIELD_WIKITTYAUTHORISATION_READER, userId);
          search.eq(BowBookmark.FQ_FIELD_WIKITTYAUTHORISATION_WRITER, userId);
          search.eq(BowBookmark.FQ_FIELD_WIKITTYAUTHORISATION_ADMIN, userId);
        search.close();
        return search;
    }

    public static WikittyQuery getBookmarkListCriteriaByUser(BowUser user,
                                                         String tagLine,
                                                         String fullTextLine,
                                                         String order,
                                                         int first) {
        WikittyQueryMaker search = new WikittyQueryMaker().and();
        addEqUser(search, user.getWikittyId());

        if (StringUtils.isNotBlank(fullTextLine)) {
            search.keyword(fullTextLine);
        }

        if (StringUtils.isNotBlank(tagLine)) {
            Set<String> tags = BowUtils.getWords(tagLine);
            search.containsAll(BowBookmark.FQ_FIELD_WIKITTYLABEL_LABELS, tags);
        }

        WikittyQuery criteria = search.end();
        criteria.addFacetField(BowBookmark.ELEMENT_FIELD_WIKITTYLABEL_LABELS);
        // on fixe le nombre de tag max souhaite par l'utilisateur
        criteria.setFacetLimit(user.getTags());

        criteria.setOffset(first);

        // on fixe le nombre max d'element a ce qu'a defini l'utilisateur
        criteria.setLimit(user.getBookmarks());

        boolean sortOrderAscending = StringUtils.startsWith(order, "asc");
        Element sortOrderField = null;
        if (StringUtils.equals(order, "ascName")) {
            sortOrderField = BowBookmark.ELEMENT_FIELD_BOWBOOKMARK_DESCRIPTION;
        } else if (StringUtils.equals(order, "ascDate")) {
            sortOrderField = BowBookmark.ELEMENT_FIELD_BOWBOOKMARK_CREATIONDATE;
        } else if (StringUtils.equals(order, "ascClick")) {
            sortOrderField = BowBookmark.ELEMENT_FIELD_BOWBOOKMARK_CLICK;
        } else if (StringUtils.equals(order, "descName")) {
            sortOrderField = BowBookmark.ELEMENT_FIELD_BOWBOOKMARK_DESCRIPTION;
        } else if (StringUtils.equals(order, "descDate")) {
            sortOrderField = BowBookmark.ELEMENT_FIELD_BOWBOOKMARK_CREATIONDATE;
        } else if (StringUtils.equals(order, "descClick")) {
            sortOrderField = BowBookmark.ELEMENT_FIELD_BOWBOOKMARK_CLICK;
        }
        if (sortOrderField == null) {
            // default sort order
            // TODO poussin 20110815 rendre configurable par l'utilisateur l'ordre par defaut qu'il prefere
            sortOrderField = BowBookmark.ELEMENT_FIELD_BOWBOOKMARK_CLICK;
            sortOrderAscending = false;
        }

        if (sortOrderAscending) {
            criteria.setSortAscending(sortOrderField);
        } else {
            criteria.setSortDescending(sortOrderField);
        }

        return criteria;
    }

    /**
     * @param url         String which contains the bookmark url
     * @param nameAndTags String which contains the name and the tags of the
     *                    bookmark separated by '|' ==> name|tag1 tag2...
     * @param user        bookmark user
     * @return Bookmark     the website bookmark, null if all fields are not correctly filled
     */
    public static BowBookmark createBookmark(String url, String nameAndTags, BowUser user) {

        // on recherche les tags qui doivent etre apres le dernier |
        // ce qu'il y a devant est la description qui peut elle aussi contenir
        // des |

        // on s'assure que la chaine est non null, et en enleve les blanc en trop
        String name = BowUtils.normalizeString(nameAndTags);
        String tags = ""; // par defaut on a pas de tag

        int nameIndex = name.lastIndexOf('|');
        if (nameIndex > 0) {
            // il faut faire tags avant name, car on change name ensuite
            tags = name.substring(nameIndex + 1);
            name = name.substring(0, nameIndex); // get the website description
        }
        BowBookmark bookmark = createBookmark(url, name, tags, user, "", "", null);

        return bookmark;
    }

    /**
     * Create new Bookmark
     *
     * @param url          of bookmark
     * @param name         of bookmark
     * @param tags         of bookmark
     * @param user         creating bookmark
     * @param privateAlias of bookmark
     * @param publicAlias
     * @param date         of creation
     * @return bookMarl created
     */
    public static BowBookmark createBookmark(String url, String name, String tags,
                                             BowUser user, String privateAlias, String publicAlias, Date date) {

        BowBookmark bookmark = new BowBookmarkImpl();

        bookmark.setDescription(BowUtils.normalizeString(name));

        bookmark.addAllLabels(BowUtils.getWords(tags));
        bookmark.setLink(BowUtils.normalizeUrl(url));

        bookmark.setClick(0);

        bookmark.setOwner(user.getWikittyId());
        bookmark.addReader(user.getWikittyId()); // only owner can read it
        bookmark.setCreationDate(BowUtils.normalizeDate(date));

        bookmark.setPrivateAlias(BowUtils.normalizeString(privateAlias));
        bookmark.setPublicAlias(BowUtils.normalizeString(publicAlias));

        return bookmark;
    }

    public static void updateBookmark(BowBookmark bookmark, String name,
                                      String link, String tags, String privateAlias, String publicAlias) {
        bookmark.setDescription(BowUtils.normalizeString(name));

        bookmark.setLink(BowUtils.normalizeUrl(link));
        bookmark.setLabels(BowUtils.getWords(tags));

        bookmark.setPrivateAlias(BowUtils.normalizeString(privateAlias));
        bookmark.setPublicAlias(BowUtils.normalizeString(publicAlias));
    }

    public static String getBookmarkTagsString(BowBookmark bookmark) {
        if (bookmark != null) {
            Set<String> tags = bookmark.getLabels();
            if (tags != null) {
                return StringUtil.join(tags, " ", true);
            }
        }
        return "";
    }

    /**
     * retourne l'image du site passer en parametre
     * @param url
     * @return
     */
    public static byte[] getScreenshot(String url) {
        byte[] result = null;
        try {
            String ssu = BowConfig.getScreenshotURL();
            if (StringUtils.isNotBlank(ssu)) {
                url = URLEncoder.encode(url, "UTF-8");
                String imageUrl = String.format(ssu, url);
                result = getImage(imageUrl);
            }
        } catch (Exception eee) {
            log.info("Can't encode URL: " + url, eee);
        }
        return result;
    }

    /**
     * retourne le favicon du site passe en parametre
     * @param urlString
     * @return
     */
    public static byte[] getFavicon(String urlString) {
        byte[] result = null;
        try {
            String favicon = null;

            // on commence par rechercher le favicon specifie dans le head
            Document doc = Jsoup.connect(urlString).get();
            org.jsoup.nodes.Element link = doc.head().select("link[rel~=(shortcut icon|icon)]").first();
            if (link != null) {
                favicon = link.attr("abs:href");
            }

            // si rien n'est specifie dans le head, on recherche un /favicon.ico
            if (favicon == null) {
                URL url = new URI(urlString).toURL();
                URL favURL = new URL(url.getProtocol(), url.getHost(), url.getPort(), "/favicon.ico");
                favicon = favURL.toExternalForm();
            }
            result = getImage(favicon);
            return result;
        } catch (Exception eee) {
            log.info("Can't parse URL: " + urlString, eee);
        }
        return result;
    }

    protected static byte[] getImage(String url) {
        byte[] result = null;
        try {
            log.debug("try to connect get image : " + url);

            CloseableHttpClient httpclient = HttpClients.createDefault();
            HttpGet httpGet = new HttpGet(url);
            CloseableHttpResponse response = httpclient.execute(httpGet);

            try {
                HttpEntity entity = response.getEntity();
                result = IOUtils.toByteArray(entity.getContent());
                // ensure it is fully consumed
                EntityUtils.consume(entity);
            } finally {
                response.close();
            }
        } catch (Exception eee) {
            log.info("Can't get result for url: " + url, eee);
        }
        return result;
    }
}
