/*
 * #%L
 * bow
 * 
 * $Id: OpenSearchBaseAction.java 324 2011-08-29 12:21:36Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/bow/tags/bow-1.1/bow-ui/src/main/java/org/chorem/bow/action/opensearch/OpenSearchBaseAction.java $
 * %%
 * Copyright (C) 2010 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow.action.opensearch;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.bow.BowUser;
import org.chorem.bow.action.BowBaseAction;

/**
 * Traite toutes les demandes faite via l'opensearch
 * a:, :, t:, ...
 * <p/>
 * Cette classe recherche le prefix et redirige vers la bonne methode execute.
 * Les classes filles n'ont plus qu'a implanter les methodes
 *
 * @author poussin
 */
abstract public class OpenSearchBaseAction extends BowBaseAction {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(OpenSearchBaseAction.class);

    private static final long serialVersionUID = 1L;

    /** [in] le token d'identification */
    protected String token;

    /** [in] la ligne de recherche soumise par l'utilisateur */
    protected String searchLine;

    /**
     * [out] le prefix trouve ou calculer par rapport a searchLine
     * avoir acces a cette valeur, peut-etre utile dans certaine implantation
     * de {@link #executeNotSupportedPrefixAction}
     */
    protected String prefix;

    public String getSearchLine() {
        return searchLine;
    }

    public void setSearchLine(String searchLine) {
        this.searchLine = searchLine;
    }

    /** @return the token */
    public String getToken() {
        return token;
    }

    /** @param token the token to set */
    public void setToken(String token) {
        this.token = token;
    }

    public String getPrefix() {
        return prefix;
    }

    abstract protected String executeTagSearchAction(String query);

    abstract protected String executeFulltextSearchAction(String query);

    abstract protected String executeWebSearchAction(String query);

    abstract protected String executeAliasSearchAction(String privateAlias);

    abstract protected String executeNotSupportedPrefixAction(String query);

    @Override
    public String execute() {
        BowUser user = getBowSession().getUser();

        String prefixSeparator = user.getPrefixSeparator();
        prefix = StringUtils.substringBefore(searchLine, prefixSeparator);
        String query = StringUtils.substringAfter(searchLine, prefixSeparator);

        if (log.isDebugEnabled()) {
            log.debug(String.format("Cut '%s' with '%s' result is '%s' and '%s'",
                                    searchLine, prefixSeparator, prefix, query));
        }

        if (StringUtils.isEmpty(prefix)) {
            // on a le separateur mais pas de prefix
            prefix = user.getDefaultPrefix();
        } else if (StringUtils.equals(searchLine, prefix)) {
            // on a pas de separateur
            prefix = user.getDefaultAction();
            query = searchLine;
        }

        if (log.isDebugEnabled()) {
            log.debug(String.format("Prefix is '%s' and query '%s'",
                                    prefix, query));
        }

        String result;
        if (StringUtils.equals(prefix, user.getTagSearchPrefix())) {
            result = executeTagSearchAction(query);
        } else if (StringUtils.equals(prefix, user.getFullTextSearchPrefix())) {
            result = executeFulltextSearchAction(query);
        } else if (StringUtils.equals(prefix, user.getWebSearchPrefix())) {
            result = executeWebSearchAction(query);
        } else if (StringUtils.equals(prefix, user.getAliasPrefix())) {
            result = executeAliasSearchAction(query);
        } else {
            result = executeNotSupportedPrefixAction(query);
        }

        return result;
    }
}
