/*
 * #%L
 * bow
 * 
 * $Id: BookmarkUtils.java 263 2011-04-26 13:17:07Z sletellier $
 * $HeadURL: http://svn.chorem.org/svn/bow/tags/bow-0.5/bow-ui/src/main/java/org/chorem/bow/BookmarkUtils.java $
 * %%
 * Copyright (C) 2010 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow;

import java.text.ParseException;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.StringUtil;

import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.FacetTopic;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.search.operators.Element;
import org.nuiton.wikitty.search.Search;

public class BookmarkUtils {

    /**
     * Logger.
     */
    public static Log log = LogFactory.getLog(BookmarkUtils.class);

    static public Criteria getBookmarkListCriteriaByUser(BowPreference user, String searchLine) {
        Criteria criteria = null;
        if (user != null) {
            if (searchLine != null && !searchLine.isEmpty()) {
                String[] words = searchLine.split("\\s+"); //Puts the tags in an array
                List<String> tags = new ArrayList<String>(Arrays.asList(words));
                criteria = Search.query()
                        .eq(BowBookmark.FQ_FIELD_BOWBOOKMARK_BOWUSER, user.getWikittyId())
                        .eq(BowBookmark.FQ_FIELD_WIKITTYLABEL_LABELS, tags).criteria()
                        .addFacetField(BowBookmark.FQ_FIELD_WIKITTYLABEL_LABELS);
            } else {
                criteria = Search.query().eq(BowBookmark.FQ_FIELD_BOWBOOKMARK_BOWUSER, user.getWikittyId()).
                        criteria().addFacetField(BowBookmark.FQ_FIELD_WIKITTYLABEL_LABELS);
            }
            criteria.setEndIndex(user.getBookmarks());
        }
        return criteria;
    }

    static public BookmarkActions createBookmarkActions(HttpServletRequest request,
            PagedResult<BowBookmark> result, String searchLine) {
        String fullText = request.getParameter("fullTextLine");
        BookmarkActions bookmarkActions = new BookmarkActions();
        bookmarkActions.setFullTextLine(fullText);
        List<BowBookmark> bookList = result.getAll();

        if (bookList != null) {
            bookmarkActions.setBookmarks(bookList);
        }
        if (fullText == null || fullText.isEmpty()) {
            if (searchLine != null && searchLine.isEmpty()) {
                bookmarkActions.emptySearchline();
            } else {
                bookmarkActions.addTags(searchLine); //Adds the new tags
            }
        }
        List<FacetTopic> topics = result.getTopic(BowBookmark.FQ_FIELD_WIKITTYLABEL_LABELS);
        bookmarkActions.createTagCloud(topics);
        return bookmarkActions;
    }

    static public Map<String, Integer> getBookmarksByImportDate(HttpServletRequest request, BowUser user) throws ParseException {
        BowSession session = BowSession.getBowSession(request);
        WikittyProxy proxy = session.getProxy();
        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, BowImport.EXT_BOWIMPORT)
                .eq(BowBookmark.FQ_FIELD_BOWBOOKMARK_BOWUSER, user.getWikittyId()).criteria()
                .addFacetField(BowImport.FQ_FIELD_BOWIMPORT_IMPORTDATE)
                .setFirstIndex(0).setEndIndex(0);

        List<FacetTopic> bookmarksImportDate =
                proxy.findAllByCriteria(BowImport.class, criteria)
                .getTopic(BowImport.FQ_FIELD_BOWIMPORT_IMPORTDATE);

        // Extract imports
        Map<String, Integer> result = new HashMap<String, Integer>();
        for (FacetTopic facet : bookmarksImportDate) {
            String dateAsString = facet.getTopicName();

            result.put(dateAsString, facet.getCount());
        }

        return result;
    }

    /**
     * @param html      String html
     * @param name      String name
     * @param user      User user
     * @return Bookmark the bookmark filled
     */
    public static BowBookmark createBookmarkFromHtml(String html, String name, BowUser user) {
        String link = getUrlFromHtml(html);
        String tags = getTagsFromHtml(html);
        Date date = getDateFromHtml(html);
        BowBookmark bookmark = createBookmark(link, name, tags, user, null, null, date);
        return bookmark;
    }

    /**
     * @param html      String html
     * @return Date     from the html
     */
    public static Date getDateFromHtml(String html) {
        Date date = null;
        if (html != null) {
            Pattern p = Pattern.compile("ADD_DATE=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                String str = m.group(1);
                if (str != null && !str.isEmpty()) {
                    long time = Long.valueOf(str) * 1000; // the date in ms
                    date = new Date(time);
                }
            }
        }
        return date;
    }

    /**
     * @param bookmarks List<Bookmarks> bookmarks
     * @return String   the html file
     */
    public static String getExportHtmlBookmark(List<BowBookmark> bookmarks) {
        String export;
        export = "<!DOCTYPE NETSCAPE-Bookmark-file-1>\n"
                + "<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=UTF-8\">\n"
                + "<DL><p>\n";
        if (bookmarks != null) {
            for (BowBookmark bookmark : bookmarks) {
                if (bookmark != null) {
                    export += "<DT><A HREF=\"";
                    export += bookmark.getLink() + "\" ";
                    Date date = bookmark.getCreationDate();
                    if (date != null) {
                        long milli = date.getTime();
                        String time = String.valueOf(milli / 1000);
                        export += "ADD_DATE=\"" + time + "\" LAST_MODIFIED=\"" + time + "\" ";
                    }
                    export += "LAST_CHARSET=\"UTF-8\" ";
                    Set<String> tags = bookmark.getLabels();
                    if (tags != null && !tags.isEmpty()) {
                        export += "SHORTCUTURL=\"" + getBookmarkTagsString(bookmark) + "\"";
                    }
                    export += ">" + bookmark.getDescription() + "</A>\n";
                }
            }
        }
        export += "</DL><p>";
        return export;
    }

    /**
     * @param html   String html
     * @return url
     */
    public static String getUrlFromHtml(String html) {
        String url = "";
        if (html != null) {
            Pattern p = Pattern.compile("A HREF=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                url = m.group(1);
            }
        }
        return url;
    }

    public static String getTagsFromHtml(String html) {
        String tags = "";
        if (html != null) {
            Pattern p = Pattern.compile("SHORTCUTURL=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                tags = m.group(1);
            }
        }
        return tags;
    }

    /**
     * @param url           String which contains the bookmark url
     * @param nameAndTags   String which contains the name and the tags of the
     *                      bookmark separated by '|' ==> name|tag1 tag2...
     * @param user          bookmark user
     * @return Bookmark     the website bookmark, null if all fields are not correctly filled
     */
    public static BowBookmark createBookmark(String url, String nameAndTags, BowUser user) {

        BowBookmark bookmark = new BowBookmarkImpl();

        int nameIndex = nameAndTags.indexOf('|'); // get the index name of the website
        if (nameIndex > 0) {
            String name = nameAndTags.substring(0, nameIndex); // get the website name
            if (!name.isEmpty()) {
                name = name.trim();
                bookmark.setDescription(name); // set the description (website name)
            }
        }
        String tags = nameAndTags;
        if (nameIndex >= 0) {
            tags = nameAndTags.substring(nameIndex + 1); // get tags, +1 because of '|'
        }
        addTagsToBookmark(tags, bookmark);
        addUrlToBookmark(url, bookmark);
        bookmark.setClick(0);
        bookmark.setBowUser(user.getWikittyId()); // set the email (user name)
        bookmark.setCreationDate(new Date()); // set the date
        return bookmark;
    }

    public static void addUrlToBookmark(String url, BowBookmark bookmark) {
        if (url != null && !url.isEmpty()) {
            String link = url.trim();
            if (! link.matches("[a-z]+://.+") ) {
                link = "http://" + link;
            }
            bookmark.setLink(link);
        } else {
            bookmark.setLink("");
        }
    }

    public static void addTagsToBookmark(String tags, BowBookmark bookmark) {
        if (tags != null && !tags.isEmpty()) {
            String[] tagsTab = tags.split("\\s+"); // put the tags in an array
            List<String> tagList = Arrays.asList(tagsTab);
            addTagsToBookmark(tagList, bookmark);
        }
    }

    public static void addTagsToBookmark(List<String> tagList, BowBookmark bookmark) {
        if (tagList != null) {
            for (String tag : tagList) {
                if (tag != null && !tag.isEmpty()) {
                    // on peut ajouter plusieurs fois le meme tag, car on les
                    // stocke dans un Set
                    bookmark.addLabels(tag);
                }
            }
        }
    }

    /**
     * Create new Bookmark
     *
     * @param url of bookmark
     * @param name of bookmark
     * @param tags of bookmark
     * @param user creating bookmark
     * @param privateAlias of bookmark
     * @param date of creation
     * @return bookMarl created
     */
    public static BowBookmark createBookmark(String url, String name, String tags,
            BowUser user, String privateAlias, String publicAlias, Date date) {

        BowBookmark bookmark = new BowBookmarkImpl();
        if (name != null) {
            name = name.trim();
            bookmark.setDescription(name);
        } else {
            bookmark.setDescription("");
        }

        addTagsToBookmark(tags, bookmark);

        addUrlToBookmark(url, bookmark);

        bookmark.setClick(0);

        bookmark.setBowUser(user.getWikittyId()); // set the email (user name)
        if (date == null) {
            date = new Date();
        }
        bookmark.setCreationDate(date); // set the date
        if (privateAlias != null) {
            privateAlias = privateAlias.trim();
            bookmark.setPrivateAlias(privateAlias);
        } else {
            bookmark.setPrivateAlias("");
        }
        if (publicAlias != null) {
            publicAlias = publicAlias.trim();
            bookmark.setPublicAlias(publicAlias);
        } else {
            bookmark.setPublicAlias("");
        }
        return bookmark;
    }

    public static void updateBookmarkName(BowBookmark bookmark, String name) {
        if (name != null) {
            name = name.trim();
            bookmark.setDescription(name);
        }
    }

    public static void updateBookmark(BowBookmark bookmark, String name,
            String link, String tags, String privateAlias, String publicAlias) {
        if (name != null) {
            name = name.trim();
            bookmark.setDescription(name);
        }
        addUrlToBookmark(link, bookmark);
        bookmark.clearLabels();
        addTagsToBookmark(tags, bookmark);
        if (privateAlias != null) {
            privateAlias = privateAlias.trim();
        }
        bookmark.setPrivateAlias(privateAlias);
        if (publicAlias != null) {
            publicAlias = publicAlias.trim();
        }
        bookmark.setPublicAlias(publicAlias);
    }

    public static String getBookmarkTagsString(BowBookmark bookmark) {
        if (bookmark != null) {
            Set<String> tags = bookmark.getLabels();
            if (tags != null) {
                return StringUtil.join(tags, " ", true);
            }
        }
        return "";
    }

}
