/*
 * Copyright 2007 Robert Hanson <iamroberthanson AT gmail.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.gwtwidgets.client.ui.canvas.impl;

import org.gwtwidgets.client.ui.canvas.Canvas;

import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Element;

/**
 * <p>
 * Implementation of canvas for Internet Explorer 6.0+ based on VML.
 * In order for this implementation to function, VML rendering must be enabled
 * on the HTML page:
 * <pre>
 * &lt;html xmlns="http://www.w3.org/1999/xhtml" xmlns:v="urn:schemas-microsoft-com:vml"&gt;
 * &lt;head&gt;
 * 	&lt;style&gt; v\:* { behavior: url(#default#VML); }&lt;/style &gt;
 * &lt;/head&gt;
 * ...
 * &lt;/html&gt;
 * </pre>
 * </p>
 * <p>
 * In contrast to {@link IECanvasImpl}, this implementation avoids creation of DOM nodes in
 * favor of string manipulation and also assumes that IE treats multiple <code>style</code>
 * attributes on the same element as a single, concatenated attribute value.
 * </p>
 * @author George Georgovassilis g.georgovassilis[at]gmail.com
 *
 */
public class IECanvasStringImpl extends Canvas{

	private final static String[] polyLineAtts = {"points", "filled"};
	private final static String[] drawArcAtts = {"startangle","endangle"};
	private final static String[] noStrokeAtts ={"stroked","filled","strokeweight"};
	private final static String[] noStokeVals = {"false","false","0"};

	private final StringArray sb = new StringArray();
	

	private int width;
	private int height;
	private double offsetLeft;
	private double offsetTop;
	private String rotation = "";
	private double strokeAlpha = 1;
	private double strokeWeight = 1;
	private String sStroke;
	private String sFill;
	private String strokeColour=getColor(0, 0, 0);
	private String fillColour=getColor(255, 255, 255);
	private double fillAlpha = 1;

	private StringArray buffer = new StringArray();
	private org.gwtwidgets.client.ui.canvas.Font font;
	private Element eGroup;
	
	private String getFill(){
		return sFill==null?sFill = "<v:fill color='"+fillColour+"' opacity='"+fillAlpha+"' />":sFill;
	}
	
	private String getStroke(){
		return sStroke==null?sStroke = "<v:stroke color='"+strokeColour+"' opacity='"+strokeAlpha+"' weight='"+strokeWeight+"'/>":sStroke;
	}

	private final static String getColor(int red, int green, int blue){
		return "rgb("+red+","+green+","+blue+")";
	}

	private void applyClipping(Element e){
		DOM.setStyleAttribute(e, "clip", "rect(0 "+width+" "+height+" 0)");
		DOM.setStyleAttribute(e, "overflow", "hidden");
	}
	
	private void closeDecl(){
		buffer.append(">\n");
	}

	private void addElementPlain(String sElem, double left, double top, double width, double height, String[] attributes, String[] values) {
		buffer.appendElement(sElem,left,top,width,height);
		for (int i=0;i<attributes.length;i++){
			buffer.appendAttribute(attributes[i], values[i]);
		}
	}

	private void addRotation(){
		buffer.append("style='rotation:").append(rotation).append("' ");
	}
	
	private void addFill(){
		buffer.append(getFill());
	}

	private void addStroke(){
		buffer.append(getStroke());
	}

	private void addElement(String sElem, double left, double top, double width, double height, String[] attributes, String[] values) {
		addElementPlain(sElem, left, top, width, height, attributes, values);
		addRotation();
		closeDecl();
		addFill();
		addStroke();
	}


	private void setSize(int width, int height){
		this.width = width;
		this.height = height;
		super.setSize(width+"px", height+"px");
		Element e = getElement();
		DOM.setElementAttribute(e, "width", width+"px");
		DOM.setElementAttribute(e, "height", height+"px");
	}
	
	private void translate(double[] x, double[] y, double angle, double centerX, double centerY){
		double cosAngle = Math.cos(angle);
		double sinAngle = Math.sin(angle);
		for (int i=0;i<x.length;i++){
			double dx = x[i] - centerX;
			double dy = y[i] - centerY;
			x[i] = centerX + dx * cosAngle - dy * sinAngle;
			y[i] = centerY + dy * cosAngle + dx * sinAngle;
		}
	}

	public void setOffset(double left, double top) {
		offsetLeft = left;
		offsetTop = top;
	}

	IECanvasStringImpl(int width, int height) {
		Element e = DOM.createDiv();
		setElement(e);
		setSize(width,height);
		applyClipping(e);
		newBuffer();
		setStroke(0, 0, 0, 1);
		setFill(255,255,255,1);
		setStrokeWeight(1);

		eGroup = DOM.createElement("v:group");
		DOM.setElementAttribute(eGroup, "coordsize", width+","+height);
		DOM.setElementAttribute(eGroup, "width", width+"px");
		DOM.setElementAttribute(eGroup, "height", height+"px");
		DOM.setStyleAttribute(eGroup, "position", "absolute");
		DOM.setStyleAttribute(eGroup, "width", width+"px");
		DOM.setStyleAttribute(eGroup, "height", height+"px");
		DOM.appendChild(e, eGroup);
	}

	public void drawRectangle(double left, double top, double width, double height) {
		double[] x = new double[]{left, left+width, left+width, left};
		double[] y = new double[]{top,top,top+height,top+height};
		drawPolygon(x, y);
	}

	public void drawLine(double fromLeft, double fromTop, double toLeft, double toTop) {
		double[] x = new double[]{fromLeft, toLeft};
		double[] y = new double[]{fromTop,toTop};
		drawPolyLine(x, y);
	}
	
	public void drawArc(double centerLeft, double centerTop, double radiusX, double radiusY, double fromAngle, double toAngle) {
		// make angle overflow compatible with FF
		if (fromAngle > toAngle){
			fromAngle-=(1+Math.floor((fromAngle-toAngle)/(Math.PI*2.0)))*Math.PI*2.0;
		}
		addElement("v:arc", centerLeft-radiusX+offsetLeft, centerTop-radiusY+offsetTop, radiusX*2.0, radiusY*2.0,
				drawArcAtts,
				new String[]{""+(fromAngle * 180.0 / Math.PI),""+(toAngle * 180.0 / Math.PI),"true"}
		);
		buffer.closeElement("v:arc");
	}

	public void setStrokeWeight(double weight) {
		strokeWeight = weight;
		sStroke = null;
	}

	public void drawPolyLine(double[] x, double[] y) {
		int length = x.length;
		
		// Unfortunately IE rotates shapes around their center and not around the coordinate system's
		// offset, thus the shape has to be translated manually. Translation is time consuming, so perform it only when an offset or rotation has
		// been specified.
		if (getRotation()!=0 || offsetLeft!=0 || offsetTop!=0){
			x = Utils.copy(x);
			y = Utils.copy(y);
			//translate(x,y,getRotation(),offsetLeft, offsetTop);
		}

		for (int i = 0; i < length; i++)
			sb.append(x[i]+offsetLeft).append(y[i]+offsetTop);

		addElement("v:polyline", 0, 0, 1, 1,
				polyLineAtts,
				new String[]{sb.toString(), "false"}
		);
		buffer.closeElement("v:polyline");
		sb.clear();
	}

	public void clear() {
		setOffset(0,0);
		setRotation(0);
		setStroke(0, 0, 0, 1);
		setFill(255, 255, 255, 1);
		setStrokeWeight(1);
		newBuffer();
	}
	

	private void newBuffer() {
		buffer.clear();
//		buffer.append("<v:group ").
//		append("coordsize='").append(width).append(",").append(height).append("' ").
//		append("width='").append(width).append("px' height='").append(height).append("px' ").
//		append("style='position:absolute;width:").append(width).append("px;height:").
//		append(height).append("px'>");
	}

	public void flush() {
//		buffer.append("</v:group>");
		DOM.setInnerHTML(eGroup, buffer.toString(""));
		setOffset(0,0);
		newBuffer();
	}

	public void setFill(int red, int green, int blue, double alpha) {
		fillColour = getColor(red, green, blue);
		fillAlpha = alpha;
		sFill = null;
	}

	public void setStroke(int red, int green, int blue, double alpha) {
		strokeColour = getColor(red, green, blue);
		this.strokeAlpha = alpha;
		sStroke = null;
	}

	public void drawPolygon(double[] x, double[] y) {
		
		// Translation is time consuming, so perform it only when an offset or rotation has
		// been specified.
		if (getRotation()!=0 || offsetLeft!=0 || offsetTop!=0){
			x = Utils.copy(x);
			y = Utils.copy(y);
			translate(x,y,getRotation(),offsetLeft, offsetTop);
		}
		
		int mod = x.length;
		int length = mod+1;
		for (int i = 0; i < length; i++) {
			sb.appendCoordinates(x[i % mod]+offsetLeft, y[i % mod]+offsetTop);
		}
		
		addElementPlain("v:polyline", 0, 0, 1, 1,
				polyLineAtts,
				new String[]{sb.toString(),"true"}
		);
		closeDecl();
		sb.clear();
		
		addStroke();
		addFill();
		buffer.closeElement("v:polyline");
		
	}
	
	public void setRotation(double angle) {
		super.setRotation(angle);
		rotation = (angle*180.0/Math.PI)+"deg";
	}

	//TODO: fix opacity for images
	public void drawImage(Element image, double sx, double sy, double swidth, double sheight, double dx, double dy, double dwidth, double dheight) {
		double imageWidth = Utils.getWidth(image);
		double imageHeight = Utils.getHeight(image);
		
		double relativeOffsetLeft = sx/imageWidth;
		double relativeOffsetRight = (imageWidth-swidth-sx)/imageWidth;
		double relativeOffsetTop = sy/imageHeight;
		double relativeOffsetBottom = (imageHeight-sheight-sy)/imageHeight;
		
		addElementPlain("v:rect", dx+offsetLeft, dy+offsetTop, dwidth, dheight,
				noStrokeAtts,
				noStokeVals
		);
		
		// painting transparent png images with alpha components causes ugly results
		if (strokeAlpha!=1)
			buffer.append("style='filter=alpha(opacity=").append(strokeAlpha*100).append(")' ");
		addRotation();
		closeDecl();

		buffer.appendImageData(DOM.getImgSrc(image), relativeOffsetLeft, relativeOffsetRight, relativeOffsetTop, relativeOffsetBottom);
		buffer.closeElement("v:rect");
	}

	public void drawText(String text, double x, double y) {
		font.drawText(text, this, x, y);
	}

	public void setFont(org.gwtwidgets.client.ui.canvas.Font font) {
		this.font = font;
	}

	public Element prepareImage(Element image) {
		return image;
	}
	
}