package io.gatling.recorder.internal.bouncycastle.crypto.generators;

import java.math.BigInteger;
import java.security.SecureRandom;

import io.gatling.recorder.internal.bouncycastle.crypto.AsymmetricCipherKeyPair;
import io.gatling.recorder.internal.bouncycastle.crypto.AsymmetricCipherKeyPairGenerator;
import io.gatling.recorder.internal.bouncycastle.crypto.KeyGenerationParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ECDomainParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ECKeyGenerationParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ECPrivateKeyParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ECPublicKeyParameters;
import io.gatling.recorder.internal.bouncycastle.math.ec.ECConstants;
import io.gatling.recorder.internal.bouncycastle.math.ec.ECMultiplier;
import io.gatling.recorder.internal.bouncycastle.math.ec.ECPoint;
import io.gatling.recorder.internal.bouncycastle.math.ec.FixedPointCombMultiplier;
import io.gatling.recorder.internal.bouncycastle.math.ec.WNafUtil;
import io.gatling.recorder.internal.bouncycastle.util.BigIntegers;

public class ECKeyPairGenerator
    implements AsymmetricCipherKeyPairGenerator, ECConstants
{
    ECDomainParameters  params;
    SecureRandom        random;

    public void init(
        KeyGenerationParameters param)
    {
        ECKeyGenerationParameters  ecP = (ECKeyGenerationParameters)param;

        this.random = ecP.getRandom();
        this.params = ecP.getDomainParameters();
    }

    /**
     * Given the domain parameters this routine generates an EC key
     * pair in accordance with X9.62 section 5.2.1 pages 26, 27.
     */
    public AsymmetricCipherKeyPair generateKeyPair()
    {
        BigInteger n = params.getN();
        int nBitLength = n.bitLength();
        int minWeight = nBitLength >>> 2;

        BigInteger d;
        for (;;)
        {
            d = BigIntegers.createRandomBigInteger(nBitLength, random);

            if (d.compareTo(ONE) < 0  || (d.compareTo(n) >= 0))
            {
                continue;
            }

            if (WNafUtil.getNafWeight(d) < minWeight)
            {
                continue;
            }

            break;
        }

        ECPoint Q = createBasePointMultiplier().multiply(params.getG(), d);

        return new AsymmetricCipherKeyPair(
            new ECPublicKeyParameters(Q, params),
            new ECPrivateKeyParameters(d, params));
    }

    protected ECMultiplier createBasePointMultiplier()
    {
        return new FixedPointCombMultiplier();
    }
}
