package com.cybelia.sandra.web.displaytag;

import java.io.Serializable;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.displaytag.pagination.PaginatedList;
import org.displaytag.properties.SortOrderEnum;

/**
 * Implantation simple de {@link PaginatedList} utilisant un {@link PaginedListExtractor} pour
 * recuperer la liste reelle a afficher et la taille totale de la liste a paginer.
 * <p/>
 * La methode {@link #fromRequest(HttpServletRequest)} permet de positionner les valeurs recuperees
 * depuis la requete.
 * <p/>
 * La methode {@link #acquireData(HttpServletRequest)} permet de recalculer la liste a afficher et la
 * taille totale de la liste.
 *
 * @author chemit
 */
public class PaginedListImpl<E> implements PaginatedList, Serializable {

    protected static final Log log = LogFactory.getLog(PaginedListImpl.class);

    /**
     * contient la liste triee et paginee a afficher
     */
    protected transient List<E> list;

    /**
     * la page couranter a afficher
     */
    protected int pageNumber;

    /**
     * le nombre d'objets par page
     */
    protected int objectsPerPage;

    /**
     * le nombre totale d'elements dans la liste (pas celle a afficher)
     */
    protected Integer fullListSize;

    /**
     * le critere de tri
     */
    protected String sortCriterion;

    /**
     * le prefixe des parametres de la requete pour recuperer les arguments
     */
    protected String requestPrefix = "";

    /**
     * l'ordre du tri
     */
    transient protected SortOrderEnum sortDirection;

    /**
     * le search id (utilise pour les recherches, mais encore par nous...)
     */
    protected String searchId;

    /**
     * l'extracteur de donnees pour recuperer la taille totale de la liste et les listes reelles a afficher
     */
    protected PaginedListExtractor<E> extractor;

    private static final long serialVersionUID = -77286294984699702L;

    /**
     * @param fullListSize la taille de la liste complete
     * @param list         la liste paginee et triee
     */
    public PaginedListImpl(int fullListSize, List<E> list) {
        this(null);
        this.fullListSize = fullListSize;
        this.list = list;
    }

    /**
     * @param extractor l'extracteur de donnees
     */
    public PaginedListImpl(PaginedListExtractor<E> extractor) {
        this.extractor = extractor;
        // par defaut on commence toujours sur la page 1
        this.pageNumber = 1;
        // par defaut on traite 25 entrees par page
        this.objectsPerPage = 25;
        // par defaut on trie dans l'ordre ascendant
        this.sortDirection = SortOrderEnum.ASCENDING;
    }

    /**
     * @param extractor      l'extracteur de donnees
     * @param objectsPerPage le nombre d'object par page
     * @param requestPrefix  le prefixe a ajouter pour obtenir les parametres de la requete
     * @param sortCriterion  le critere de tri par defaut
     * @param sortDirection  l'ordre de tri par defaut
     */
    public PaginedListImpl(PaginedListExtractor<E> extractor, int objectsPerPage, String requestPrefix,
                           String sortCriterion, SortOrderEnum sortDirection) {
        this.extractor = extractor;
        this.objectsPerPage = objectsPerPage;
        this.requestPrefix = requestPrefix;
        this.sortCriterion = sortCriterion;
        this.sortDirection = sortDirection;
        // par defaut on commence toujours sur la page 1
        this.pageNumber = 1;
    }

    @Override
    public List<E> getList() {
        return list;
    }

    @Override
    public int getPageNumber() {
        return pageNumber;
    }

    @Override
    public int getObjectsPerPage() {
        return objectsPerPage;
    }

    @Override
    public int getFullListSize() {
        return fullListSize;
    }

    @Override
    public String getSortCriterion() {
        return sortCriterion;
    }

    @Override
    public SortOrderEnum getSortDirection() {
        return sortDirection;
    }

    @Override
    public String getSearchId() {
        return searchId;
    }

    public void setList(List<E> list) {
        this.list = list;
    }

    public void setPageNumber(int pageNumber) {
        this.pageNumber = pageNumber;
    }

    public void setObjectsPerPage(int objectsPerPage) {
        this.objectsPerPage = objectsPerPage;
    }

    public void setFullListSize(Integer fullListSize) {
        this.fullListSize = fullListSize;
    }

    public void setSortCriterion(String sortCriterion) {
        this.sortCriterion = sortCriterion;
    }

    public void setSortDirection(SortOrderEnum sortDirection) {
        this.sortDirection = sortDirection;
    }

    public void setSearchId(String searchId) {
        this.searchId = searchId;
    }

    public void setRequestPrefix(String requestPrefix) {
        this.requestPrefix = requestPrefix;
    }

    /**
     * Recupere les parametres depuis une requete de servlet.
     *
     * @param request la requete utilisee pour recuperee les valeur du POJO
     * @return l'instance
     */
    public PaginedListImpl<E> fromRequest(HttpServletRequest request) {
        // pageNumber
        String page = request.getParameter(requestPrefix + "page");
        if (page != null && !page.trim().isEmpty()) {
            Integer pageNumber1 = Integer.valueOf(page);
            if (log.isDebugEnabled()) {
                log.debug("change pageNumber : " + pageNumber1);
            }
            setPageNumber(pageNumber1);
        }

        //sortCriterion
        String criterion = request.getParameter(requestPrefix + "sort");
        if (criterion != null && !criterion.trim().isEmpty()) {
            if (log.isDebugEnabled()) {
                log.debug("change sortCriterion : " + criterion);
            }
            setSortCriterion(criterion);
        }

        //sortDirection
        // attention, on supose ici qu'on a defini les variables suivantes dans displaytag.properties :
        // pagination.sort.asc.value=ascending
        // pagination.sort.desc.value=descending
        String direction = request.getParameter(requestPrefix + "dir");
        if (direction != null && !direction.trim().isEmpty()) {
            SortOrderEnum orderEnum = SortOrderEnum.fromName(direction);
            if (log.isDebugEnabled()) {
                log.debug("change sortDirection : " + orderEnum);
            }
            setSortDirection(orderEnum);
        }
        return this;
    }

    /**
     * La methode pour remplir les donnees de la liste, à savoir :
     * <pre>
     *  - le nombre totale d'elements de la liste (fullListSize)
     *  - la liste partielle (list)
     * </pre>
     *
     * @param request la requete courante
     * @return l'instance
     * @throws Exception if any pb
     */
    public PaginedListImpl<E> acquireData(HttpServletRequest request) throws Exception {
        if (fullListSize == null) {
            // on doit recuperer du service la taille totale de la liste
            setFullListSize(extractor.getFullListSize(request));
        }
        if (fullListSize == 0) {
            // no data to retreave, so do not make the extractor call
            setList(null);
        } else {
            // on recupere la liste a afficher
            int startIndex = (pageNumber - 1) * objectsPerPage;
            int endIndex = startIndex + objectsPerPage - 1;
            if (endIndex > fullListSize) {
                endIndex = fullListSize;
            }
            setList(extractor.getList(request, startIndex, endIndex, sortCriterion, sortDirection.getCode()));
        }
        return this;
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append(super.toString());
        builder.append("<pageNumber:");
        builder.append(pageNumber);
        builder.append(", objectsPerPage:");
        builder.append(objectsPerPage);
        builder.append(", fulListSize:");
        builder.append(fullListSize);
        builder.append(", sortCriterion:");
        builder.append(sortCriterion);
        builder.append(", sortDirection:");
        builder.append(sortDirection);
        builder.append(">");
        return builder.toString();
    }
}
