package com.cybelia.sandra.web.action.loading.model;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * Une classe pour representer un jour dans l'entete du tableau et le nombre de colonne qu'il occupe
 *
 * @author chemit
 */
public class Day {

    /**
     * Calcule le tableau des {@link Day} contenus entre les deux dates
     * donnees.
     *
     * @param startingDate la date de debut
     * @param endingDate   la date de fin
     * @return le tableau des dates calcules
     */
    public static Day[] newDays(Date startingDate, Date endingDate) {

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(endingDate);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);

        Date lastDate = calendar.getTime();

        calendar.setTime(startingDate);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        int nbDays = -1;

        while (calendar.getTime().before(lastDate)) {
            calendar.add(Calendar.DAY_OF_YEAR, 1);
            nbDays++;
        }

        if (nbDays == -1) {
            // special case when there is only one day to build
            return new Day[]{newPartialDay(startingDate, endingDate)};
        }

        List<Day> result = new ArrayList<Day>();
        // build first day
        result.add(newFirstDay(startingDate));

        calendar.setTime(startingDate);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.add(Calendar.DAY_OF_YEAR, 1);

        while (nbDays > 0) {
            // buld a complete day
            result.add(newCompleteDay(calendar.getTime()));
            calendar.roll(Calendar.DAY_OF_YEAR, true);
            nbDays--;
        }
        // build last day
        result.add(newLastDay(endingDate));

        return result.toArray(new Day[result.size()]);
    }

    /**
     * To build a partial day from firstDate hour to lastDate hour
     *
     * @param beginDate the day with begin hour
     * @param endDate   the day with ending hour
     * @return the builded day
     */
    static Day newPartialDay(Date beginDate, Date endDate) {
        Day day = new Day(beginDate);
        Calendar calendar = Calendar.getInstance();

        calendar.setTime(beginDate);
        day.firstHour = calendar.get(Calendar.HOUR_OF_DAY);

        calendar.setTime(endDate);
        day.lastHour = calendar.get(Calendar.HOUR_OF_DAY);
        return day;
    }

    /**
     * To build a begin day, from his hour to the end of the day
     *
     * @param date the day with begin hour
     * @return the builded day
     */
    static Day newFirstDay(Date date) {
        Day day = new Day(date);
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        day.firstHour = calendar.get(Calendar.HOUR_OF_DAY);
        day.lastHour = 23;
        return day;
    }

    /**
     * To build a end day, from 0:00 on the day to the hour of the given date
     *
     * @param date the day with end hour
     * @return the builded day
     */
    static Day newLastDay(Date date) {
        Day day = new Day(date);
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        day.firstHour = 0;
        day.lastHour = calendar.get(Calendar.HOUR_OF_DAY);
        return day;
    }

    static Day newCompleteDay(Date date) {
        Day day = new Day(date);
        day.firstHour = 0;
        day.lastHour = 23;
        return day;
    }

    /**
     * le jour concerne
     */
    protected Date day;

    /**
     * la premiere heure
     */
    protected int firstHour;

    /**
     * la derniere heure
     */
    protected int lastHour;

    protected Day(Date day, int firstHour, int lastHour) {
        this.day = day;
        this.firstHour = firstHour;
        this.lastHour = lastHour;
    }

    protected Day(Date day) {
        this.day = day;
    }

    public Date getDay() {
        return day;
    }

    public int getFirstHour() {
        return firstHour;
    }

    public int getLastHour() {
        return lastHour;
    }

    public int getNbHours() {
        return lastHour - firstHour + 1;
    }

}
