package com.cybelia.sandra.web.action;

import com.cybelia.sandra.SandraConfig;
import com.cybelia.sandra.SandraConfigHelper;
import com.cybelia.sandra.entities.Camion;
import com.cybelia.sandra.entities.Chauffeur;
import com.cybelia.sandra.entities.Eleveur;
import com.cybelia.sandra.entities.Label;
import com.cybelia.sandra.entities.Societe;
import com.cybelia.sandra.entities.Tour;
import com.cybelia.sandra.entities.Transporteur;
import com.cybelia.sandra.entities.UserIndicateurs;
import com.cybelia.sandra.entities.Usine;
import com.cybelia.sandra.entities.notifier.Event;
import com.cybelia.sandra.services.LogicException;
import com.cybelia.sandra.services.ServiceWeb;
import com.cybelia.sandra.web.displaytag.PaginedListImpl;
import java.text.ParseException;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import javax.naming.NamingException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts.util.MessageResources;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.taas.entities.TaasUser;

public class ApplicationSession {

    protected static final Log log = LogFactory.getLog(ApplicationSession.class);

    public static final Class[] PROFIL_TYPES = {
            Camion.class,
            Chauffeur.class,
            Eleveur.class,
            Societe.class,
            Transporteur.class,
            Usine.class
    };

    protected static MessageResources messages = MessageResources.getMessageResources("sandra");

    public enum Cache {

        Societe(Societe.class) {

            @Override
            Map<String, String> getCacheInstance(ApplicationSession session) {
                return session.namingSocietes;
            }

            @Override
            String buildNamingLibelle(TopiaEntity topiaEntity) {
                Societe s = (Societe) topiaEntity;
                return s.getCode();
            }

            @Override
            List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException {
                return serviceWeb.getSocietes(null);
            }

            @Override
            public String getDefaultEntry() throws NamingException, LogicException, TopiaException {

                ServiceWeb serviceWeb = ServiceFactory.newServiceWeb();
                TaasUser user = serviceWeb.getUser();

                // For all user other than admin
                if (user.getLink() != null) {
                    Societe userSociete = ServiceFactory.newServiceWeb().getUserSociete(null, user);
                    return userSociete.getTopiaId();
                } else {
                    return super.getDefaultEntry();
                }
            }
        },

        Transporteur(Transporteur.class) {

            @Override
            Map<String, String> getCacheInstance(ApplicationSession session) {
                return session.namingTransporteurs;
            }

            @Override
            String buildNamingLibelle(TopiaEntity entity) {
                Transporteur transporteur = (Transporteur) entity;
                return transporteur.getName() + " (" + transporteur.getSociete().getCode() + "-" + transporteur.getCode() + ")";
            }

            @Override
            List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException {
                return serviceWeb.getTransporteurs(null);
            }
        },

        Camion(Camion.class) {

            @Override
            Map<String, String> getCacheInstance(ApplicationSession session) {
                return session.namingCamions;
            }

            @Override
            String buildNamingLibelle(TopiaEntity entity) {
                Camion camion = (Camion) entity;
                return camion.getSociete().getCode() + "-" + camion.getCode();
            }

            @Override
            List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException {
                return serviceWeb.getCamions(null);
            }

            @Override
            public void clear(ApplicationSession session) {
                super.clear(session);
                // must clear also camionPrincipal
                CamionPrincipal.clear(session);
                CamionActif.clear(session);
            }
        },

        CamionActif(Camion.class) {

            @Override
            Map<String, String> getCacheInstance(ApplicationSession session) {
                return session.namingCamionsActif;
            }

            @Override
            String buildNamingLibelle(TopiaEntity entity) {
                Camion camion = (Camion) entity;
                return camion.getSociete().getCode() + "-" + camion.getCode();
            }

            @Override
            List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException {
                return serviceWeb.getCamions(null, null, true);
            }
        },

        CamionPrincipal(Camion.class) {

            @Override
            Map<String, String> getCacheInstance(ApplicationSession session) {
                return session.namingCamionsPrincipaux;
            }

            @Override
            String buildNamingLibelle(TopiaEntity entity) {
                Camion camion = (Camion) entity;
                return camion.getSociete().getCode() + "-" + camion.getCode();
            }

            @Override
            List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException {
                return serviceWeb.getCamionsPrincipaux(null);
            }
        },

        Chauffeur(Chauffeur.class) {

            @Override
            Map<String, String> getCacheInstance(ApplicationSession session) {
                return session.namingChauffeurs;
            }

            @Override
            String buildNamingLibelle(TopiaEntity entity) {
                Chauffeur chauffeur = (Chauffeur) entity;
                return chauffeur.getNom() + " " + chauffeur.getPrenom() + " (" + chauffeur.getSociete().getCode() + "-" + chauffeur.getCode() + ")";
            }

            @Override
            List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException {
                return serviceWeb.getChauffeurs(null);
            }
        },

        Usine(Usine.class) {

            @Override
            Map<String, String> getCacheInstance(ApplicationSession session) {
                return session.namingUsines;
            }

            @Override
            String buildNamingLibelle(TopiaEntity entity) {
                Usine usine = (Usine) entity;
                return usine.getCode();
            }

            @Override
            List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException {
                return serviceWeb.getUsines(null);
            }
        },

        //fixme : seul les admins peuvent les utiliser donc a mettre dans le context applicatif ?
        Event(Event.class) {

            @Override
            Map<String, String> getCacheInstance(ApplicationSession session) {
                return session.namingEvents;
            }

            @Override
            String buildNamingLibelle(TopiaEntity entity) {
                Event data = (Event) entity;
                String key = "notifier.event.name." + data.getType();
                String i18n = messages.getMessage(key);
                if (i18n != null) {
                    return i18n;
                } else {
                    return "???" + key + "???";
                }
            }

            @Override
            List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException {
                return ServiceFactory.newServiceNotifier().getEvents(null);
            }
        },

        //fixme: peut-etre que ce cache est trop consomateur ? a verifier
        User(TaasUser.class) {

            @Override
            Map<String, String> getCacheInstance(ApplicationSession session) {
                return session.namingUsers;
            }

            @Override
            String buildNamingLibelle(TopiaEntity entity) {
                TaasUser user = (TaasUser) entity;
                return user.getLogin() + " (" + user.getEmail() + ")";
            }

            @Override
            List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException {
                return serviceWeb.getAllUsers(null);
            }
        },

        //fixme: peut-etre que ce cache est trop consomateur ? a verifier
        UserIndicateurs(UserIndicateurs.class) {

            @Override
            Map<String, String> getCacheInstance(ApplicationSession session) {
                return session.namingUsers;
            }

            @Override
            String buildNamingLibelle(TopiaEntity entity) {
                UserIndicateurs userIndicateurs = (UserIndicateurs) entity;
                return userIndicateurs.getTaasUser().getLogin() + " (" + userIndicateurs.getTaasUser().getEmail() + ")";
            }

            @Override
            List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException {
                return serviceWeb.getAllUserIndicateurs(null);
            }
        },

        //fixme : les labels devrait etre dans le context applicatif car pas de permissions dessus ?
        Label(Label.class) {

            @Override
            Map<String, String> getCacheInstance(ApplicationSession session) {
                return session.namingLabels;
            }

            @Override
            String buildNamingLibelle(TopiaEntity entity) {
                Label data = (Label) entity;
                return data.getValeur();
            }

            @Override
            List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException {
                return serviceWeb.getLabels(null);
            }
            @Override
            void addEntry(Map<String, String> namingCache, TopiaEntity topiaEntity) {
                Label l = (Label) topiaEntity;
                namingCache.put(l.getCategorie() + ":" + l.getId(), buildNamingLibelle(topiaEntity));
            }
        };

        private final Class<? extends TopiaEntity> entityClass;

        public Map<String, String> getCache(ApplicationSession session) throws NamingException, LogicException, TopiaException {
            Map<String, String> namingCache = getCacheInstance(session);
            buildNamingCache(namingCache, false);
            return namingCache;
        }

        public int size(ApplicationSession session) throws NamingException, LogicException, TopiaException {
            return getCache(session).size();
        }

        public void clear(ApplicationSession session) {
            log.info("clear cache for " + this);
            getCacheInstance(session).clear();
        }

        public static Cache valueOf(Class<? extends TopiaEntity> klazz) {
            for (Cache cache : values()) {
                if (klazz == cache.entityClass) {
                    return cache;
                }
            }
            return null;
        }

        public String getDefaultEntry() throws NamingException, LogicException, TopiaException {
            ServiceWeb serviceWeb = ServiceFactory.newServiceWeb();
            List<? extends TopiaEntity> list = getListFromService(serviceWeb);
            if (list != null) {
                if (list.size() == 1) {
                    return list.get(0).getTopiaId();
                }
            }
            return null;
        }

        Cache(Class<? extends TopiaEntity> entityClass) {
            this.entityClass = entityClass;
        }

        void buildNamingCache(Map<String, String> namingCache, boolean force) throws NamingException, LogicException, TopiaException {
            if (force || namingCache.isEmpty()) {
                ServiceWeb serviceWeb = ServiceFactory.newServiceWeb();
                List<? extends TopiaEntity> list = getListFromService(serviceWeb);
                if (list != null) {
                    for (TopiaEntity topiaEntity : list) {
                        addEntry(namingCache, topiaEntity);
                    }
                }
            }
        }

        void addEntry(Map<String, String> namingCache, TopiaEntity topiaEntity) {
            namingCache.put(topiaEntity.getTopiaId(), buildNamingLibelle(topiaEntity));
        }

        abstract String buildNamingLibelle(TopiaEntity entity);

        abstract List<? extends TopiaEntity> getListFromService(ServiceWeb serviceWeb) throws LogicException, TopiaException;

        abstract Map<String, String> getCacheInstance(ApplicationSession session);

    }

    public static ApplicationSession get(HttpServletRequest request) throws Exception {
        return get(request.getSession());
    }

    public static ApplicationSession get(HttpSession session) throws Exception {
        // Take inactivity interval in properties
        // Fix in web.xml
//        session.setMaxInactiveInterval(ServiceHelper.getInactiveInterval());

        ApplicationSession result = (ApplicationSession) session.getAttribute("applicationSession");
        if (result == null) {
            session.setAttribute("applicationSession", result = new ApplicationSession());
        }
        return result;
    }

    // Message de bienvenue
    protected String messageBienvenue;

    // CopyRight
    protected String copyright;

    // Clef google map
    protected String keyGoogleMap;

    // Sandra version
    protected String sandraVersion;

    // Previous RAZ date
    protected Date previousRAZ;

    // Message footer  
    protected String msgFooter;

    /**
     * l'utilisateur connecte
     */
    protected TaasUser user;

    // Cache topiaId -> libelle
    protected Map<String, String> namingSocietes = new LinkedHashMap<String, String>();
    protected Map<String, String> namingTransporteurs = new LinkedHashMap<String, String>();
    protected Map<String, String> namingCamions = new LinkedHashMap<String, String>();
    protected Map<String, String> namingCamionsActif = new LinkedHashMap<String, String>();
    protected Map<String, String> namingCamionsPrincipaux = new LinkedHashMap<String, String>();
    protected Map<String, String> namingChauffeurs = new LinkedHashMap<String, String>();
    protected Map<String, String> namingUsines = new LinkedHashMap<String, String>();
    protected Map<String, String> namingEleveurs = new LinkedHashMap<String, String>();
    protected Map<String, String> namingEvents = new LinkedHashMap<String, String>();
    protected Map<String, String> namingUsers = new LinkedHashMap<String, String>();
    protected Map<String, String> namingProfils = new LinkedHashMap<String, String>();
    protected Map<String, String> namingRisques = new LinkedHashMap<String, String>();
    protected Map<String, String> namingAcces = new LinkedHashMap<String, String>();
    protected Map<String, String> namingEquipement = new LinkedHashMap<String, String>();
    protected Map<String, String> namingTime = new LinkedHashMap<String, String>();
    protected Map<String, String> namingActionType = new LinkedHashMap<String, String>();
    protected Map<String, String> gpsState = new LinkedHashMap<String, String>();
    protected Map<String, String> nivSecu = new LinkedHashMap<String, String>();
    protected Map<String, String> operators = new LinkedHashMap<String, String>();
    
    // Cache categorie:id -> valeur
    protected Map<String, String> namingLabels = new LinkedHashMap<String, String>();

    // Cache for step navigation
    protected List<Tour> cachedToursList = null;
    protected String cachedToursTopiaId = null;
    protected String cachedCamionTopiaId = null;
    protected Date cachedDebut = null;
    protected Date cachedFin = null;

    // Cache tours
    protected PaginedListImpl<Tour> listTours;


    // getters & setters
    public List<Tour> getCachedToursList() {
        return cachedToursList;
    }

    public void setCachedToursList(List<Tour> cachedToursList) {
        this.cachedToursList = cachedToursList;
    }

    public String getCachedToursTopiaId() {
        return cachedToursTopiaId;
    }

    public void setCachedToursTopiaId(String cachedToursTopiaId) {
        this.cachedToursTopiaId = cachedToursTopiaId;
    }

    public String getCachedCamionTopiaId() {
        return cachedCamionTopiaId;
    }

    public void setCachedCamionTopiaId(String cachedCamionTopiaId) {
        this.cachedCamionTopiaId = cachedCamionTopiaId;
    }

    public Date getCachedDebut() {
        return cachedDebut;
    }

    public void setCachedDebut(Date cachedDebut) {
        this.cachedDebut = cachedDebut;
    }

    public Date getCachedFin() {
        return cachedFin;
    }

    public void setCachedFin(Date cachedFin) {
        this.cachedFin = cachedFin;
    }

    public PaginedListImpl<Tour> getListTours() {
        return listTours;
    }

    public void setListTours(PaginedListImpl<Tour> listTours) {
        this.listTours = listTours;
    }

    public String getMessageBienvenue() throws Exception {
        if (messageBienvenue == null) {
            ServiceWeb serviceWeb = ServiceFactory.newServiceWeb();
            messageBienvenue = serviceWeb.getMessageBienvenue(null);
        }
        return messageBienvenue;
    }

    public String getCopyright() throws Exception {
        if (copyright == null) {
            ServiceWeb serviceWeb = ServiceFactory.newServiceGuest();
            copyright = serviceWeb.getCopyright(null);
        }
        return copyright;
    }

    public String getGoogleMapKey() {
        if (keyGoogleMap == null) {
            keyGoogleMap = SandraConfigHelper.getGoogleMapKey(SandraConfig.getConfig());
        }
        return keyGoogleMap;
    }

    public String getSandraVersion() {
        if (sandraVersion == null) {
            sandraVersion = SandraConfigHelper.getSandraVersion(SandraConfig.getConfig());
        }
        return sandraVersion;
    }

    public Date getPreviousRAZ() {
        if (previousRAZ == null) {
            try {
                previousRAZ = SandraConfigHelper.getPreviousRAZDate(SandraConfig.getConfig());
            } catch (ParseException eee) {
                log.info("Failed to load previous date of RAZ", eee);
            }
        }
        return previousRAZ;
    }

    public TaasUser getUser() throws Exception {
        if (user == null) {
            ServiceWeb serviceWeb = ServiceFactory.newServiceWeb();
            user = serviceWeb.getUser();
        }
        return user;
    }

    public String getFooterMsg() throws Exception {
        if (msgFooter == null) {
            msgFooter = "Sandra " + getSandraVersion() + " - " + getCopyright();
        }
        return msgFooter;
    }

    public Map<String, String> getNaming(Class<? extends TopiaEntity> klass) throws Exception {
        Cache cache = Cache.valueOf(klass);
        if (cache == null) {
            return null;
        }
        return cache.getCache(this);
    }

    public void clearNaming(Class<? extends TopiaEntity> klass) {
        Cache.valueOf(klass).clear(this);
    }

    public Map<String, String> getNamingSocietes() throws Exception {
        return Cache.Societe.getCache(this);
    }

    public Map<String, String> getNamingTransporteurs() throws Exception {
        return Cache.Transporteur.getCache(this);
    }

    public Map<String, String> getNamingCamions() throws Exception {
        return Cache.Camion.getCache(this);
    }

    public Map<String, String> getNamingCamionsActifs() throws Exception {
        return Cache.CamionActif.getCache(this);
    }

    public Map<String, String> getNamingCamionsPrincipaux() throws Exception {
        return Cache.CamionPrincipal.getCache(this);
    }

    public Map<String, String> getNamingChauffeurs() throws Exception {
        return Cache.Chauffeur.getCache(this);
    }

    public Map<String, String> getNamingUsines() throws Exception {
        return Cache.Usine.getCache(this);
    }

    public Map<String, String> getNamingEvents() throws Exception {
        return Cache.Event.getCache(this);
    }

    public Map<String, String> getNamingUsers() throws Exception {
        return Cache.User.getCache(this);
    }

    public Map<String, String> getNamingUserIndicateurs() throws Exception {
        return Cache.UserIndicateurs.getCache(this);
    }

    public Map<String, String> getNamingLabels() throws Exception {
        return Cache.Label.getCache(this);
    }

    public Map<String, String> getNamingLabelRisques() throws Exception {
        // FIXME : voir si tout est bien trie
        return MapUtils.orderedMap(getNamingLabelByCategorie(namingRisques, "RSK"));
    }

    public Map<String, String> getNamingLabelAccessSilos() throws Exception {
        return getNamingLabelByCategorie(namingAcces, "IAS");
    }

    public Map<String, String> getNamingLabelEquipements() throws Exception {
        return getNamingLabelByCategorie(namingEquipement, "IMC");
    }

    public Map<String, String> getGpsStates() throws Exception {
        return getNamingLabelByCategorie(gpsState, "GPS");
    }

    public Map<String, String> getNivSecu() throws Exception {
        return getNamingLabelByCategorie(nivSecu, "NDS");
    }

    public Map<String, String> getOperators() throws Exception {
        return getNamingLabelByCategorie(operators, "OPE");
    }

    public Map<String, String> getNamingTime() throws Exception {
        return getNamingLabelByCategorie(namingTime, "ECH");
    }

    public Map<String, String> getNamingActionType() throws Exception {
        return getNamingLabelByCategorie(namingActionType, "ACT");
    }

    protected Map<String, String> getNamingLabelByCategorie(Map<String, String> naming, String categorie) throws Exception {
        if (naming.isEmpty()) {
            Map<String, String> labels = Cache.Label.getCache(this);
            for (String key : labels.keySet()) {
                if (key.matches("^" + categorie + "\\:.+$")) {
                    String id = key.split(":")[1];
                    naming.put(id, labels.get(key));
                }
            }
        }
        return naming;
    }

    public synchronized Map<String, String> getNamingProfils() {
        if (namingProfils.isEmpty()) {
            for (Class profilType : PROFIL_TYPES) {
                namingProfils.put(profilType.getName(), profilType.getSimpleName());
            }
        }
        return namingProfils;
    }

    /**
     * Liste des templates pour la notification *
     */
    protected List<String> templates;

    public List<String> getTemplates() {
        return templates;
    }

    public void setTemplates(List<String> templates) {
        this.templates = templates;
    }

}
