package com.cybelia.sandra;

import com.cybelia.sandra.notifier.SandraNotifier;
import freemarker.core.Environment;
import freemarker.template.Configuration;
import freemarker.template.Template;
import java.io.FileReader;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.mail.EmailAttachment;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.MultiPartEmail;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.ApplicationConfig;

/**
 * @author sletellier
 */
public class EmailHelper {

    private static final Log log = LogFactory.getLog(EmailHelper.class);

    public static void sendEmail(String[] to, String subject, String msg, String... attachments) {
        log.info("Mail will sent to subject : " + subject + " to " + Arrays.toString(to));
        ApplicationConfig config = SandraConfig.getConfig();

        // Création mail
        MultiPartEmail email = new MultiPartEmail();
        email.setCharset("utf-8");

        email.setHostName(SandraSchedulerConfigHelper.getSandraEmailHost(config));
        email.setSmtpPort(SandraSchedulerConfigHelper.getSandraEmailPort(config));
        email.setSubject(subject);
        try {
            email.setFrom(SandraSchedulerConfigHelper.getSandraEmailForm(config));
            for (String recipient : to) {
                email.addTo(recipient);
            }

            email.addPart(msg, MultiPartEmail.TEXT_PLAIN + ";charset=UTF-8");

            for (String attachmentPath : attachments) {
                EmailAttachment attachment = new EmailAttachment();
                attachment.setPath(attachmentPath);
                email.attach(attachment);
            }

            // pour l'erreur java.lang.String.concat avec NullPointerException,
            // la solution est de supprimer -e dans le fichier .mailcap
            // envoi du mail
            email.send();

            log.info("Mail sent subject : " + subject + " to " + Arrays.toString(to));
        } catch (EmailException e) {
            String msgErr = "EmailException impossible d'envoyer le mail";
            log.fatal(msgErr, e);
            throw new RuntimeException(msgErr, e);
        }
    }

    public static void sendEmail(String to, String subject, String message, String... attachments) {
        sendEmail(new String[]{to}, subject, message, attachments);
    }

    public static void sendEmailToAdmin(String subject, String message, String... attachments) {
        sendEmail(SandraSchedulerConfigHelper.getSandraEmailTo(SandraConfig.getConfig()), subject, message, attachments);
    }

    public static void sendEmailFromTemplate(String templateName, Map<String, Object> context, String to, List<String> pjs) {
        sendEmailFromTemplate(templateName, context, new String[]{to}, pjs);
    }
    
    public static void sendEmailFromTemplate(String templateName, Map<String, Object> context, String[] to, List<String> pjs) {
        if (log.isDebugEnabled()) {
            log.debug("sendEmailFromTemplate");
        }

        // Recuperation du template
        ApplicationConfig appConfig = SandraConfig.getConfig();
        String dirName = SandraSchedulerConfigHelper.getNotifierTemplateDir(appConfig);
        String fileName = dirName + templateName + SandraSchedulerConfigHelper.getNotifierTemplateExt(appConfig);

        if (log.isDebugEnabled()) {
            log.debug("dirName : " + dirName + " fileName : " + fileName);
        }

        // Rendu du template
        StringWriter content = new StringWriter();

        // Configuration
        Configuration config = new Configuration();

        // Encoding par default
        config.setDefaultEncoding("utf8");
        try {
            Template template = new Template(dirName, new FileReader(fileName), config, "utf8");
            if (log.isDebugEnabled()){
                log.debug("Template encoding : " + template.getEncoding());
            }
            Environment env = template.createProcessingEnvironment(context, content);

            env.setOutputEncoding("utf8");
            env.process();
        } catch (Exception e) {
            log.error("Generating failed fileName : " + fileName + " content : " + content.toString(), e);
        }

        // Envoi de la notification
        String subject = "";
        String body = "";
        StringBuffer message = content.getBuffer();
        int index = message.indexOf("\n");
        if (index != -1) {
            subject = message.substring(0, index).trim();
            body = message.substring(index).trim();
        } else {
            subject = "(no subject)";
            body = message.toString().trim();
        }
        if (!body.isEmpty()) {
            if (log.isInfoEnabled()) {
                log.info("sendEmail to : " + to + " subject : " + subject);
//                log.info("body :\n " + body);
            }
            EmailHelper.sendEmail(to, subject, body, pjs.toArray(new String[pjs.size()]));
        }
    }
        
    public static void sendNotification(String name, List<String[]> datas) {
        try {
            // Extraction des donnees
            List<List<String>> reformatData = SandraHelper.reformatData(datas);

            // Creation dun string de la forme topiaId&&topiaId...
            String etapeTopiaIdsAsString = SandraHelper.convertToListForNotifier(reformatData.get(0));
            String eleveurTopiaIdsAsString = SandraHelper.convertToListForNotifier(reformatData.get(1));
            String tourTopiaIdsAsString = SandraHelper.convertToListForNotifier(reformatData.get(2));
            String camionTopiaIdsAsString = SandraHelper.convertToListForNotifier(reformatData.get(3));

            // Envoie de la notification
            ApplicationConfig config = SandraConfig.getConfig();

            SandraNotifier sandraNotifier = new SandraNotifier();
            sandraNotifier.notifyEvent(name,
                    SandraSchedulerConfigHelper.getSandraName(config),
                    SandraSchedulerConfigHelper.getSandraUrl(config),
                    etapeTopiaIdsAsString,
                    eleveurTopiaIdsAsString,
                    tourTopiaIdsAsString,
                    camionTopiaIdsAsString);

        } catch (TopiaException ex) {
            log.error("TopiaException : ", ex);
        }
    }
}
