package fr.onema.sispea.struts.referential.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.notification.MailType;
import fr.onema.sispea.service.notification.SispeaMail;
import fr.onema.sispea.service.referential.CollectivityDto;
import fr.onema.sispea.service.referential.OrganismType;
import fr.onema.sispea.service.referential.RPQSDto;
import fr.onema.sispea.service.referential.RPQSType;
import fr.onema.sispea.service.referential.TerritoryDto;
import fr.onema.sispea.struts.common.menu.MenuConstants;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Value;

import java.io.File;
import java.util.Collections;
import java.util.List;

/**
 * This action update a collectivity RPQS.
 *
 * @author CS
 */
public class UpdateRPQSAction extends AbstractCollectivityAction {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = Logger.getLogger(UpdateRPQSAction.class);

    private static final String URL_PREFIX = "http://";

    @Value("${sispea.site.urlRoot}")
    protected boolean siteUrlRoot;

    /**
     * The rpqs id to delete
     */
    protected Integer rpqsId;

    /**
     * The rpqs original filename.
     */
    protected String rpqsFileName;

    /**
     * The rpqs content/MIME type.
     */
    protected String rpqsContentType;

    /**
     * The rpqs file.
     */
    protected File rpqs;

    /**
     * The rpqs link.
     */
    protected String rpqsLink;

    /**
     * The rpqs link.
     */
    protected String rpqsName;

    /**
     * The rpqs type.
     */
    protected String rpqsType;

    /**
     * Territory associated to current organism displayed.
     */
    protected TerritoryDto territory;

    /**
     * Flag to know if RPQS Publication is currently modified.
     */
    protected boolean modifPublication;

    @Override
    protected String doExecute() throws Exception {
        String result = SUCCESS;

        if (logger.isDebugEnabled()) {
            logger.debug("update collectivity RPQS");
        }

        // check params
        checkExercise();
        checkId();

        // Get collectivity model instance.
        collectivity = referentialService.readCollectivity(orgId, exercise, true);

        // Test if collectivity has been correctly retrieved.
        if (collectivity == null) {
            addActionError(getText(COLLECTIVITY_PREFIX + "error.noCollectivity", id));
            result = ERROR;
        }

        // load territory
        territory = territoryService.readTerritory(collectivity.getId(), OrganismType.Collectivity);

        // Open menu according to the current JSP item.
        openMenu(MenuConstants.MENU_KEY_SEE_COLLECTIVITY);

        // no right check : data visible for everybody

        // Test the user query type.
        if (submitValue.equals(getText(COLLECTIVITY_PREFIX + "cancelRPQS"))) {
            // RPQS is no more modified.
            modifPublication = false;
        } else if (submitValue.equals(getText(COLLECTIVITY_PREFIX + "saveRPQS"))) {

            // new RPQS
            RPQSDto lRpqs = new RPQSDto();
            lRpqs.setExercise(exercise);

            // check name
            if (rpqsName == null || rpqsName.trim().length() == 0) {
                // Error.
                addActionError(getText(COLLECTIVITY_PREFIX + "error.noRpqsName"));
                result = ERROR;
            } else {
                // name
                lRpqs.setName(rpqsName);

                // Verify that the URL or File radio button has been chosen.
                if (getText("fr.onema.sispea.jsp.fixRef.showCollectivity.inputURL").equals(getRpqsType())) {


                    // Update the RPQS type.
                    lRpqs.setType(RPQSType.Url);

                    // Update the RPQS Link.
                    if (!rpqsLink.startsWith(URL_PREFIX)) {
                        rpqsLink = URL_PREFIX + rpqsLink;
                    }
                    lRpqs.setLink(rpqsLink);

                    // If URL radio button chosen, test the RPQS link field.
                    if (lRpqs.getLink() == null || lRpqs.getLink().equals("")) {
                        // Error.
                        addActionError(getText(COLLECTIVITY_PREFIX + "error.noRpqsLink"));
                        result = ERROR;
                    }
                } else if (getText("fr.onema.sispea.jsp.fixRef.showCollectivity.selectFile").equals(getRpqsType())) {

                    // Update the RPQS type.
                    lRpqs.setType(RPQSType.File);

                    // If File radio button chosen, test the RPQS filename.
                    if (rpqs == null) {
                        // Error.
                        addActionError(getText(COLLECTIVITY_PREFIX + "error.noRpqsFile"));
                        result = ERROR;
                    } else {
                        // Update filename.
                        lRpqs.setFilename(rpqsFileName);

                        // Update content type.
                        lRpqs.setContentType(rpqsContentType);

                        // Load the file, and save it with the collectivity into the database.
                        lRpqs.setFile(rpqs);
                    }

                } else {
                    // Error.
                    addActionError(getText(COLLECTIVITY_PREFIX + "error.noRpqsType"));
                    result = ERROR;
                }
            }

            // save RPQS ?
            if (!ERROR.equals(result)) {
                // RPQS is no more modified.
                modifPublication = false;

                // Save collectivity into DB.
                referentialService.addRpqsToCollectivity(collectivity, lRpqs);

                // reload
                collectivity = referentialService.readCollectivity(orgId, exercise, true);

                // Send notification mail
                notifRpqsSaved(lRpqs, collectivity);
            }

        } else if (submitValue.equals(getText(COLLECTIVITY_PREFIX + "publishRPQS"))) {

            // RPQS is being modified.
            modifPublication = true;

            // If RPQS is published...
        } else if ((submitValue.equals(getText(COLLECTIVITY_PREFIX + "deleteRPQS")))) {

            // check if rpqs id is set
            if (rpqsId == null) {
                // Error.
                addActionError(getText(COLLECTIVITY_PREFIX + "error.noRpqsIdToDelete"));
                result = ERROR;
            } else {
                // delete rpqs
                referentialService.deleteRpqs(rpqsId);
            }

            // reload
            collectivity = referentialService.readCollectivity(orgId, exercise, true);

        }

        // change menu ?
        openMenu(getProperMenuKey(territory));

        return result;
    }

    /**
     * This method sends a notification when a RQPS is saved.
     *
     * @param pRpqs         RPQS
     * @param pCollectivity Collectivity
     * @return true if the mail has been sent successfully.
     * @throws SispeaException
     */
    protected boolean notifRpqsSaved(RPQSDto pRpqs, CollectivityDto pCollectivity) throws SispeaException {
        // collectivity territory
        TerritoryDto lCollectivityTerritory = territoryService.readTerritory(pCollectivity);

        // The e-mail to send
        SispeaMail eMail = new SispeaMail(MailType.rpqsUploaded, getCurrentUser(), lCollectivityTerritory, pRpqs.getExercise());
        eMail.setModifier(getCurrentUser());
        eMail.addParam("collectivityName", pCollectivity.getName());
        eMail.addParam("rpqsName", pRpqs.getName());
        eMail.addParam("year", pRpqs.getExercise().getStrYear());

        // build RPQS url

        // get related users
        List<String> lUsers = userService.readCollectivityRelatedUsersEmail(pCollectivity.getId(), exerciseService.getCurrentExerciseId());
        for (String email : lUsers) {
            if (email != null) {
                eMail.addToAddress(email);
            }
        }

        if (pRpqs.getType() == RPQSType.File) {
            eMail.addParam("rpqsFile", "true");
            List<RPQSDto> lRpqsList = pCollectivity.getRpqsList();
            Collections.reverse(lRpqsList);
            for (RPQSDto lRpqs : lRpqsList) {
                if (lRpqs.getName().equals(pRpqs.getName())) {
                    // build url for rpqs download
                    String lRpqsUrl = siteUrlRoot + getRpqsDownloadUrlFilled(pCollectivity.getId(), pRpqs.getExercise().getId(), lRpqs.getId());
                    eMail.addParam("rpqsLink", lRpqsUrl);
                }
            }
        } else {
            eMail.addParam("rpqsLink", pRpqs.getLink());
        }

        eMail.setLang(getLocale().getLanguage());

        return userService.sendMail(eMail, false);
    }

    public Integer getRpqsId() {
        return rpqsId;
    }

    public void setRpqsId(Integer pRpqsId) {
        rpqsId = pRpqsId;
    }

    public String getRpqsFileName() {
        return rpqsFileName;
    }

    public void setRpqsFileName(String rpqsFileName) {
        this.rpqsFileName = rpqsFileName;
    }

    public String getRpqsContentType() {
        return rpqsContentType;
    }

    public void setRpqsContentType(String rpqsContentType) {
        this.rpqsContentType = rpqsContentType;
    }

    public File getRpqs() {
        return rpqs;
    }

    public void setRpqs(File rpqs) {
        this.rpqs = rpqs;
    }

    public String getRpqsLink() {
        return rpqsLink;
    }

    public void setRpqsLink(String rpqsLink) {
        this.rpqsLink = rpqsLink;
    }

    public String getRpqsName() {
        return rpqsName;
    }

    public void setRpqsName(String rpqsName) {
        this.rpqsName = rpqsName;
    }

    public String getRpqsType() {
        return rpqsType;
    }

    public void setRpqsType(String rpqsType) {
        this.rpqsType = rpqsType;
    }

    public TerritoryDto getTerritory() {
        return territory;
    }

    public void setTerritory(TerritoryDto territory) {
        this.territory = territory;
    }

    public boolean getModifPublication() {
        return modifPublication;
    }

    public void setModifPublication(boolean modifPublication) {
        this.modifPublication = modifPublication;
    }

    public String getRpqsDownloadUrlFilled(Integer pCollectivityId, Integer pChosenExerciseId, Integer pRpqsId) {
        return RPQS_DOWNLOAD_URL + "?id=" + pCollectivityId + "&chosenExerciseId=" + pChosenExerciseId + "&rpqsId=" + pRpqsId;
    }

}