package fr.onema.sispea.struts.exchange.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import com.opensymphony.xwork2.Action;
import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.data.ExerciseDto;
import fr.onema.sispea.service.data.ExerciseService;
import fr.onema.sispea.service.exchange.ExportDto;
import fr.onema.sispea.service.exchange.FileExchangeService;
import fr.onema.sispea.service.exchange.FileType;
import fr.onema.sispea.service.referential.CompetenceDto;
import fr.onema.sispea.service.referential.ReferentialService;
import fr.onema.sispea.service.referential.TerritoryDto;
import fr.onema.sispea.service.user.UserDto;
import fr.onema.sispea.struts.AbstractSispeaAction;
import fr.onema.sispea.struts.exchange.bean.AvailableDownload;
import fr.onema.sispea.struts.search.bean.BackType;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * This action handles the file downloading.
 * It allows the end user to download data and referential.
 *
 * @author CS
 */
public class FileDownloadAction extends AbstractSispeaAction {

    public static final String[] EMPTY_STRING_ARRAY = new String[]{};

//    protected static final Locale DEFAULT_LOCALE = Locale.FRENCH;

    private static final long serialVersionUID = 1L;

    private static final Logger logger = Logger.getLogger(FileDownloadAction.class);

    @Autowired
    protected transient FileExchangeService fileExchangeService;

    @Autowired
    protected transient ExerciseService exerciseService;

    @Autowired
    protected transient ReferentialService referentialService;

    /**
     * territory id to extract
     */
    protected Integer extractTerritoryId;

    /**
     * territory to extract
     */
    protected TerritoryDto extractTerritory;

    /**
     * The filename.
     */
    protected String fileName;

    /**
     * file type
     */
    protected FileType fileType;

    /**
     * Flag to note if the data should be included in downloaded file
     */
    protected Boolean withData;

    /**
     * Flag to note if the referential should be included in downloaded file
     */
    protected Boolean withReferential;

    /**
     * the competence list
     */
    protected List<CompetenceDto> competences = new ArrayList<>();

    /**
     * the chosen competences
     */
    protected String[] chosenCompetences = EMPTY_STRING_ARRAY;

    /**
     * The name of the returned file.
     */
    protected String contentDisposition;

    /**
     * The content type (MIME type).
     */
    protected String contentType;

    /**
     * The file size.
     */
    protected Long fileSize;

    /**
     * The CSV input stream.
     */
    protected transient InputStream inputStream;

    /**
     * file format to download
     */
    protected String downloadFormat;

    /**
     * the selected exercise id
     */
    protected Integer exerciseId;

    /**
     * the user selected territories
     */
    protected List<TerritoryDto> userSelection;

    /**
     * Teritory Id selected returned from "Enhance my selections"
     */
    protected Integer selectionId;

    protected Map<Integer, String> exerciseList;

    /**
     * List of available prepared export documents.
     * This list is calculated from the user rights.
     */
    private List<AvailableDownload> availableDownloads;

    public FileDownloadAction() {
        withReferential = Boolean.FALSE;
        withData = Boolean.FALSE;
    }

    @Override
    protected String doExecute() throws Exception {
        if (logger.isDebugEnabled()) {
            logger.debug("Downloading file");
        }

        // nothing

        return Action.SUCCESS;
    }

    /**
     * This method fetches the available downloads from the database.
     * The downloads depend on the user rights.
     *
     * @throws SispeaException
     */
    protected void loadAvailableDownloads(UserDto pUser) throws SispeaException {

        if (pUser.isGeneralAdmin()) {
            // Use the file exchange service to get all available downloads
            Collection<ExportDto> exports = fileExchangeService.loadAvailableDownloads(pUser);

            // Loop through the exports to create the GUI beans
            if (exports != null) {
                availableDownloads = new ArrayList<>();
                for (ExportDto export : exports) {
                    // Format the generation date
                    String generationDate = null;
                    if (export.getGenerationDate() != null) {
                        SimpleDateFormat dateFormat = new SimpleDateFormat(getFormatDateOnly());
                        generationDate = dateFormat.format(export.getGenerationDate());
                    }
                    availableDownloads.add(new AvailableDownload(export.getId(), getText(export.getName()), generationDate));
                }
            }
        }
    }

    /**
     * Manage the default values to display on jsp
     */
    protected void manageDefaultValues() {

        // data
        withData = Boolean.TRUE;
        withReferential = Boolean.FALSE;

        // format
        UserDto lUser = getCurrentUser();
        if (lUser != null && lUser.isDdt()) {
            downloadFormat = getText("fr.onema.sispea.exchange.download.file.formatXML");
        } else {
            downloadFormat = getText("fr.onema.sispea.exchange.download.file.formatCSV");
        }

    }

    /**
     * check competence params
     *
     * @throws SispeaException
     */
    protected void checkCompetence() throws SispeaException {

        // load competences
        competences = referentialService.readCompetences();
    }

    /**
     * check user's selections and fill corresponding list
     *
     * @throws SispeaException
     */
    protected void checkUsersSelections() throws SispeaException {

        // connected ?
        UserDto lUser = getCurrentUser();
        if (lUser == null) {
            // not connected -> search in session
            userSelection = getSispeaSession().getSelectedTerritories();
        } else {
            // connected -> search in DB
            userSelection = referentialService.readUserSelections(lUser);

            // current user's territory ?
            if (lUser.getTerritory() != null) {

                // if current user territory not in list -> add it
                if (!userSelection.contains(lUser.getTerritory())) {
                    // if territory is the France then only admin can have it
                    if (lUser.getTerritory().isSuperVirtual()) {
                        if (lUser.isGeneralAdmin()) {
                            userSelection.add(lUser.getTerritory());
                        }
                    } else {
                        userSelection.add(lUser.getTerritory());
                    }
                }
            }
        }

        // sort list
        Collections.sort(userSelection);

    }

    /**
     * @return the list of user's selected territories
     */
    public Map<String, String> getSelectedTerritories() {
        // init result
        Map<String, String> lRes = new TreeMap<>();

        // add user's reference territory

        if (userSelection != null && userSelection.size() > 0) {

            for (TerritoryDto lTerritory : userSelection) {
                boolean lAddTerritory = true;

                // do not add general territory -> only for admin
                if (lTerritory.isSuperVirtual() && (!getCurrentUser().isGeneralAdmin())) {
                    lAddTerritory = false;
                }

                // add ?
                if (lAddTerritory) {
                    lRes.put(lTerritory.getId().toString(), lTerritory.getName());
                }
            }
        }

        // result
        return lRes;
    }

    /**
     * @return the list of exercises
     */
    public Map<Integer, String> getExerciseList() throws SispeaException {
        return exerciseList;
    }

    public Integer getBackPageId() {
        return BackType.download.getId();
    }

    public TerritoryDto getExtractTerritory() {
        return extractTerritory;
    }

    public void setExtractTerritory(TerritoryDto pExtractTerritory) {
        extractTerritory = pExtractTerritory;
    }

    public String getContentType() {
        return contentType;
    }

    public void setContentType(String pContentType) {
        contentType = pContentType;
    }

    public Long getFileSize() {
        return fileSize;
    }

    public void setFileSize(Long pFileSize) {
        fileSize = pFileSize;
    }

    public InputStream getInputStream() {
        return inputStream;
    }

    public void setInputStream(InputStream pInputStream) {
        inputStream = pInputStream;
    }

    public String getDownloadFormat() {
        return downloadFormat;
    }

    public void setDownloadFormat(String pDownloadFormat) {
        downloadFormat = pDownloadFormat;
    }

    public String getFileName() {
        return fileName;
    }

    public void setFileName(String pFileName) {
        fileName = pFileName;
    }

    public Boolean getWithData() {
        return withData;
    }

    public void setWithData(Boolean pWithData) {
        withData = pWithData;
    }

    public Boolean getWithReferential() {
        return withReferential;
    }

    public void setWithReferential(Boolean pWithReferential) {
        withReferential = pWithReferential;
    }

    public List<CompetenceDto> getCompetences() {
        return competences;
    }

    public void setCompetences(List<CompetenceDto> pCompetences) {
        competences = pCompetences;
    }

    public String[] getChosenCompetences() {
        return chosenCompetences;
    }

    public void setChosenCompetences(String[] pChosenCompetences) {
        chosenCompetences = pChosenCompetences;
    }

    public FileType getFileType() {
        return fileType;
    }

    public void setFileType(FileType pFileType) {
        fileType = pFileType;
    }

    public Integer getExerciseId() {
        return exerciseId;
    }

    public void setExerciseId(Integer pExerciseId) {
        exerciseId = pExerciseId;
    }

    public Integer getExtractTerritoryId() {
        return extractTerritoryId;
    }

    public void setExtractTerritoryId(Integer pExtractTerritoryId) {
        extractTerritoryId = pExtractTerritoryId;
    }

    public List<TerritoryDto> getUserSelection() {
        return userSelection;
    }

    public void setUserSelection(List<TerritoryDto> pUserSelection) {
        userSelection = pUserSelection;
    }

    public String getContentDisposition() {
        return contentDisposition;
    }

    public void setContentDisposition(String pContentDisposition) {
        contentDisposition = pContentDisposition;
    }

    public List<AvailableDownload> getAvailableDownloads() {
        return availableDownloads;
    }

    public Integer getSelectionId() {
        return selectionId;
    }

    public void setSelectionId(Integer pSelectionId) {
        selectionId = pSelectionId;
    }

    protected void loadExcerciseList() throws SispeaException {

        exerciseList = new TreeMap<>();
        List<ExerciseDto> lExercises = exerciseService.readExercises();
        for (ExerciseDto lExercise : lExercises) {
            exerciseList.put(lExercise.getId(), lExercise.getYear().toString());
        }

    }
}
