package fr.onema.sispea.struts.admin.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.data.DataListColorType;
import fr.onema.sispea.service.data.DataListDto;
import fr.onema.sispea.service.data.ExerciseService;
import fr.onema.sispea.service.data.IndicatorDto;
import fr.onema.sispea.service.data.IndicatorService;
import fr.onema.sispea.service.data.VPDto;
import fr.onema.sispea.service.referential.CompetenceDto;
import fr.onema.sispea.service.referential.ReferentialService;
import fr.onema.sispea.service.referential.TerritoryDto;
import fr.onema.sispea.service.user.UserDto;
import fr.onema.sispea.struts.AbstractSispeaAction;
import fr.onema.sispea.struts.search.bean.BackType;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;

/**
 * This action handles the new exercise management.
 * It allows the end user to initiate a new exercise.
 *
 * @author CS
 */
public abstract class AbstractDataListsAction extends AbstractSispeaAction {

    private static final long serialVersionUID = 1L;

    @Autowired
    protected transient ReferentialService referentialService;

    @Autowired
    protected transient IndicatorService indicatorService;

    @Autowired
    protected transient ExerciseService exerciseService;

    /**
     * String listing indicators separated by semi columns
     */
    protected String indicators;

    /**
     * Lists of IP/VP
     */
    protected List<DataListDto> lists;

    /**
     * Currently selected list
     */
    protected DataListDto currentList;

    /**
     * the user selected territories
     */
    protected List<TerritoryDto> userSelections;

    /**
     * Id of the territory selected in the combo box
     */
    protected Integer territoryId;

    /**
     * The Id of the list selected in the table
     */
    protected Integer selectedListId;

    /**
     * Teritory Id selected returned from "Enhance my selections"
     */
    protected Integer selectionId;

    /**
     * Id of the color selected in the combo box
     */
    protected Integer colorId;

    /**
     * Id of the selected From exercise.
     */
    protected Integer fromExerciseId;

    /**
     * Id of the selected To exercise.
     */
    protected Integer toExerciseId;

    /**
     * Id of the selected competence.
     */
    protected Integer competenceId;

    protected Map<String, String> competenceList;

    protected Map<String, String> exerciseList;

    public Map<String, String> getCompetenceList() {
        return competenceList;
    }

    public Map<String, String> getExerciseList() {
        return exerciseList;
    }

    protected void loadExerciseList() throws SispeaException {

        exerciseList = exerciseService.readExercisesMapAsc();

    }

    protected void loadCompetenceList() throws SispeaException {

        competenceList = new TreeMap<>();
        List<CompetenceDto> lCompetences = referentialService.readCompetences();
        for (CompetenceDto lCompetence : lCompetences) {
            competenceList.put(lCompetence.getId().toString(), lCompetence.getName());
        }

    }


    protected void manageViewActions() throws SispeaException {

        if (selectedListId == null && selectionId != null) {
            // if user just enhanced his selections -> select the territory he just added
            territoryId = selectionId;
        }
    }

    protected void loadLists() throws SispeaException {
        // retrieve lists
        lists = indicatorService.readDataLists();

        // is there any selected list for edition?
        if (selectedListId != null) {
            // a list is selected: load it
            for (DataListDto lDl : lists) {
                if (lDl.getId().equals(selectedListId)) {
                    currentList = lDl;
                    territoryId = lDl.getTerritory().getId();
                    colorId = lDl.getColorCode().getDbId();
                    if (lDl.getFromExercise() != null) {
                        fromExerciseId = lDl.getFromExercise().getId();
                    }
                    if (lDl.getToExercise() != null) {
                        toExerciseId = lDl.getToExercise().getId();
                    }
                    if (lDl.getCompetence() != null) {
                        competenceId = lDl.getCompetence().getId();
                    }
                }
            }
        }
    }


    /**
     * called directly from the jsp
     *
     * @return the list of user's selected territories
     */
    public Map<String, String> getSelectedTerritories() {
        // init result
        Map<String, String> lRes = new TreeMap<>();

        // add user's reference territory

        if (userSelections != null && userSelections.size() > 0) {

            for (TerritoryDto lTerritory : userSelections) {
                boolean lAddTerritory = true;

                // do not add general territory -> only for admin
                if (lTerritory.isSuperVirtual() && (!getCurrentUser().isGeneralAdmin())) {
                    lAddTerritory = false;
                }

                // add ?
                if (lAddTerritory) {
                    lRes.put(lTerritory.getId().toString(), lTerritory.getName());
                }
            }
        }

        // result
        return lRes;
    }


    /**
     * check user's selections and fill corresponding list
     *
     * @throws SispeaException
     */
    protected void checkUsersSelections() throws SispeaException {

        // connected ?
        UserDto lUser = getCurrentUser();
        if (lUser == null) {
            // not connected -> search in session
            userSelections = getSispeaSession().getSelectedTerritories();
        } else {
            // connected -> search in DB
            userSelections = referentialService.readUserSelections(lUser);

            // current user's territory ?
            if (lUser.getTerritory() != null) {

                // if current user territory not in list -> add it
                if (!userSelections.contains(lUser.getTerritory())) {
                    // if territory is the France then only admin can have it
                    if (lUser.getTerritory().isSuperVirtual()) {
                        if (lUser.isGeneralAdmin()) {
                            userSelections.add(lUser.getTerritory());
                        }
                    } else {
                        userSelections.add(lUser.getTerritory());
                    }
                }
            }
        }

        // sort list
        Collections.sort(userSelections);
    }


    /**
     * @return the list of colors available
     */
    public Map<Integer, String> getListColorsAvailable() {
        // init result
        Map<Integer, String> lRes = new TreeMap<>();

        for (DataListColorType color : DataListColorType.values()) {
            //if (!isColorAlreadyUsed(color) || (currentList != null && currentList.getColorCode().equals(color))) {
            lRes.put(color.getDbId(), getText(color.getLabelKey()));
            //}
        }

        // result
        return lRes;
    }

    /**
     * @param pColorId
     * @return the filename from the Id
     */
    public String getColorFilenameFromId(Integer pColorId) {
        String lRes = "";

        for (DataListColorType color : DataListColorType.values()) {
            if (Objects.equals(color.getDbId(), pColorId)) {
                lRes = color.getFileName();
            }
        }
        return lRes;
    }


    /*
     * not used anymore as we don't limit available colors to unused ones
    private boolean isColorAlreadyUsed(DataListColorType pColor) {

        if(lists != null && lists.size() > 0) {
            for (Iterator<DataList> iterator = lists.iterator(); iterator.hasNext();) {
                DataList lDataList = (DataList) iterator.next();

                if(lDataList.getColorCode().equals(pColor)){
                    return true;
                }
            }
        }

        return false;
    }
    */


    /**
     * @param pIps
     * @param pVps
     * @return the IPs and VPs as a string separated by semi columns
     */
    public String getDataAsString(List<IndicatorDto> pIps, List<VPDto> pVps, Boolean pReadable) {
        String lRes = "";

        Collections.sort(pIps);
        Collections.sort(pVps);

        for (IndicatorDto lIp : pIps) {
            lRes += lIp.getCode() + ";";
        }

        if (pVps.size() > 0) {
            for (Iterator<VPDto> iterator = pVps.iterator(); iterator.hasNext(); ) {
                VPDto lPv = iterator.next();

                lRes += lPv.getCode();

                if (iterator.hasNext()) {
                    if (pReadable != null && pReadable.equals(Boolean.TRUE)) {
                        lRes += "; ";
                    } else {
                        lRes += ";";
                    }
                }
            }
        } else {
            // remove last semi column
            lRes = lRes.substring(0, lRes.length() - 1);
        }


        return lRes;
    }

    /**
     * @param pReferential
     * @param pCode
     * @return an IP from its code
     */
    protected IndicatorDto getIndicatorByCode(List<IndicatorDto> pReferential, String pCode) {

        for (IndicatorDto lIndicator : pReferential) {
            if (lIndicator.getCode().equals(pCode)) {
                return lIndicator;
            }
        }
        return null;
    }

    /**
     * @param pReferential
     * @param pCode
     * @return a PV from its code
     */
    protected VPDto getPvByCode(List<VPDto> pReferential, String pCode) {

        for (VPDto lPv : pReferential) {
            if (lPv.getCode().equals(pCode)) {
                return lPv;
            }
        }
        return null;
    }

    /**
     * @param pReferential
     * @param pId
     * @return a territory from its Id
     */
    protected TerritoryDto getTerritoryById(List<TerritoryDto> pReferential, Integer pId) {

        for (TerritoryDto lTerritory : pReferential) {
            if (lTerritory.getId().equals(pId)) {
                return lTerritory;
            }
        }
        return null;
    }

    public Integer getBackPageId() {
        return BackType.dataLists.getId();
    }

    public String getIndicators() {
        return indicators;
    }

    public void setIndicators(String pIndicators) {
        indicators = pIndicators;
    }

    public List<TerritoryDto> getUserSelections() {
        return userSelections;
    }

    public void setUserSelections(List<TerritoryDto> pUserSelections) {
        userSelections = pUserSelections;
    }

    public List<DataListDto> getLists() {
        return lists;
    }

    public void setLists(List<DataListDto> pLists) {
        lists = pLists;
    }

    public DataListDto getCurrentList() {
        return currentList;
    }

    public void setCurrentList(DataListDto pCurrentList) {
        currentList = pCurrentList;
    }

    public Integer getSelectedListId() {
        return selectedListId;
    }

    public void setSelectedListId(Integer pSelectedListId) {
        selectedListId = pSelectedListId;
    }

    public Integer getTerritoryId() {
        return territoryId;
    }

    public void setTerritoryId(Integer pTerritoryId) {
        territoryId = pTerritoryId;
    }

    public Integer getSelectionId() {
        return selectionId;
    }

    public void setSelectionId(Integer pSelectionId) {
        selectionId = pSelectionId;
    }

    public Integer getColorId() {
        return colorId;
    }

    public void setColorId(Integer pColorId) {
        colorId = pColorId;
    }

    public Integer getFromExerciseId() {
        return fromExerciseId;
    }

    public void setFromExerciseId(Integer fromExerciseId) {
        this.fromExerciseId = fromExerciseId;
    }

    public Integer getToExerciseId() {
        return toExerciseId;
    }

    public void setToExerciseId(Integer toExerciseId) {
        this.toExerciseId = toExerciseId;
    }

    public Integer getCompetenceId() {
        return competenceId;
    }

    public void setCompetenceId(Integer competenceId) {
        this.competenceId = competenceId;
    }

}
