/*
 * #%L
 * MS-Access Importer
 * 
 * $Id: AbstractAccessDataSource.java 1301 2011-02-14 18:45:59Z chemit $
 * $HeadURL: https://svn.mpl.ird.fr/osiris/observe/msaccess-importer/tags/msaccess-importer-1.2/src/main/java/fr/ird/msaccess/importer/AbstractAccessDataSource.java $
 * %%
 * Copyright (C) 2010 IRD, Codelutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package fr.ird.msaccess.importer;

import com.healthmarketscience.jackcess.Column;
import com.healthmarketscience.jackcess.Database;
import com.healthmarketscience.jackcess.Table;
import org.apache.commons.collections.primitives.ArrayIntList;
import org.apache.commons.collections.primitives.IntList;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityEnum;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

/**
 * Une data source sur une base access
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public abstract class AbstractAccessDataSource<T extends TopiaEntityEnum, M extends AbstractAccessEntityMeta<T>> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractAccessDataSource.class);

    /** La liste des méta données reconnues pour la base ouverte. */
    protected M[] metas;

    /** le cache des données (on lie une seule fois les tables) */
    protected Map<M, Map<String, Object>[]> cache;

    protected Map<String, Set<String>> tableColumns;

    protected Map<String, Set<String>> unusedTableColumns;

    protected final Class<M> metaClass;

    protected final Class<? extends AccessEntityMetaProvider<T, M>> providerClass;

    public enum DataSourceState {
        INIT,
        LOAD
    }

    protected DataSourceState state;

    protected final File dbFile;

    protected boolean isInit() {
        return state != null && state.ordinal() >= DataSourceState.INIT.ordinal();
    }

    protected boolean isLoad() {
        return state != null && state.ordinal() >= DataSourceState.LOAD.ordinal();
    }

    protected void checkIfInit() {
        if (!isInit()) {
            throw new IllegalStateException("Data source " + dbFile + " was not init!");
        }
    }

    protected void checkIfLoad() {
        if (!isLoad()) {
            throw new IllegalStateException("Data source " + dbFile + " was not loaded!");
        }
    }

    @SuppressWarnings({"unchecked"})
    public AbstractAccessDataSource(Class<? extends AbstractAccessEntityMeta> metaClass,
                                    Class<? extends AccessEntityMetaProvider<T, M>> providerClass,
                                    File dbFile) {

        if (metaClass == null) {
            throw new NullPointerException("metaClass parameter can not be null");
        }
        this.metaClass = (Class<M>) metaClass;

        if (providerClass == null) {
            throw new NullPointerException("providerClass parameter can not be null");
        }
        this.providerClass = providerClass;

        if (dbFile == null) {
            throw new NullPointerException("dbFile parameter can not be null");
        }

        if (!dbFile.exists()) {
            throw new IllegalArgumentException("dbFile " + dbFile + " does not exists");
        }

        this.dbFile = dbFile;
    }

    protected abstract M[] newMetaArray(Collection<M> iterable);

    protected abstract void onTableMissing(M meta);

    protected abstract void onPropertyMissing(M meta,
                                              String property,
                                              String column);

    protected abstract void onPKeyMissing(M meta, String pkey);

    public void init() throws Exception {
        if (isInit()) {
            if (log.isWarnEnabled()) {
                log.warn("Datasource " + dbFile + " was already init, will skip it.");
            }
            return;
        }

        // instanciate provider of metas
        AccessEntityMetaProvider<T, M> metaProvider = providerClass.newInstance();

        // get metas from the provider
        Set<M> metas = metaProvider.getMetas();

        // get a new connexion to access db
        Database connexion = getConnexion();

        try {

            // get the list of table names required (says all the one in meta)

            Set<String> requiredTables = new HashSet<String>();
            for (M meta : metas) {
                requiredTables.add(meta.getTableName());
            }

            tableColumns = new TreeMap<String, Set<String>>();
            unusedTableColumns = new TreeMap<String, Set<String>>();

            // scan table
            for (Table table : connexion) {
                String tableName = table.getName();
                Map<String, Set<String>> result;
                if (requiredTables.contains(tableName)) {
                    result = tableColumns;
                } else {
                    result = unusedTableColumns;
                }
                Set<String> columnNames = new TreeSet<String>();
                List<Column> columns = table.getColumns();
                for (Column column : columns) {
                    columnNames.add(column.getName());
                }
                if (log.isDebugEnabled()) {
                    log.debug("detected ms-access table " + tableName + " with columns " + columnNames);
                }

                Set<String> value = Collections.unmodifiableSet(columnNames);

                result.put(tableName, value);
            }

            // scan metas and match them to detected structure
            for (M meta : metas) {
                if (log.isInfoEnabled()) {
                    log.info("Will init meta : " + meta);
                }
                initMeta(meta);
            }

            this.metas = newMetaArray(metas);

            // coming here make the init was safely done
            state = DataSourceState.INIT;

        } finally {

            closeConnexion(connexion);
        }
    }

    public void load() throws IOException {
        if (isLoad()) {
            if (log.isWarnEnabled()) {
                log.warn("Datasource " + dbFile +
                         " was already loaded, will skip it.");
            }
            return;
        }

        Database connexion = getConnexion();

        try {
            cache = new HashMap<M, Map<String, Object>[]>();

            for (M meta : metas) {
                Map<String, Object>[] data = loadTable(meta, connexion);
                cache.put(meta, data);
            }

            // coming here make the loading was safely done
            state = DataSourceState.LOAD;

        } finally {

            closeConnexion(connexion);
        }
    }

    protected final M initMeta(M meta) throws Exception {

        if (log.isDebugEnabled()) {
            log.debug("check entity :\n" + meta);
        }

        // check table name
        String tableName = meta.getTableName();

        if (!tableColumns.containsKey(tableName)) {

            onTableMissing(meta);

            return meta;
        }

        Set<String> columns = tableColumns.get(tableName);

        List<AbstractAccessEntityMeta.PropertyMapping> mapping =
                new ArrayList<AbstractAccessEntityMeta.PropertyMapping>();

        // check property mapping
        for (AbstractAccessEntityMeta.PropertyMapping entry :
                meta.getPropertyMapping()) {
            String property = entry.getProperty();
            String column = entry.getColumn();
            if (columns.contains(column)) {

                mapping.add(entry);
            } else {

                onPropertyMissing(meta, property, column);
            }
        }

        // changement du mapping
        meta.setPropertyMapping(mapping.toArray(
                new AbstractAccessEntityMeta.PropertyMapping[mapping.size()]));

        // check pkeys
        for (String pkey : meta.getPkeys()) {
            if (!columns.contains(pkey)) {

                onPKeyMissing(meta, pkey);
            }
        }

        return meta;
    }

    @SuppressWarnings({"unchecked"})
    public final Map<String, Object>[] loadTable(M meta,
                                                 Database connexion) throws IOException {

        String tableName = meta.getTableName();

        // load table from access
        Table table = connexion.getTable(tableName);
        int rowCount = table.getRowCount();
        if (log.isDebugEnabled()) {
            log.debug("Load table " + tableName + " with " + rowCount + " row(s).");
        }
        Map<String, Object>[] result = new Map[rowCount];
        int i = 0;
        Iterator<Map<String, Object>> itr = table.iterator();
        IntList errors = new ArrayIntList();
        while (i < rowCount && itr.hasNext()) {
            Map<String, Object> map = null;
            try {
                map = itr.next();
            } catch (Exception e) {
                if (log.isErrorEnabled()) {
                    log.error("Could not read row " + i + " of table " + tableName, e);
                }
                errors.add(i);
            }
            result[i++] = map;
        }
        if (!errors.isEmpty()) {
            if (log.isWarnEnabled()) {
                log.warn("[" + meta.getType() + "] Could not load " + errors.size() + " row(s) : " + errors);
            }
        }
        meta.setErrorRows(errors.toArray(new int[errors.size()]));

        return result;
    }

    protected Database getConnexion() throws IOException {
        Database database = Database.open(dbFile);

        if (database == null) {
            throw new IllegalStateException("Required an access connexion");
        }

        return database;
    }

    protected void closeConnexion(Database connexion) {
        if (connexion != null && connexion.getPageChannel() != null && connexion.getPageChannel().isOpen()) {
            try {
                connexion.close();
            } catch (Exception e) {
                if (log.isErrorEnabled()) {
                    log.error("Could not close access connexion", e);
                }
            }
        }
    }

    @Override
    protected void finalize() throws Throwable {
        destroy();
        super.finalize();
    }

    public final void destroy() {

        if (isInit()) {
            tableColumns.clear();
            unusedTableColumns.clear();
            for (M meta : metas) {
                meta.clear();
            }
        }

        if (isLoad()) {
            for (M m : cache.keySet()) {
                Map<String, Object>[] maps = cache.get(m);
                for (Map<String, Object> map : maps) {
                    if (map != null) {
                        map.clear();
                    }
                }
            }
            cache.clear();
        }
        metas = null;
        tableColumns = null;
        unusedTableColumns = null;
        cache = null;
        state = null;
    }

    public final Set<String> getTableNames() {
        checkIfInit();
        return tableColumns.keySet();
    }

    public final Set<String> getUnusedTableNames() {
        checkIfInit();
        return unusedTableColumns.keySet();
    }

    public Set<String> getTableColumns(String tableName) {
        checkIfInit();
        return tableColumns.get(tableName);
    }

    public Set<String> getUnusedTableColumns(String tableName) {
        checkIfInit();
        return unusedTableColumns.get(tableName);
    }

    public final boolean hasError() {
        checkIfInit();
        for (M meta : metas) {
            if (meta.hasError()) {
                return true;
            }
        }
        return false;
    }

    public final boolean hasWarning() {
        checkIfInit();
        for (M meta : metas) {
            if (meta.hasWarning()) {
                return true;
            }
        }
        return false;
    }

    public final M[] getMetaWithError() {
        checkIfInit();
        List<M> result = new ArrayList<M>();
        for (M meta : metas) {
            if (meta.hasError()) {
                result.add(meta);
            }
        }
        return newMetaArray(result);
    }

    public final M[] getMetaWithWarning() {
        checkIfInit();
        List<M> result = new ArrayList<M>();
        for (M meta : metas) {
            if (meta.hasWarning()) {
                result.add(meta);
            }
        }
        return newMetaArray(result);
    }

    public M[] getMetas() {
        checkIfInit();
        return metas;
    }

    public M[] getMetaForType(Class<?> type) {
        checkIfInit();
        List<M> result = new ArrayList<M>();
        for (M meta : metas) {
            if (type.isAssignableFrom(meta.getType().getContract())) {
                result.add(meta);
            }
        }
        return newMetaArray(result);
    }

    public M getMeta(T type) {
        checkIfInit();
        for (M meta : metas) {
            if (type.equals(meta.getType())) {
                return meta;
            }
        }
        return null;
    }

    @SuppressWarnings({"unchecked"})
    public final Map<String, Object>[] getTableData(M meta) {
        checkIfLoad();
        Map<String, Object>[] result = cache.get(meta);
        return result;
    }

    public final Map<String, Object> getTableDataRow(M meta, int row) {
        checkIfLoad();
        Map<String, Object>[] result = getTableData(meta);
        return result[row];
    }

    @SuppressWarnings({"unchecked"})
    public final <E extends TopiaEntity> E[] loadAssociation(M meta,
                                                             M container,
                                                             Object[] pkeys) throws Exception {
        checkIfLoad();
        List<E> result = new ArrayList<E>();
        int row = 0;
        Map<String, Object>[] data = getTableData(meta);
        for (Map<String, Object> map : data) {
            if (map == null) {
                row++;
                continue;
            }
            Object[] pkey = getPkey(container.getPkeys(), map);
            if (Arrays.equals(pkeys, pkey)) {
                E e = (E) meta.newEntity(row, getPkey(meta, map));
                result.add(e);
            }
            row++;
        }
        E[] r = (E[]) Array.newInstance(meta.getType().getContract(), result.size());
        int i = 0;
        for (E e : result) {
            r[i++] = e;
        }
        return r;
    }

    @SuppressWarnings({"unchecked"})
    public final <E extends TopiaEntity> List<E> loadEntities(M meta) {
        checkIfLoad();
        List<E> result = new ArrayList<E>();

        int row = 0;
        for (Map<String, Object> map : getTableData(meta)) {
            Object[] pkey = getPkey(meta, map);
            E e = (E) meta.newEntity(row++, pkey);
            result.add(e);
        }
        return result;
    }

    public Object[] getPkey(M meta, Map<String, Object> map) {
        return getPkey(meta.getPkeys(), map);
    }

    public Object[] getPkey(List<String> keys, Map<String, Object> map) {
        Object[] result = new Object[keys.size()];
        int i = 0;
        for (String key : keys) {
            Object o = map.get(key);
            result[i++] = o;
        }
        return result;
    }

}
