/*
 * #%L
 * Refcomp :: GWT client
 * 
 * $Id: SkillsTableScreen.java 79 2011-01-27 17:45:56Z jcouteau $
 * $HeadURL: svn+ssh://jcouteau@scm.mulcyber.toulouse.inra.fr/svnroot/refcomp/tags/refcomp-0.1/refcomp-client/src/main/java/fr/inra/refcomp/client/agent/SkillsTableScreen.java $
 * %%
 * Copyright (C) 2010 - 2011 Inra
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.inra.refcomp.client.agent;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.uibinder.client.UiBinder;
import com.google.gwt.uibinder.client.UiField;
import com.google.gwt.uibinder.client.UiHandler;
import com.google.gwt.uibinder.client.UiTemplate;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HTMLPanel;
import com.google.gwt.user.client.ui.InlineLabel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.ListBox;
import com.google.gwt.user.client.ui.Panel;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.TextBox;
import fr.inra.refcomp.client.constants.Menu;
import fr.inra.refcomp.client.constants.RefcompCallBack;
import fr.inra.refcomp.client.constants.RefcompMessages;
import fr.inra.refcomp.client.services.StorageService;
import fr.inra.refcomp.client.services.StorageServiceAsync;
import fr.inra.refcomp.client.table.DefaultTableModel;
import fr.inra.refcomp.client.table.FlexTableWithModel;
import fr.inra.refcomp.client.table.GWTTableModel;
import fr.inra.refcomp.entities.Agent;
import fr.inra.refcomp.entities.AgentSkill;
import fr.inra.refcomp.entities.Cati;
import fr.inra.refcomp.entities.Department;
import fr.inra.refcomp.entities.EntitiesList;
import fr.inra.refcomp.entities.Frequency;
import fr.inra.refcomp.entities.Skill;
import fr.inra.refcomp.entities.Unit;
import fr.inra.refcomp.entities.User;
import fr.inra.refcomp.services.InvalidEntityException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import org.nuiton.wikitty.entities.BusinessEntity;
import org.nuiton.wikitty.entities.WikittyAuthorisation;

/**
 * User: couteau
 * Date: 31/12/10
 */
public class SkillsTableScreen extends Composite {

    public static int TECH_SKILL_COL = 0;
    public static int TECH_SKILL_EDIT_COL = 1;
    public static int DOMAIN_SKILL_COL = 2;
    public static int DOMAIN_SKILL_EDIT_COL = 3;
    public static int FREQUENCY_COL = 4;
    public static int COMMENT_COL = 5;
    public static int RIGHT_COL = 6;
    public static int SHOW_PATH_COL = 7;
    public static int DELETE_SKILL_COL = 8;
    public static int DOMAIN_SKILL_ID_COL = 9;
    public static int TECH_SKILL_ID_COL = 10;
    public static int FREQUENCY_ID_COL = 11;
    public static int SKILL_ID_COL = 12;

    private final StorageServiceAsync service = GWT
            .create(StorageService.class);

    @UiTemplate("SkillsTableScreen.ui.xml")
    interface MyUiBinder extends UiBinder<Panel, SkillsTableScreen> {
    }

    private static final MyUiBinder binder = GWT.create(MyUiBinder.class);

    private RefcompMessages messages = GWT.create(RefcompMessages.class);

    protected List<Frequency> cachedFrequencies = new ArrayList<Frequency>();

    protected List<AgentSkill> skillsList = new ArrayList<AgentSkill>();

    @UiField
    FlexTableWithModel skillsTable;

    @UiField
    InlineLabel agentName;

    @UiField
    Button addButton;

    @UiField
    Button cancelButton;

    @UiField
    Button saveButton;

    @UiField
    InlineLabel departmentName;

    @UiField
    InlineLabel catiName;

    @UiField
    InlineLabel unitsName;

    @UiField
    InlineLabel address;

    @UiField
    Anchor email;

    @UiField
    HTMLPanel agentInfoPanel;

    @UiField
    HTMLPanel errorPanel;

    @UiField
    Label errorLabel;

    /**
     * Boolean to know if the screen is in edition state or only reading.
     */
    protected Boolean edition = false;

    /**
     * Manage the add button. Perform operation only when not in edition mode.
     * Button should not be visible otherwise.
     *
     * @param event the click event.
     */
    @UiHandler("addButton")
    void doClickAdd(ClickEvent event) {
        if (edition) {
            service.addSkill(new RefcompCallBack<AgentSkill>() {
                @Override
                public void onSuccess(AgentSkill result) {
                    skillsList.add(result);
                    displayEditionSkill(result, new EntitiesList());
                }
            });
        }
    }

    @UiHandler("saveButton")
    void doClickSave(ClickEvent event) {
        if (edition) {

            errorPanel.setVisible(false);

            List<AgentSkill> skills = new ArrayList<AgentSkill>();

            //need to be final to be accessed in inner class
            final List<String> privateSkills = new ArrayList<String>();
            final List<String> publicSkills = new ArrayList<String>();

            for (int i = 1; i < skillsTable.getRowCount(); i++) {
                String id = skillsTable.getText(i, SKILL_ID_COL);
                AgentSkill editedSkill = getSkill(id);

                String technicalSkillId = skillsTable.getText(i, TECH_SKILL_ID_COL);
                //String domainSkillId = skillsTable.getText(i, DOMAIN_SKILL_ID_COL);
                ListBox frequencyBox = (ListBox)skillsTable.getWidget(i, FREQUENCY_COL);
                String frequencyId = frequencyBox.getValue(frequencyBox.getSelectedIndex());
                TextBox commentBox = (TextBox)skillsTable.getWidget(i, COMMENT_COL);
                String comment = commentBox.getText();

                editedSkill.setTechnicalSkill(technicalSkillId);
                //editedSkill.setDomainSkill(domainSkillId);
                editedSkill.setFrequency(frequencyId);
                editedSkill.setComment(comment);

                skills.add(editedSkill);

                ListBox rightsBox = (ListBox)skillsTable.getWidget(i,RIGHT_COL);
                String right = rightsBox.getValue(rightsBox.getSelectedIndex());
                if ("private".equals(right)) {
                    privateSkills.add(id);
                } else {
                    publicSkills.add(id);
                }
            }

            //save everything
            service.saveMySkills(skills, new RefcompCallBack<EntitiesList>() {
                @Override
                public void onSuccess(EntitiesList result) {
                    service.setPrivateSkills(privateSkills,
                            new RefcompCallBack<Void>() {
                                @Override
                                public void onSuccess(Void result) {
                                    service.setPublicSkills(publicSkills,
                                            new RefcompCallBack<Void>() {
                                                @Override
                                                public void onSuccess(Void result) {
                                                    //Do nothing
                                                }
                                            });
                                }
                            });

                    List<BusinessEntity> entities = result.getEntities();
                    skillsList = new ArrayList<AgentSkill>();
                    for (BusinessEntity entity : entities) {
                        AgentSkill skill = (AgentSkill) entity;
                        skillsList.add(skill);
                    }
                }

                @Override
                public void onFailure(Throwable caught) {
                    if (caught instanceof InvalidEntityException) {
                        errorPanel.setVisible(true);
                        errorLabel.setText(caught.getMessage());
                    } else {
                        Menu.doClickLogout(null);
                    }
                }
            });
        }
    }

    @UiHandler("cancelButton")
    void doClickCancel(ClickEvent event) {
        if (edition) {
            errorPanel.setVisible(false);
            AgentUtil.checkAgent();
            initEditionMode();
        }
    }

    public SkillsTableScreen() {
        this(false);
    }

    /**
     * Init the screen. If edition mode, the screen is initialized and filled in
     * with logged in user info. If reading mode, init the screen but let the
     * table empty as don't know which user to display.
     *
     * @param edition true for edition mode, false for reading
     */
    public SkillsTableScreen(Boolean edition) {

        this.edition = edition;

        if (edition) {
            AgentUtil.checkAgent();
        }

        init();

        if (edition) {
            initEditionMode();
        }

    }

    public SkillsTableScreen(String agentId) {
        init();

        //Get the user name and display it
        service.getAgentAndDependencies(agentId, new RefcompCallBack<EntitiesList>() {
            @Override
            public void onSuccess(EntitiesList result) {

                List<BusinessEntity> entities = result.getEntities();

                if (!entities.isEmpty()) {
                    Agent agent = (Agent) result.getEntities().get(0);

                    agentName.setText(agent.getFirstName() + " " + agent.getLastName());
                    email.setHref("mailto:" + agent.getEmail());
                    email.setText(agent.getEmail());
                    address.setText(agent.getAddress());

                    Department department = (Department)result.getDependency(agent.getDepartment());
                    if (department != null) {
                        String text = department.getName();
                        String fullName = department.getFullName();
                        if (fullName != null && !fullName.isEmpty()) {
                            text += " - " + fullName;
                        }
                        departmentName.setText(text);
                    }

                    Cati cati = (Cati)result.getDependency(agent.getCati());
                    if (cati != null) {
                        String text = cati.getName();
                        String fullName = cati.getFullName();
                        if (fullName != null && !fullName.isEmpty()) {
                            text += " - " + fullName;
                        }
                        catiName.setText(text);
                    }

                    Set<String> unitsIds = agent.getUnit();
                    String units = "";
                    for (String id:unitsIds) {
                        Unit unit = (Unit)result.getDependency(id);
                        if (unit != null) {
                            String text = unit.getName();
                            String fullName = unit.getFullName();
                            if (fullName != null && !fullName.isEmpty()) {
                                text += " - " + fullName;
                            }
                            units += ", " + text;
                            if (address.getText().equals("")){
                                address.setText(unit.getAddress());
                            }
                        }
                    }
                    units = units.substring(1);
                    unitsName.setText(units);
                }
            }
        });

        //Get the skills of the logged in user and display them
        AsyncCallback<EntitiesList> callback = new RefcompCallBack<EntitiesList>() {
            public void onSuccess(EntitiesList result) {
                initTable();

                List<BusinessEntity> entities = result.getEntities();

                skillsList = new ArrayList<AgentSkill>();

                for (BusinessEntity entity : entities) {
                    AgentSkill skill = (AgentSkill) entity;
                    skillsList.add(skill);
                    displayReadingSkill(skill, result);
                }

                setReadingMode();
            }
        };

        service.getSkills(agentId, callback);
    }

    public void initEditionMode(){
        //Get the logged in user and display its name
        service.getLoggedInUser(new RefcompCallBack<User>() {

            @Override
            public void onSuccess(User result) {
                agentName.setText(result.getFirstName() + " " + result.getLastName());
            }
        });

        //Get the skills of the logged in user and display them
        AsyncCallback<EntitiesList> callback = new RefcompCallBack<EntitiesList>() {
            public void onSuccess(EntitiesList result) {

                initTable();

                List<BusinessEntity> entities = result.getEntities();

                skillsList = new ArrayList<AgentSkill>();

                for (BusinessEntity entity : entities) {
                    AgentSkill skill = (AgentSkill)entity;
                    skillsList.add(skill);
                    displayEditionSkill(skill, result);
                }

                setEditionMode();
            }
        };

        service.getSkills(callback);
    }

    /**
     * Add a skill in edition mode
     * @param skill the skill to add
     * @param dependencies the EntitiesList containing the dependencies
     */
    protected void displayEditionSkill(AgentSkill skill,
                                       EntitiesList dependencies) {
        GWTTableModel model = skillsTable.getModel();
        final int row = model.getRowCount();

        //Tech skill
        Skill techSkill = (Skill) dependencies.getDependency(
                skill.getTechnicalSkill());
        if (techSkill != null) {
            model.setValueAt(techSkill.getName(), row, TECH_SKILL_COL);
            model.setValueAt(techSkill.getWikittyId(), row, TECH_SKILL_ID_COL);
        }

        //Domain skill
        //Useless now, commented so that it can be re-activated easily jcouteau 22/01/2011
        /*Skill domainSkill = (Skill) dependencies.getDependency(
                skill.getDomainSkill());
        if (domainSkill != null) {
            model.setValueAt(domainSkill.getName(), row, DOMAIN_SKILL_COL);
            model.setValueAt(domainSkill.getWikittyId(), row,
                    DOMAIN_SKILL_ID_COL);
        }*/

        //Edit tech skill button
        IdButton editTechnicalSkillButton = new IdButton(messages.edit(),
                skill.getWikittyId());
        editTechnicalSkillButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {

                PopupPanel popup = new PopupPanel(false);
                popup.setWidget(new SkillsTreeSelection(SkillsTableScreen.this,
                        row, true));
                popup.center();
                popup.show();
            }
        });
        model.setValueAt(editTechnicalSkillButton, row, TECH_SKILL_EDIT_COL);

        //Edit domain skill button
        //Useless now, commented so that it can be re-activated easily jcouteau 22/01/2011
        /*IdButton editDomainSkillButton = new IdButton(messages.edit(),
                skill.getWikittyId());
        editDomainSkillButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                PopupPanel popup = new PopupPanel(false);
                popup.setWidget(new SkillsTreeSelection(SkillsTableScreen.this,
                        row, false));
                popup.center();
                popup.show();
            }
        });
        model.setValueAt(editDomainSkillButton, row, DOMAIN_SKILL_EDIT_COL);*/

        //Frequency
        ListBox frequencyBox = new ListBox();
        for (Frequency freq : cachedFrequencies) {
            frequencyBox.addItem(freq.getName(), freq.getWikittyId());
        }
        for (int i = 0; i < frequencyBox.getItemCount(); i++) {
            String value = frequencyBox.getValue(i);
            if (value.equals(skill.getFrequency())) {
                frequencyBox.setSelectedIndex(i);
            }
        }
        model.setValueAt(frequencyBox, row, FREQUENCY_COL);
        model.setValueAt(skill.getFrequency(), row, FREQUENCY_ID_COL);

        //right
        WikittyAuthorisation right = (WikittyAuthorisation) dependencies.getDependency(skill.getWikittyId());
        String rightString = "public";
        if (right != null && right.getReader() != null && !right.getReader().isEmpty()) {
            rightString = "private";
        }
        ListBox rightsBox = new ListBox();
        rightsBox.addItem(messages.publicRight(), "public");
        rightsBox.addItem(messages.privateRight(), "private");
        for (int i = 0; i < rightsBox.getItemCount(); i++) {
            String value = rightsBox.getValue(i);
            if (value.equals(rightString)) {
                rightsBox.setSelectedIndex(i);
            }
        }
        model.setValueAt(rightsBox, row, RIGHT_COL);

        //comment
        TextBox commentBox = new TextBox();
        commentBox.setText(skill.getComment());
        model.setValueAt(commentBox, row, COMMENT_COL);

        //Delete button
        IdButton deleteSkillButton = new IdButton(messages.delete(),
                skill.getWikittyId());
        deleteSkillButton.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {

                Boolean validate = Window.confirm(messages.sure());

                if (validate) {

                    final String id = ((IdButton) event.getSource()).getId();

                    service.removeSkill(id, new RefcompCallBack<Void>() {
                        @Override
                        public void onSuccess(Void result) {
                            removeSkill(id);
                        }
                    });
                }
            }
        });
        model.setValueAt(deleteSkillButton, row, DELETE_SKILL_COL);

        //Id
        model.setValueAt(skill.getWikittyId(), row, SKILL_ID_COL);

        //pathButton
        IdButton pathButton = new IdButton(messages.path(),
                skill.getTechnicalSkill());
        pathButton.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {

                final String id = ((IdButton) event.getSource()).getId();

                service.getPath(id, new RefcompCallBack<String>() {
                    @Override
                    public void onSuccess(String result) {
                        PopupPanel popup = new PopupPanel(false);
                        popup.setWidget(new PathPanel(result));
                        popup.center();
                        popup.show();
                    }
                });
            }
        });
        model.setValueAt(pathButton, row, SHOW_PATH_COL);

        //Show/hide the proper columns
        setEditionMode();
    }

    /**
     * Add a skill in reading mode
     * @param skill the skill to add
     * @param dependencies the dependencies for displaying
     */
    protected void displayReadingSkill(AgentSkill skill,
                                       EntitiesList dependencies) {

        GWTTableModel model = skillsTable.getModel();
        final int row = model.getRowCount();

        //Tech skill
        Skill techSkill = (Skill) dependencies.getDependency(
                skill.getTechnicalSkill());
        if (techSkill != null) {
            model.setValueAt(techSkill.getName(), row, TECH_SKILL_COL);
        }

        //Domain skill
        //Useless now, commented so that it can be re-activated easily jcouteau 22/01/2011
        /*Skill domainSkill = (Skill) dependencies.getDependency(
                skill.getDomainSkill());
        if (domainSkill != null) {
            model.setValueAt(domainSkill.getName(), row, DOMAIN_SKILL_COL);
        }*/

        //Frequency
        Frequency frequency = (Frequency) dependencies.getDependency(
                skill.getFrequency());
        if (frequency != null) {
            model.setValueAt(frequency.getName(), row, FREQUENCY_COL);
        }

        //comment
        model.setValueAt(skill.getComment(), row, COMMENT_COL);

        //Show/hide the proper columns
        setReadingMode();

    }

    /**
     * Display/Hide the right columns for edition mode
     */
    protected void setEditionMode() {
        setColumnVisible(TECH_SKILL_ID_COL, false);
        setColumnVisible(DOMAIN_SKILL_ID_COL, false);
        setColumnVisible(DOMAIN_SKILL_COL, false);
        setColumnVisible(DOMAIN_SKILL_EDIT_COL, false);
        setColumnVisible(SKILL_ID_COL, false);
        setColumnVisible(FREQUENCY_ID_COL, false);

        addButton.setVisible(true);
        cancelButton.setVisible(true);
        saveButton.setVisible(true);
        agentInfoPanel.setVisible(false);
    }

    /**
     * Display/hide the right columns for reading mode
     */
    protected void setReadingMode(){
        setColumnVisible(TECH_SKILL_ID_COL, false);
        setColumnVisible(DOMAIN_SKILL_ID_COL, false);
        setColumnVisible(DOMAIN_SKILL_COL, false);
        setColumnVisible(SKILL_ID_COL, false);
        setColumnVisible(DOMAIN_SKILL_EDIT_COL, false);
        setColumnVisible(TECH_SKILL_EDIT_COL, false);
        setColumnVisible(FREQUENCY_ID_COL, false);
        setColumnVisible(RIGHT_COL, false);

        addButton.setVisible(false);
        cancelButton.setVisible(false);
        saveButton.setVisible(false);
        agentInfoPanel.setVisible(true);
    }

    /**
     * Init the table with column headers
     */
    protected void initTable() {
        skillsTable.clear(true);

        Object[][] data = new Object[][]{};

        String[] columnNames = new String[]{
                messages.skill(),
                "",
                messages.linkedSkill(),
                "",
                messages.frequency(),
                messages.comment(),
                messages.right(),
                "",
                "",
                "wikittyId",
                "mainSkillId",
                "secondSkillId",
                "frequencyId"
        };

        GWTTableModel model = new DefaultTableModel(data, columnNames);

        skillsTable.setModel(model);

        skillsTable.setText(0, TECH_SKILL_COL, messages.skill());
        skillsTable.setText(0, DOMAIN_SKILL_COL, messages.linkedSkill());
        skillsTable.setText(0, FREQUENCY_COL, messages.frequency());
        skillsTable.setText(0, COMMENT_COL, messages.comment());
        skillsTable.setText(0, SKILL_ID_COL, "wikittyId");
        skillsTable.setText(0, TECH_SKILL_ID_COL, "mainSkillId");
        skillsTable.setText(0, DOMAIN_SKILL_ID_COL, "secondSkillId");
        skillsTable.setText(0, RIGHT_COL, "Right");

        skillsTable.getRowFormatter().addStyleName(0, "tableHeader");

    }

    /**
     * Show/hide a column
     * @param col the column number
     * @param b true to set visible, false to hide
     */
    protected void setColumnVisible(int col, boolean b) {
        for (int i = 0; i < skillsTable.getRowCount(); i++) {
            skillsTable.getCellFormatter().setVisible(i, col, b);
        }
    }

    /**
     * Init the Composite and the frequencies cache
     */
    protected void init() {
        initWidget(binder.createAndBindUi(this));

        //init the cached frequencies
        service.getFrequencies(new RefcompCallBack<List<Frequency>>() {
            @Override
            public void onSuccess(List<Frequency> result) {
                cachedFrequencies = result;
            }
        });

        errorPanel.setVisible(false);
    }

    protected void setTechnicalSkill(int row, Skill skill){
        if (skill != null) {
            GWTTableModel model = skillsTable.getModel();

            model.setValueAt(skill.getWikittyId(),row, TECH_SKILL_ID_COL);
            model.setValueAt(skill.getName(),row, TECH_SKILL_COL);

            IdButton button = (IdButton)model.getValueAt(SHOW_PATH_COL, row);
            button.setId(skill.getWikittyId());
        }
    }

    /*protected void setDomainSkill(int row, Skill skill){
        if (skill != null) {
            GWTTableModel model = skillsTable.getModel();

            model.setValueAt(skill.getWikittyId(), row, DOMAIN_SKILL_ID_COL);
            model.setValueAt(skill.getName(), row, DOMAIN_SKILL_COL);
        }
    }*/

    protected void removeSkill(String skillId) {
        //Start at 1 to avoid looping on header
        for(int i=1;i< skillsTable.getRowCount();i++){
            String id = skillsTable.getText(i, SKILL_ID_COL);
            if (skillId.equals(id)) {
                skillsTable.removeRow(i);
            }
        }
    }

    protected AgentSkill getSkill(String id) {
        for (AgentSkill skill : skillsList) {
            if (id.equals(skill.getWikittyId())){
                return skill;
            }
        }
        return null;
    }

    /**
     * Button that know the id of the skill it is referring
     */
    protected class IdButton extends Button {

        protected String id;

        public IdButton(String title, String id) {
            super(title);
            this.id = id;
        }

        public String getId() {
            return id;
        }

        public void setId(String id) {
            this.id = id;
        }

    }

}
