package fr.ifremer.wao.web.action;

/*
 * #%L
 * Wao :: Web
 * %%
 * Copyright (C) 2009 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.ImmutableSet;
import fr.ifremer.wao.WaoTechnicalException;
import fr.ifremer.wao.services.service.ImportErrorException;
import fr.ifremer.wao.services.service.administration.ReferentialService;
import fr.ifremer.wao.web.WaoJspActionSupport;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;

public abstract class AbstractImportCsvAction extends WaoJspActionSupport {

    private static final long serialVersionUID = 1L;

    private static final Log log = LogFactory.getLog(AbstractImportCsvAction.class);

    protected static final ImmutableSet<String> CSV_CONTENT_TYPES =
            ImmutableSet.of("text/csv", "text/comma-separated-values", "application/vnd.ms-excel", "text/plain");

    protected transient ReferentialService service;

    protected File csvFile;

    protected String csvFileContentType;

    protected String csvFileFileName;

    public void setCsvFile(File csvFile) {
        this.csvFile = csvFile;
    }

    public void setCsvFileContentType(String csvFileContentType) {
        this.csvFileContentType = csvFileContentType;
    }

    public void setCsvFileFileName(String csvFileFileName) {
        this.csvFileFileName = csvFileFileName;
    }

    public void setService(ReferentialService service) {
        this.service = service;
    }

    protected abstract void importCsv(InputStream csvInputStream) throws ImportErrorException;

    protected abstract String getSuccessMessage();

    @Override
    public String execute() {

        String result = SUCCESS;

        if (CSV_CONTENT_TYPES.contains(csvFileContentType)) {

            InputStream csvInputStream = null;

            try {

                csvInputStream = new FileInputStream(csvFile);

                if (log.isInfoEnabled()) {
                    log.info(getAuthenticatedWaoUser().getWaoUser() + " is doing a csv import in action " + this + " with file " + csvFileFileName);
                }

                importCsv(csvInputStream);

                session.addMessage(getSuccessMessage());

            } catch (FileNotFoundException e) {

                if (log.isErrorEnabled()) {
                    log.error("should never occur", e);
                }

                throw new WaoTechnicalException(e);

            } catch (ImportErrorException e) {

                session.addErrorMessages(e.getMessage(getLocale()));

                result = ERROR;

            } finally {

                IOUtils.closeQuietly(csvInputStream);

            }

        } else {

            if (log.isWarnEnabled()) {
                log.warn("content type " + csvFileContentType + " is not CSV compatible");
            }

            session.addErrorMessages(t("wao.ui.import.csvTypeRequired", csvFileContentType));

            result = ERROR;

        }

        return result;

    }

}
