package fr.ifremer.wao.web;

/*
 * #%L
 * Wao :: Web
 * %%
 * Copyright (C) 2009 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.ImmutableMap;
import fr.ifremer.wao.WaoUtils;
import fr.ifremer.wao.entity.ObsProgram;
import fr.ifremer.wao.services.AuthenticatedWaoUser;
import fr.ifremer.wao.web.action.AbstractSamplingPlanAction;
import org.apache.commons.lang3.time.DateUtils;
import org.nuiton.util.pagination.PaginationResult;

import java.util.Date;

public class WaoJspActionSupport extends WaoActionSupport {

    private static final long serialVersionUID = 1L;

    protected WaoSession getSession() {
        return session;
    }

    public String getInstanceDisclaimer() {
        return applicationConfig.getInstanceDisclaimer();
    }

    public AuthenticatedWaoUser getAuthenticatedWaoUser() {
        return getSession().getAuthenticatedWaoUser();
    }

    public ObsProgram getObsProgram() {
        return getAuthenticatedWaoUser().getObsProgram();
    }

    public boolean isObsMer() {
        return getObsProgram().isObsMer();
    }

    public boolean isObsVente() {
        return getObsProgram().isObsVente();
    }

    public String getText(Enum<?> enumeration) {
        return getText(enumeration.getClass().getSimpleName() + "." + enumeration.name());
    }

    @Override
    public String getText(String aTextName, String defaultValue) {
        String text = super.getText(aTextName, defaultValue);
        if ("invalid.fieldvalue.filter.periodFrom".equals(aTextName)) {
            if (this instanceof AbstractSamplingPlanAction) {
                text = getText("wao.ui.invalidMonthFormat");
            } else {
                text = getText("wao.ui.invalidDateFormat");
            }
        }
        return text;
    }

    public String formatMonth(Date date) {
        return date == null ? "" : WaoUtils.formatMonth(getLocale(), date);
    }

    public String formatDate(Date date) {
        return date == null ? "" : WaoUtils.formatDate(getLocale(), date);
    }

    public String formatDateMonth(Date date) {
        return date == null ? "" : WaoUtils.formatDateMonth(getLocale(), date);
    }

    public String formatDateYear(Date date) {
        return date == null ? "" : WaoUtils.formatDateYear(getLocale(), date);
    }

    public String formatDateTime(Date date) {
        return date == null ? "" : WaoUtils.formatDateTime(getLocale(), date);
    }

    public boolean isGoogleAnalyticsEnabled() {
        return applicationConfig.isGoogleAnalyticsEnabled();
    }

    /**
     * From the full url of the action (a=with all get parameters), obtain the pagination url (ending with parameter
     * {@code parger.pageIndex=}.
     *
     * @param fullUrl the full url
     * @return the pagination url
     */
    public String getPaginationUrl(String fullUrl) {
        // remove  pageIndex parameter
        String paginationUrl = fullUrl.replaceAll("pageNumber=([^&])+[&]*", "");
        // clean &
        paginationUrl = paginationUrl.replaceAll("&amp;&amp;", "&amp;");
        // add [?|&]pageIndex=
        if (!paginationUrl.contains("?")) {
            paginationUrl += "?";
        } else if (!paginationUrl.endsWith("?")) {
            paginationUrl += "&";
        }
        paginationUrl += "pageNumber=";
        return paginationUrl;
    }

    public String getOneYearAgoDatePlaceHolder() {
        return formatDate(DateUtils.addYears(getNow(), -1));
    }

    protected Date getNow() {
        return new Date();
    }

    protected long getPaginationFirstPage(PaginationResult<?> paginationResult, int rangeSize) {
        int pageIndex = paginationResult.getCurrentPage().getPageNumber();
        long result = pageIndex;
        long pagesNumber = paginationResult.getPageCount();
        if (pagesNumber <= rangeSize || pageIndex < rangeSize) {
            // stick to begin
            return 1;
        }
        if (pagesNumber - pageIndex < rangeSize) {

            // just center
            result = pagesNumber - rangeSize;
            if (result < 1) {
                result = 1;
            }
            return result;
        }

        if (pageIndex >= rangeSize) {

            // center
            result = pageIndex - (rangeSize / 2);
        }
        return result;
    }

    protected long getPaginationLastPage(PaginationResult<?> paginationResult, int rangeSize, long firstPagination) {
        long result = firstPagination + rangeSize;
        if (result > paginationResult.getPageCount()) {
            result = paginationResult.getPageCount();
        }
        return result;
    }

    public boolean isLocaleFrench() {
        return WaoUtils.isFrench(getLocale());
    }

    protected boolean isSamplingPlanFullViewByDefault() {
        boolean fullView = getAuthenticatedWaoUser().isAdmin() || getAuthenticatedWaoUser().isProfessional();
        return fullView;
    }

    public boolean isContactsListFullViewByDefault() {
        boolean fullView = getAuthenticatedWaoUser().isAdmin() || getAuthenticatedWaoUser().isProfessional();
        return fullView;
    }

    public ImmutableMap<String, Object> getDefaultCookie() {
        return ImmutableMap.of(
                "version", (Object) "4.2",
                "samplingPlanFullView", isSamplingPlanFullViewByDefault(),
                "contactsListFullView", isContactsListFullViewByDefault(),
                "samplingPlanDaysView", false
        );
    }

}
