package fr.ifremer.wao.web.action;

/*
 * #%L
 * Wao :: Web
 * %%
 * Copyright (C) 2009 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.opensymphony.xwork2.Preparable;
import fr.ifremer.wao.WaoTechnicalException;
import fr.ifremer.wao.WaoUtils;
import fr.ifremer.wao.entity.Contact;
import fr.ifremer.wao.entity.ObsProgram;
import fr.ifremer.wao.services.AuthenticatedWaoUser;
import fr.ifremer.wao.services.service.ContactDataInputDateAfterTodayException;
import fr.ifremer.wao.services.service.ContactDataInputDateBeforeObservationEndDateException;
import fr.ifremer.wao.services.service.ContactNotUpdatableException;
import fr.ifremer.wao.services.service.ContactObservationEndDateAfterTodayException;
import fr.ifremer.wao.services.service.ContactObservationEndDateBeforeBeginDateException;
import fr.ifremer.wao.services.service.ContactRestitutionDateBeforeDataInputDateException;
import fr.ifremer.wao.services.service.ContactWithObservedDataControlToCorrectionAskedException;
import fr.ifremer.wao.services.service.ContactsService;
import fr.ifremer.wao.services.service.DuplicatedContactMainObserverInSecondaryObserversException;
import fr.ifremer.wao.services.service.IllegalAcceptationException;
import fr.ifremer.wao.services.service.InvalidContactObservationBeginDateException;
import fr.ifremer.wao.services.service.MismatchContactMainObserverCompanyException;
import fr.ifremer.wao.services.service.MismatchContactSecondaryObserverCompanyException;
import fr.ifremer.wao.services.service.MissingContactCommentAdminException;
import fr.ifremer.wao.services.service.MissingContactCommentException;
import fr.ifremer.wao.services.service.MissingContactDataInputDateException;
import fr.ifremer.wao.services.service.MissingContactDataReliabilityException;
import fr.ifremer.wao.services.service.MissingContactMainObserverException;
import fr.ifremer.wao.services.service.MissingContactMammalsInfoException;
import fr.ifremer.wao.services.service.MissingContactObservationBeginDateException;
import fr.ifremer.wao.services.service.MissingContactObservationEndDateException;
import fr.ifremer.wao.services.service.MissingContactObservedDataControlException;
import fr.ifremer.wao.services.service.MissingContactRestitutionException;
import fr.ifremer.wao.services.service.MissingContactStateMotifException;
import fr.ifremer.wao.services.service.MissingContactTerrestrialLocationException;
import fr.ifremer.wao.services.service.UnknownContactIdException;
import fr.ifremer.wao.services.service.UnwantedContactContactStateMotifException;
import fr.ifremer.wao.services.service.UpdateContactCommand;
import fr.ifremer.wao.web.WaoJsonActionSupport;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Created on 4/3/14.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 4.0
 */
public class ValidateContactJsonAction extends WaoJsonActionSupport implements Preparable {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(ValidateContactJsonAction.class);

    /**
     * Id of contact to treat.
     */
    protected String contactId;

    /**
     * New validation state.
     */
    protected Boolean validationState;

    /**
     * Success message
     */
    protected String successMessage;

    /**
     * Error message if validation, or any was wrong while update.
     */
    protected String errorMessage;

    protected transient ContactsService service;

    protected UpdateContactCommand updateContactCommand;

    public void setService(ContactsService service) {
        this.service = service;
    }

    public void setContactId(String contactId) {
        this.contactId = contactId;
    }

    public void setValidationState(Boolean validationState) {
        this.validationState = validationState;
    }

    public String getContactId() {
        return contactId;
    }

    public Boolean getValidationState() {
        return validationState;
    }

    public String getSuccessMessage() {
        return successMessage;
    }

    public String getErrorMessage() {
        return errorMessage;
    }

    public boolean isSuccessful() {
        return errorMessage == null;
    }

    @Override
    public void prepare() {

        Preconditions.checkState(StringUtils.isNotEmpty(contactId));

        try {
            updateContactCommand = service.newUpdateContactCommand(
                    session.getAuthenticatedWaoUser(),
                    contactId);
        } catch (UnknownContactIdException e) {
            addActionError(t("wao.ui.error.unknownContactId"));
        }
    }

    @Override
    public String execute() {

        AuthenticatedWaoUser authenticatedWaoUser = session.getAuthenticatedWaoUser();

        if (log.isInfoEnabled()) {
            log.info("user " + authenticatedWaoUser + " wants to change validation for contact " + contactId);
        }

        if (authenticatedWaoUser.isAdmin()) {
            updateContactCommand.setValidationProgram(validationState);
            if (log.isInfoEnabled()) {
                log.info("setting contact validation program to " + validationState);
            }
        } else {
            updateContactCommand.setValidationCompany(validationState);
            if (log.isInfoEnabled()) {
                log.info("setting contact validation company to " + validationState);
            }
        }

        ObsProgram obsProgram = updateContactCommand.getContact().getObsProgram();

        try {
            service.validate(authenticatedWaoUser, updateContactCommand);
        } catch (ContactNotUpdatableException e) {
            session.addErrorMessages(t("wao.ui.contacts.validation.failure.not.updatable"));

        } catch (UnwantedContactContactStateMotifException e) {
            errorMessage =  t("wao.ui.form.Contact.error.unwantedContactStateMotif");

        } catch (MissingContactObservationEndDateException e) {
            String state = WaoUtils.l(getLocale(), e.getContact().getContactState().toI18Able(obsProgram));
            errorMessage =  t("wao.ui.form.Contact.error.missingObservationEndDate", state);

        } catch (ContactDataInputDateBeforeObservationEndDateException e) {
            errorMessage =  t("wao.ui.form.Contact.error.dataInputDateBeforeObservationEndDate");

        } catch (InvalidContactObservationBeginDateException e) {
            errorMessage =  t("wao.ui.form.Contact.error.invalidObservationBeginDate");

        } catch (MissingContactRestitutionException e) {
            errorMessage =  t("wao.ui.form.Contact.error.missingRestitution");

        } catch (ContactDataInputDateAfterTodayException e) {
            errorMessage =  t("wao.ui.form.Contact.error.dataInputDateAfterToday");

        } catch (MissingContactCommentException e) {
            String state = WaoUtils.l(getLocale(), e.getContact().getContactState().toI18Able(obsProgram));
            errorMessage =  t("wao.ui.form.Contact.error.missingComment", state);

        } catch (MismatchContactMainObserverCompanyException e) {
            String companyName = e.getCompany().getName();
            String observerLogin = e.getObserver().getLogin();
            errorMessage =  t("wao.ui.form.Contact.error.mismatchCompanyForObserver", observerLogin, companyName);

        } catch (ContactRestitutionDateBeforeDataInputDateException e) {
            errorMessage =  t("wao.ui.form.Contact.error.transmissionDateBeforeDataInputDate");

        } catch (ContactObservationEndDateBeforeBeginDateException e) {
            errorMessage =  t("wao.ui.form.Contact.error.observationEndDateBeforeBeginDate");

        } catch (DuplicatedContactMainObserverInSecondaryObserversException e) {
            errorMessage =  t("wao.ui.form.Contact.error.duplicatedMainObserverInSecondaryObservers");

        } catch (MissingContactDataReliabilityException e) {
            errorMessage =  t("wao.ui.form.Contact.error.missingDataReliability");

        } catch (MismatchContactSecondaryObserverCompanyException e) {
            String companyName = e.getCompany().getName();
            String observerLogin = e.getObserver().getLogin();
            errorMessage =  t("wao.ui.form.Contact.error.mismatchCompanyForObserver", observerLogin, companyName);

        } catch (MissingContactObservedDataControlException e) {
            errorMessage =  t("wao.ui.form.Contact.error.missingObservedDataControl");

        } catch (ContactWithObservedDataControlToCorrectionAskedException e) {
            errorMessage = t("wao.ui.form.Contact.error.observedDataControlToCorrectionAsked");

        } catch (MissingContactStateMotifException e) {
            errorMessage =  t("wao.ui.form.Contact.error.missingContactStateMotif");

        } catch (MissingContactDataInputDateException e) {
            errorMessage =  t("wao.ui.form.Contact.error.missingDataInputDate");

        } catch (MissingContactMainObserverException e) {
            errorMessage =  t("wao.ui.form.Contact.error.missingMainObserver");

        } catch (MissingContactCommentAdminException e) {
            String dataReliability = WaoUtils.l(getLocale(), e.getContact().getDataReliability());
            errorMessage =  t("wao.ui.form.Contact.error.missingCommentAdmin", dataReliability);

        } catch (ContactObservationEndDateAfterTodayException e) {
            errorMessage =  t("wao.ui.form.Contact.error.observationEndDateAfterToday");

        } catch (MissingContactObservationBeginDateException e) {
            String state = WaoUtils.l(getLocale(), e.getContact().getContactState().toI18Able(obsProgram));
            errorMessage =  t("wao.ui.form.Contact.error.missingObservationBeginDate", state);

        } catch (MissingContactTerrestrialLocationException e) {
            errorMessage =  t("wao.ui.form.Contact.error.missingContactTerrestrialLocation");

        } catch (MissingContactMammalsInfoException e) {
            errorMessage =  t("wao.ui.form.Contact.error.missingContactMammalsInfo");

        } catch (IllegalAcceptationException e) {
            errorMessage =  t("wao.ui.form.Contact.error.illegalAcceptationException");

        }

        if (isSuccessful()) {

            try {
                service.save(updateContactCommand);
            } catch (ContactNotUpdatableException e) {
                throw new WaoTechnicalException("should never occur", e);
            }

            if (log.isInfoEnabled()) {
                log.info("changing validation state successful");
            }

            if (validationState == null) {
                successMessage = t("wao.ui.contacts.validation.to.unvalidate.state.success");
            } else if (validationState) {
                successMessage = t("wao.ui.contacts.validation.to.accept.state.success");
            } else {
                successMessage = t("wao.ui.contacts.validation.to.reject.state.success");
            }

        } else {
            if (log.isInfoEnabled()) {
                log.info("changing validation state failed for reason " + errorMessage);
            }
        }

        return SUCCESS;
    }

    public boolean isAcceptable(Contact contact) {
        boolean acceptable;
        if (session.getAuthenticatedWaoUser().isAuthorizedToChangeValidationCompany()) {
            acceptable = contact.isAcceptableByCompany();
        } else if (session.getAuthenticatedWaoUser().isAuthorizedToChangeValidationProgram()) {
            acceptable = contact.isAcceptableByProgram();
        } else {
            throw new IllegalStateException();
        }
        return acceptable;
    }

    public boolean isRefusable(Contact contact) {
        boolean refuseable;
        if (session.getAuthenticatedWaoUser().isAuthorizedToChangeValidationCompany()) {
            refuseable = contact.isRefusableByCompany();
        } else if (session.getAuthenticatedWaoUser().isAuthorizedToChangeValidationProgram()) {
            refuseable = contact.isRefusableByProgram();
        } else {
            throw new IllegalStateException();
        }
        return refuseable;
    }

    public boolean isUnacceptable(Contact contact) {
        boolean unacceptable;
        if (session.getAuthenticatedWaoUser().isAuthorizedToChangeValidationCompany()) {
            unacceptable = contact.isUnacceptableByCompany();
        } else if (session.getAuthenticatedWaoUser().isAuthorizedToChangeValidationProgram()) {
            unacceptable = contact.isUnacceptableByProgram();
        } else {
            throw new IllegalStateException();
        }
        return unacceptable;
    }

    public Boolean getValidationProgram() {
        return updateContactCommand.getContact().getValidationProgram();
    }

    public Boolean getValidationCompany() {
        return updateContactCommand.getContact().getValidationCompany();
    }

}
