package fr.ifremer.wao.web;

/*
 * #%L
 * Wao :: Web
 * %%
 * Copyright (C) 2009 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.opensymphony.xwork2.util.LocalizedTextUtil;
import fr.ifremer.wao.ContactsFilter;
import fr.ifremer.wao.WaoApplicationConfig;
import fr.ifremer.wao.WaoTopiaApplicationContext;
import fr.ifremer.wao.WaoTopiaPersistenceContext;
import fr.ifremer.wao.services.DefaultWaoServiceContext;
import fr.ifremer.wao.services.WaoApplicationContext;
import fr.ifremer.wao.services.WaoServiceContext;
import fr.ifremer.wao.services.WaoWebApplicationContext;
import fr.ifremer.wao.services.service.BoatsFilterValues;
import fr.ifremer.wao.services.service.BoatsFilterValuesCacheKey;
import fr.ifremer.wao.services.service.ContactsFilterValues;
import fr.ifremer.wao.services.service.ContactsFilterValuesCacheKey;
import fr.ifremer.wao.services.service.InitWaoService;
import fr.ifremer.wao.services.service.SamplingPlan;
import fr.ifremer.wao.services.service.SamplingPlanCacheKey;
import fr.ifremer.wao.services.service.Synthesis;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.LogManager;
import org.apache.log4j.PropertyConfigurator;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.i18n.init.I18nInitializer;

import java.io.File;
import java.util.Date;
import java.util.Locale;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

public class DefaultWaoApplicationContext implements WaoApplicationContext {

    private static Log log = LogFactory.getLog(DefaultWaoApplicationContext.class);

    public static final String APPLICATION_CONTEXT_PARAMETER = "WaoApplicationContext";

    protected WaoTopiaApplicationContext topiaApplicationContext;

    protected WaoApplicationConfig applicationConfig;

    protected WaoWebApplicationContext webApplicationContext;

    protected Random random;

    protected Cache<SamplingPlanCacheKey, SamplingPlan> samplingPlansCache;

    protected Cache<BoatsFilterValuesCacheKey, BoatsFilterValues> boatsFilterValuesCache;

    protected Cache<ContactsFilter, Synthesis> synthesesCache;

    protected Cache<ContactsFilterValuesCacheKey, ContactsFilterValues> contactsFilterValuesCache;

    @Override
    public WaoApplicationConfig getApplicationConfig() {
        if (applicationConfig == null) {
            applicationConfig = new WaoApplicationConfig();
        }
        return applicationConfig;
    }

    public WaoTopiaApplicationContext getTopiaApplicationContext() {
        if (topiaApplicationContext == null) {
            topiaApplicationContext = new WaoTopiaApplicationContext(getApplicationConfig().getTopiaProperties());
        }
        return topiaApplicationContext;
    }

    @Override
    public String newUuid() {
        return UUID.randomUUID().toString();
    }

    @Override
    public WaoWebApplicationContext getWebApplicationContext() {
        if (webApplicationContext == null) {
            webApplicationContext = new DefaultWaoWebApplicationContext(getApplicationConfig().getInstanceUrl());
        }
        return webApplicationContext;
    }

    @Override
    public Date getNow() {
        Date now = new Date();
        return now;
    }

    @Override
    public Random getRandom() {
        if (random == null) {
            random = new Random();
        }
        return random;
    }

    @Override
    public Cache<SamplingPlanCacheKey, SamplingPlan> getSamplingPlansCache() {
        if (samplingPlansCache == null) {
            samplingPlansCache = newCacheBuilder(100).expireAfterAccess(30, TimeUnit.DAYS).build();
        }
        return samplingPlansCache;
    }

    @Override
    public Cache<BoatsFilterValuesCacheKey, BoatsFilterValues> getBoatsFilterValuesCache() {
        if (boatsFilterValuesCache == null) {
            boatsFilterValuesCache = newCacheBuilder(100).expireAfterWrite(18, TimeUnit.HOURS).build();
        }
        return boatsFilterValuesCache;
    }

    @Override
    public Cache<ContactsFilter, Synthesis> getSynthesesCache() {
        if (synthesesCache == null) {
            synthesesCache = newCacheBuilder(100).expireAfterAccess(30, TimeUnit.DAYS).build();
        }
        return synthesesCache;
    }

    @Override
    public Cache<ContactsFilterValuesCacheKey, ContactsFilterValues> getContactsFilterValuesCache() {
        if (contactsFilterValuesCache == null) {
            contactsFilterValuesCache = newCacheBuilder(100).expireAfterWrite(18, TimeUnit.HOURS).build();
        }
        return contactsFilterValuesCache;
    }

    protected <K, V> CacheBuilder<K, V> newCacheBuilder(int expectedMaximumCacheSize) {
        int maximumCacheSize = applicationConfig.isCachingEnabled() ? expectedMaximumCacheSize : 0;
        CacheBuilder<K, V> cacheBuilder = (CacheBuilder<K, V>) CacheBuilder.newBuilder().maximumSize(maximumCacheSize);
        if (log.isInfoEnabled()) {
            cacheBuilder.recordStats();
        }
        return cacheBuilder;
    }

    @Override
    public WaoTopiaPersistenceContext newPersistenceContext() {

        WaoTopiaPersistenceContext persistenceContext = getTopiaApplicationContext().newPersistenceContext();

        if (applicationConfig.isCachingEnabled()) {

            CacheInvalidationTopiaEntityListener cacheInvalidationTopiaEntityListener =
                    new CacheInvalidationTopiaEntityListener(
                            getSamplingPlansCache(),
                            getBoatsFilterValuesCache(),
                            getSynthesesCache());

            persistenceContext.getTopiaFiresSupport().addTopiaEntityListener(cacheInvalidationTopiaEntityListener);
            persistenceContext.getTopiaFiresSupport().addTopiaTransactionListener(cacheInvalidationTopiaEntityListener);

        }

        return persistenceContext;

    }

    @Override
    public WaoServiceContext newServiceContext(WaoTopiaPersistenceContext persistenceContext, Locale locale) {

        DefaultWaoServiceContext newServiceContext =
                new DefaultWaoServiceContext(this, persistenceContext, locale);

        return newServiceContext;

    }

    @Override
    public void close() {

        if (log.isInfoEnabled()) {
            log.info("samplingPlansCache stats: " + getSamplingPlansCache().stats().toString());
            log.info("boatsFilterValuesCache stats: " + getBoatsFilterValuesCache().stats().toString());
            log.info("contactsFilterValuesCache stats: " + getContactsFilterValuesCache().stats().toString());
            log.info("synthesesCache stats: " + getSynthesesCache().stats().toString());
        }

        if (topiaApplicationContext != null) {

            topiaApplicationContext.close();

        }

    }

    public void init() {

        if (getApplicationConfig().isLogConfigurationProvided()) {

            File log4jConfigurationFile = getApplicationConfig().getLogConfigurationFile();

            String log4jConfigurationFileAbsolutePath = log4jConfigurationFile.getAbsolutePath();

            if (log4jConfigurationFile.exists()) {

                if (log.isInfoEnabled()) {
                    log.info("will use logging configuration " + log4jConfigurationFileAbsolutePath);
                }

                // reset logger configuration
                LogManager.resetConfiguration();

                // use generate log config file
                PropertyConfigurator.configure(log4jConfigurationFileAbsolutePath);

                log = LogFactory.getLog(DefaultWaoApplicationContext.class);

            } else {
                if (log.isWarnEnabled()) {
                    log.warn("there is no file " + log4jConfigurationFileAbsolutePath + ". Default logging configuration will be used.");
                }
            }

        } else {
            log.info("will use default logging configuration");
        }

        I18nInitializer initializer = new DefaultI18nInitializer("wao");
        // to show none translated sentences
        initializer.setMissingKeyReturnNull(true);

        I18n.init(initializer, Locale.FRANCE);

        LocalizedTextUtil.addDefaultResourceBundle("i18n.wao-web");

        WaoTopiaPersistenceContext persistenceContext = newPersistenceContext();

        WaoServiceContext serviceContext = newServiceContext(persistenceContext, Locale.FRANCE);

        InitWaoService initWaoService =
                serviceContext.newService(InitWaoService.class);

        initWaoService.init();

        persistenceContext.close();

    }

}
