package fr.ifremer.wao.web;

/*
 * #%L
 * Wao :: Web
 * %%
 * Copyright (C) 2009 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.cache.Cache;
import fr.ifremer.wao.ContactsFilter;
import fr.ifremer.wao.entity.Boat;
import fr.ifremer.wao.entity.Contact;
import fr.ifremer.wao.entity.ElligibleBoat;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.services.service.BoatsFilterValues;
import fr.ifremer.wao.services.service.BoatsFilterValuesCacheKey;
import fr.ifremer.wao.services.service.SamplingPlan;
import fr.ifremer.wao.services.service.SamplingPlanCacheKey;
import fr.ifremer.wao.services.service.Synthesis;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.event.TopiaEntityEvent;
import org.nuiton.topia.persistence.event.TopiaEntityListener;
import org.nuiton.topia.persistence.event.TopiaTransactionEvent;
import org.nuiton.topia.persistence.event.TopiaTransactionListener;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Avec ce listener, on scrute ce qu'il se passe sur l'application pour invalider les caches
 * qu'il faut.
 */
public class CacheInvalidationTopiaEntityListener implements TopiaEntityListener, TopiaTransactionListener {

    private static final Log log = LogFactory.getLog(CacheInvalidationTopiaEntityListener.class);

    protected Cache<SamplingPlanCacheKey, SamplingPlan> samplingPlansCache;

    protected Cache<BoatsFilterValuesCacheKey, BoatsFilterValues> boatsFilterValuesCache;

    protected Cache<ContactsFilter, Synthesis> synthesesCache;

    protected Set<String> dirtySampleRowIds = new HashSet<>();

    protected Set<String> dirtyBoatIds = new HashSet<>();

    protected Set<String> dirtyContactIds = new HashSet<>();

    protected boolean invalidateAllSampleRows = false;

    protected boolean invalidateAllBoats = false;

    protected boolean invalidateAllContacts = false;

    /**
     * On désactive pour se reposer uniquement sur l'invalidation au bout d'un certain temps.
     * @deprecated le temps du test en prod pour voir si ça améliore mais si on garde, à refactorer
     */
    @Deprecated
    protected boolean boatsFilterValuesCacheInvalidationEnabled = false;

    public CacheInvalidationTopiaEntityListener(
            Cache<SamplingPlanCacheKey, SamplingPlan> samplingPlansCache,
            Cache<BoatsFilterValuesCacheKey, BoatsFilterValues> boatsFilterValuesCache,
            Cache<ContactsFilter, Synthesis> synthesesCache) {
        this.samplingPlansCache = samplingPlansCache;
        this.boatsFilterValuesCache = boatsFilterValuesCache;
        this.synthesesCache = synthesesCache;
    }

    @Override
    public void create(TopiaEntityEvent event) {
        TopiaEntity entity = event.getEntity();
        if (entity instanceof SampleRow) {
            invalidateAllSampleRows = true;
        } else if (entity instanceof ElligibleBoat) {
            ElligibleBoat elligibleBoat = (ElligibleBoat) entity;
            dirtyBoatIds.add(elligibleBoat.getBoat().getTopiaId());
        } else if (entity instanceof Boat) {
            invalidateAllBoats = true;
        } else if (entity instanceof Contact) {
            Contact contact = (Contact) entity;
            dirtyBoatIds.add(contact.getBoat().getTopiaId());
            dirtySampleRowIds.add(contact.getSampleRow().getTopiaId());
            invalidateAllContacts = true;
        }
    }

    @Override
    public void load(TopiaEntityEvent event) {

    }

    @Override
    public void update(TopiaEntityEvent event) {
        TopiaEntity entity = event.getEntity();
        if (entity instanceof SampleRow) {
            dirtySampleRowIds.add(entity.getTopiaId());
        } else if (entity instanceof ElligibleBoat) {
            ElligibleBoat elligibleBoat = (ElligibleBoat) entity;
            dirtyBoatIds.add(elligibleBoat.getBoat().getTopiaId());
        } else if (entity instanceof Boat) {
            dirtyBoatIds.add(entity.getTopiaId());
        } else if (entity instanceof Contact) {
            Contact contact = (Contact) entity;
            dirtyBoatIds.add(contact.getBoat().getTopiaId());
            dirtySampleRowIds.add(contact.getSampleRow().getTopiaId());
            dirtyContactIds.add(contact.getTopiaId());
        }
    }

    @Override
    public void delete(TopiaEntityEvent event) {
        TopiaEntity entity = event.getEntity();
        if (entity instanceof SampleRow) {
            samplingPlansCache.invalidateAll();
        } else if (entity instanceof Contact) {
            Contact contact = (Contact) entity;
            dirtyBoatIds.add(contact.getBoat().getTopiaId());
            dirtySampleRowIds.add(contact.getSampleRow().getTopiaId());
            dirtyContactIds.add(contact.getTopiaId());
        }
    }

    @Override
    public void commit(TopiaTransactionEvent event) {
        if (log.isTraceEnabled()) {
            log.trace("will commit for entities " + event.getEntities());
        }
        if (invalidateAllSampleRows) {
            samplingPlansCache.invalidateAll();
            synthesesCache.invalidateAll();
        } else {
            Set<SamplingPlanCacheKey> samplingPlansCacheKeysToInvalidate = new HashSet<>();
            for (Map.Entry<SamplingPlanCacheKey, SamplingPlan> entry : samplingPlansCache.asMap().entrySet()) {
                if (CollectionUtils.containsAny(entry.getValue().getSampleRowIds(), dirtySampleRowIds)) {
                    samplingPlansCacheKeysToInvalidate.add(entry.getKey());
                }
            }
            samplingPlansCache.invalidateAll(samplingPlansCacheKeysToInvalidate);
            Set<ContactsFilter> synthesesCacheKeysToInvalidate = new HashSet<>();
            for (Map.Entry<ContactsFilter, Synthesis> entry : synthesesCache.asMap().entrySet()) {
                if (CollectionUtils.containsAny(entry.getValue().getSampleRowIds(), dirtySampleRowIds)) {
                    synthesesCacheKeysToInvalidate.add(entry.getKey());
                }
            }
            synthesesCache.invalidateAll(synthesesCacheKeysToInvalidate);
        }
        if (invalidateAllBoats) {
            if (boatsFilterValuesCacheInvalidationEnabled) {
                boatsFilterValuesCache.invalidateAll();
            }
        } else {
            Set<BoatsFilterValuesCacheKey> boatsFilterValuesCacheKeysToInvalidate = new HashSet<>();
            for (Map.Entry<BoatsFilterValuesCacheKey, BoatsFilterValues> entry : boatsFilterValuesCache.asMap().entrySet()) {
                if (CollectionUtils.containsAny(entry.getValue().getBoatIds(), dirtyBoatIds)) {
                    boatsFilterValuesCacheKeysToInvalidate.add(entry.getKey());
                }
            }
            if (boatsFilterValuesCacheInvalidationEnabled) {
                boatsFilterValuesCache.invalidateAll(boatsFilterValuesCacheKeysToInvalidate);
            }
        }
        clear();
    }

    @Override
    public void rollback(TopiaTransactionEvent event) {
        clear();
    }

    protected void clear() {
        dirtySampleRowIds.clear();
        dirtyBoatIds.clear();
        dirtyContactIds.clear();
        invalidateAllSampleRows = false;
        invalidateAllBoats = false;
        invalidateAllContacts = false;
    }
}
