package fr.ifremer.wao.web.action;

/*
 * #%L
 * Wao :: Web
 * %%
 * Copyright (C) 2009 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Objects;
import com.opensymphony.xwork2.Preparable;
import fr.ifremer.wao.BoatsFilter;
import fr.ifremer.wao.entity.Boat;
import fr.ifremer.wao.entity.Company;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.services.service.BoatsFilterValues;
import fr.ifremer.wao.services.service.BoatsList;
import fr.ifremer.wao.services.service.BoatsService;
import fr.ifremer.wao.services.service.SamplingPlanService;
import fr.ifremer.wao.services.service.administration.CompaniesService;
import fr.ifremer.wao.web.WaoJspActionSupport;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.pagination.PaginationParameter;
import org.nuiton.util.pagination.PaginationResult;

import java.util.List;

/**
 * Created on 4/1/14.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 4.0
 */
public class BoatsAction extends WaoJspActionSupport implements Preparable {

    private static final long serialVersionUID = 1L;

    private static final Log log = LogFactory.getLog(BoatsAction.class);

    protected transient BoatsService service;

    protected transient CompaniesService companiesService;

    protected transient SamplingPlanService samplingPlanService;

    protected int pageNumber; // 1-based page number

    protected long firstRangePage;

    protected long lastRangePage;

    /**
     * Boats filter.
     */
    protected BoatsFilter filter;

    /**
     * List of boats according to the input pager and filter.
     */
    protected BoatsList boatList;

    protected String startBoatSelectionForSampleRowId;

    protected SampleRow startBoatSelectionForSampleRow;

    public void setService(BoatsService service) {
        this.service = service;
    }

    public BoatsFilter getFilter() {
        if (filter == null) {
            prepare();
        }
        return filter;
    }

    public List<Boat> getBoats() {
        return boatList.getBoats().getElements();
    }

    public BoatsFilterValues getFilterValues() {
        return boatList.getFilterValues();
    }

    public PaginationResult<Boat> getPagination() {
        return boatList.getBoats();
    }

    public void setStartBoatSelectionForSampleRowId(String startBoatSelectionForSampleRowId) {
        this.startBoatSelectionForSampleRowId = startBoatSelectionForSampleRowId;
    }

    public void setSamplingPlanService(SamplingPlanService samplingPlanService) {
        this.samplingPlanService = samplingPlanService;
    }

    public void setCompaniesService(CompaniesService companiesService) {
        this.companiesService = companiesService;
    }

    public void setPageNumber(int pageNumber) {
        this.pageNumber = pageNumber;
    }

    @Override
    public void prepare() {

        if (filter == null) {
            filter = service.newBoatsFilter(getAuthenticatedWaoUser());
        }

        if (StringUtils.isNotBlank(startBoatSelectionForSampleRowId)) {

            session.setStartBoatSelectionForSampleRowId(startBoatSelectionForSampleRowId);

            if (log.isInfoEnabled()) {
                log.info("user " + getAuthenticatedWaoUser().getWaoUser() + " is starting to select a boat for sample row " + startBoatSelectionForSampleRow);
            }

        }

        if (session.getStartBoatSelectionForSampleRowId() != null) {

            startBoatSelectionForSampleRow = samplingPlanService.getSampleRow(session.getStartBoatSelectionForSampleRowId());

        }

    }

    @Override
    public String execute() {

        filter = Objects.firstNonNull(session.getBoatsFilter(), filter);

        return applyFilter();

    }

    public String resetFilter() {

        filter = service.newBoatsFilter(getAuthenticatedWaoUser());

        return applyFilter();

    }

    public String applyFilter() {

        getSession().setBoatsFilter(filter);

        PaginationParameter pager = service.newBoatsPaginationParameter(pageNumber, 25);
        boatList = service.getBoatsList(getAuthenticatedWaoUser(), filter, pager);

        firstRangePage = getPaginationFirstPage(boatList.getBoats(), 10);
        lastRangePage = getPaginationLastPage(boatList.getBoats(), 10, firstRangePage);

        return SUCCESS;

    }

    public SampleRow getStartBoatSelectionForSampleRow() {
        return startBoatSelectionForSampleRow;
    }

    public List<Company> getAllCompanies() {
        return companiesService.getAllCompanies();
    }

    public long getFirstRangePage() {
        return firstRangePage;
    }

    public long getLastRangePage() {
        return lastRangePage;
    }
}
