/*
 * #%L
 * IsisFish
 * 
 * $Id: AcceptorFactory.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.widget.filter;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Pattern;

/**
 * This class is an Acceptor factory.
 * <p>
 * For simple {@link Acceptor} we use a cache to use an unique instance.
 *
 * @author chemit
 */
public class AcceptorFactory {

    static private Log log = LogFactory.getLog(AcceptorFactory.class);

    /** cache of simple acceptors indexed by their klazz hashcode */
    protected static Map<Integer, Acceptor<?>> cache;

    /**
     * Obtain a new instance of a CompositeFilterAcceptor using as delegate
     * Acceptor a simple one in cache
     *
     * @param klazz  klazz of simple Acceptor to use
     * @param getter ValueGetter to use
     * @return a new instance of CompositeFilterAcceptor
     */
    public static <M, T> FilterAcceptor<M, T> createCompositeFilterAcceptor(
            Class<T> klazz, ValueGetter<M, T> getter) {
        return new CompositeFilterAcceptor<M, T>(klazz, getter);
    }

    /**
     * Obtain a acceptor from caceh via his klazz key
     *
     * @param klazz klazz of Acceptor to get
     * @return the Acceptor register in cache via his klazz
     */
    @SuppressWarnings({"unchecked"})
    public static <T> Acceptor<T> getAcceptor(Class<T> klazz) {
        return (Acceptor<T>) getCache().get(klazz.hashCode());
    }

    /** @return the unique instance of cache of Acceptor */
    protected synchronized static Map<Integer, Acceptor<?>> getCache() {
        if (cache == null) {
            cache = new TreeMap<Integer, Acceptor<?>>();

            // register simple implementation of acceptors
            // if you need more implements them
            cache.put(String.class.hashCode(), new StringAcceptor());
            cache.put(Boolean.class.hashCode(), new BooleanAcceptor());
            cache.put(Short.class.hashCode(), new ShortAcceptor());
            cache.put(Integer.class.hashCode(), new IntegerAcceptor());
            cache.put(Long.class.hashCode(), new LongAcceptor());
            cache.put(Float.class.hashCode(), new ShortAcceptor());
            cache.put(Double.class.hashCode(), new DoubleAcceptor());
            cache.put(DateInterval.class.hashCode(), new DateIntervalAcceptor());
            for (Acceptor<?> acceptor : cache.values()) {
                log.debug("registred " + acceptor);
            }
        }
        return cache;
    }

    /** Integer acceptor implementation */
    static class ShortAcceptor extends AbstractAcceptor<Short> {

        public Class<Short> getKlazz() {
            return Short.class;
        }

        public boolean accept(Short meta, Short value) {
            return super.accept(meta, value) && meta.equals(value);
        }

        public Short extract(Object args) {
            return args == null ? 0 : Short.valueOf(args.toString());
        }
    }

    /** Integer acceptor implementation */
    static class IntegerAcceptor extends AbstractAcceptor<Integer> {

        public Class<Integer> getKlazz() {
            return Integer.class;
        }

        public boolean accept(Integer meta, Integer value) {
            return super.accept(meta, value) && meta.equals(value);
        }

        public Integer extract(Object args) {
            return args == null ? 0 : Integer.valueOf(args.toString());
        }
    }

    /** Long acceptor implementation */
    public static class LongAcceptor extends AbstractAcceptor<Long> {

        public Class<Long> getKlazz() {
            return Long.class;
        }

        public boolean accept(Long meta, Long value) {
            return super.accept(meta, value) && meta.equals(value);
        }

        public Long extract(Object args) {
            return (args == null ? 0 : Long.valueOf(args.toString()));
        }
    }

    /** Float acceptor implementation */
    public static class FloatAcceptor extends AbstractAcceptor<Float> {

        public Class<Float> getKlazz() {
            return Float.class;
        }

        public boolean accept(Float meta, Float value) {
            return super.accept(meta, value) && meta.equals(value);
        }

        public Float extract(Object args) {
            return (args == null ? 0 : Float.valueOf(args.toString()));
        }
    }

    /** Double acceptor implementation */
    public static class DoubleAcceptor extends AbstractAcceptor<Double> {

        public Class<Double> getKlazz() {
            return Double.class;
        }

        public boolean accept(Double meta, Double value) {
            return super.accept(meta, value) && meta.equals(value);
        }

        public Double extract(Object args) {
            return (args == null ? 0 : Double.valueOf(args.toString()));
        }
    }

    /** String acceptor implementation */
    static class StringAcceptor extends AbstractAcceptor<String> {

        public Class<String> getKlazz() {
            return String.class;
        }

        public boolean accept(String meta, String value) {

            if (!super.accept(meta, value)) {
                return false;
            } else {
                if (!value.endsWith(".*") || value.endsWith("$")) {
                    // we always perform a prefix search (to help user)
                    value += ".*";
                }
                //TODO Should keep a cache of pattern to increase performance on
                //TODO long list (
                Pattern pattern = Pattern.compile(value, Pattern.CASE_INSENSITIVE);
                return pattern.matcher(meta).matches();
            }
        }

        public String extract(Object args) {
            return args == null ? "" : args.toString();
        }
    }

    /** Boolean acceptor implementation */
    static class BooleanAcceptor extends AbstractAcceptor<Boolean> {

        public Class<Boolean> getKlazz() {
            return Boolean.class;
        }

        public boolean accept(Boolean meta, Boolean value) {
            return super.accept(meta, value) && meta.equals(value);
        }

        public Boolean extract(Object args) {
            return Boolean.valueOf(args == null ? "false" : args.toString());
        }
    }

    /** DateInterval acceptor implementation */
    static class DateIntervalAcceptor extends AbstractAcceptor<DateInterval> {

        public Class<DateInterval> getKlazz() {
            return DateInterval.class;
        }

        public boolean accept(DateInterval meta, DateInterval value) {
            return super.accept(meta, value) &&
                    (value.getD0() == -1 || value.getD0() <= meta.getD0()) &&
                    (value.getD1() == -1 || value.getD1() >= meta.getD1());
        }

        public DateInterval extract(Object args) {
            if (args == null || !(args.getClass().isArray() && ((Object[]) args).length == 2))
                return null;
            Object o0, o1;
            o0 = ((Object[]) args)[0];
            o1 = ((Object[]) args)[1];
            return new DateInterval(o0 == null ? -1 : Long.valueOf(o0.toString()),
                    o1 == null ? -1 : Long.valueOf(o1.toString()));
        }

        public boolean canUse(Object[] value) {
            return super.canUse(value) || (value != null && value.length > 1 && !"".equals(value[1]));
        }
    }

    /** DateInterval acceptor implementation */
    static abstract class AbstractAcceptor<T> implements Acceptor<T> {

        public boolean accept(T meta, T value) {
            return !(meta == null || value == null);
        }

        public boolean canUse(Object[] value) {
            return value != null && value.length > 0 && !"".equals(value[0]);
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder(super.toString());
            sb.append(" [klazz ").append(getKlazz().getSimpleName()).append(']');
            return sb.toString();
        }
    }
}
