/*
 * #%L
 * $Id: SensitivityInputHandler.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL: https://svn.codelutin.com/isis-fish/tags/isis-fish-4.3.1.2/src/main/java/fr/ifremer/isisfish/ui/sensitivity/SensitivityInputHandler.java $
 * %%
 * Copyright (C) 2011 - 2012 Ifremer, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.sensitivity;

import static org.nuiton.i18n.I18n.t;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.io.File;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.text.JTextComponent;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.gui.MatrixPanelEditor;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityContextable;
import org.nuiton.util.FileUtil;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.entities.FisheryRegion;
import fr.ifremer.isisfish.mexico.MexicoHelper;
import fr.ifremer.isisfish.simulator.sensitivity.DesignPlan;
import fr.ifremer.isisfish.simulator.sensitivity.Factor;
import fr.ifremer.isisfish.simulator.sensitivity.FactorGroup;
import fr.ifremer.isisfish.types.Month;
import fr.ifremer.isisfish.types.TimeStep;
import fr.ifremer.isisfish.ui.input.InputContentUI;
import fr.ifremer.isisfish.ui.input.InputHandler;
import fr.ifremer.isisfish.ui.input.InputOneEquationUI;
import fr.ifremer.isisfish.ui.input.tree.FisheryDataProvider;
import fr.ifremer.isisfish.ui.input.tree.FisheryTreeHelper;
import fr.ifremer.isisfish.ui.input.tree.FisheryTreeNode;
import fr.ifremer.isisfish.ui.input.tree.FisheryTreeRenderer;
import fr.ifremer.isisfish.ui.sensitivity.wizard.FactorWizardUI;
import fr.ifremer.isisfish.ui.sensitivity.wizard.SensitivityWizardHandler;
import fr.ifremer.isisfish.ui.simulator.RuleChooser;
import fr.ifremer.isisfish.ui.simulator.SimulAction;

/**
 * Handler for sensitivity tab ui (fishery region factors).
 * 
 * @author chatellier
 * @version $Revision: 3969 $
 * 
 * Last update : $Date: 2014-04-17 18:48:13 +0200 (Thu, 17 Apr 2014) $
 * By : $Author: echatellier $
 */
public class SensitivityInputHandler extends InputHandler {

    /** Class logger. */
    private static Log log = LogFactory.getLog(SensitivityInputHandler.class);

    /**
     * 
     * @param sensitivityTabUI
     */
    public void loadFisheryRegionTree(SensitivityTabUI sensitivityTabUI) {
        FisheryRegion fisheryRegion = sensitivityTabUI.getFisheryRegion();

        if (fisheryRegion == null) {
            // show empty region ui
            sensitivityTabUI.getCardlayoutPrincipal().show(sensitivityTabUI.getInputPanePrincipal(),"none");
            TreeModel model = new DefaultTreeModel(null);
            sensitivityTabUI.getFisheryRegionTree().setModel(model);
        }
        else {
            // init tree model loader with fishery region
            FisheryTreeHelper treeHelper = new FisheryTreeHelper();
            FisheryDataProvider dataProvider = new FisheryDataProvider(fisheryRegion);
            treeHelper.setDataProvider(dataProvider);
            TreeModel model = treeHelper.createTreeModel(fisheryRegion);
            sensitivityTabUI.getFisheryRegionTree().setModel(model);
            sensitivityTabUI.getFisheryRegionTree().setCellRenderer(new FisheryTreeRenderer(dataProvider));
            treeHelper.setUI(sensitivityTabUI.getFisheryRegionTree(), true, false, null);

            // global context value : fisheryRegion, regionStorage, treeHelper
            sensitivityTabUI.setContextValue(fisheryRegion);
            sensitivityTabUI.setContextValue(treeHelper);
            sensitivityTabUI.setContextValue(model);
            sensitivityTabUI.setContextValue(fisheryRegion.getTopiaContext());

            sensitivityTabUI.getCardlayoutPrincipal().show(sensitivityTabUI.getInputPanePrincipal(),"normale");
        }
    }

    /**
     * Changement de selection dans l'arbre de la pecherie.
     * 
     * @param sensitivityTabUI
     * @param event
     */
    public void nodeSelectionChanged(SensitivityTabUI sensitivityTabUI, TreeSelectionEvent event) {

        TreePath newTreePath = event.getNewLeadSelectionPath();

        if (newTreePath != null) {
            Object lastTreePath = newTreePath.getLastPathComponent();
            if (lastTreePath instanceof FisheryTreeNode) {
                FisheryTreeNode isisTreeNode = (FisheryTreeNode)lastTreePath;

                Class<?> internalClass = isisTreeNode.getInternalClass();

                // noeud qui n'en charge pas d'autres (= un bean)
                TopiaEntityContextable topiaEntity = null;
                String topiaId = isisTreeNode.getId();

                try {
                    if (isisTreeNode.isStaticNode()) {
                        FisheryRegion fisheryRegion = sensitivityTabUI.getContextValue(FisheryRegion.class);
                        TopiaContext topiaContext = fisheryRegion.getTopiaContext();
                        topiaEntity = (TopiaEntityContextable)topiaContext.findByTopiaId(topiaId);
                    }

                    InputContentUI inputContentUI = getUIInstanceForBeanClass(internalClass, sensitivityTabUI);

                    // mandatory set
                    inputContentUI.getSaveVerifier().reset(); // before set bean !!!
                    if (topiaEntity != null) {
                        inputContentUI.getSaveVerifier().addCurrentEntity(topiaEntity);
                        inputContentUI.getSaveVerifier().setInputContentUI(inputContentUI);
                    }

                    inputContentUI.setBean((TopiaEntityContextable)topiaEntity);
                    inputContentUI.setActive(topiaEntity != null);
                    inputContentUI.setLayer(true);
                    inputContentUI.setSensitivity(true);

                    // add initialized ui to panel
                    sensitivityTabUI.getCardlayoutPrincipal().show(sensitivityTabUI.getInputPanePrincipal(), "normale");
                    sensitivityTabUI.getInputPane().removeAll();
                    sensitivityTabUI.getInputPane().add(inputContentUI, BorderLayout.CENTER);
                    sensitivityTabUI.getInputPane().repaint();
                    sensitivityTabUI.getInputPane().validate();
                } catch (Exception ex) {
                    throw new IsisFishRuntimeException("Can't display bean " + topiaId, ex);
                }
            }
        }
    }

    /**
     * Add new continuous factor group in factor tree.
     * 
     * @param sensitivityTabUI
     * @param continuous continuous
     */
    public void addNewFactorGroup(SensitivityTabUI sensitivityTabUI, boolean continuous) {
        String factorName = JOptionPane.showInputDialog(sensitivityTabUI, t("isisfish.sensitivity.newfactorname"),
                t("isisfish.sensitivity.title"), JOptionPane.QUESTION_MESSAGE);
        
        if (StringUtils.isNotBlank(factorName)) {
            FactorGroup rootFactorGroup = sensitivityTabUI.getSimulAction().getFactorGroup();
            FactorGroup factorGroup = new FactorGroup(factorName, continuous);
            rootFactorGroup.addFactor(factorGroup);
            sensitivityTabUI.setFactorModel();
        }
    }

    /**
     * Move factors to another factorgroup.
     * 
     * @param sensitivityTabUI
     * @param selectedFactorGroup
     * @param movedFactors
     */
    public void moveFactor(SensitivityTabUI sensitivityTabUI, FactorGroup selectedFactorGroup, List<Factor> movedFactors) {
        try {
            // add all factors, to do first, throw
            // exception if can't be done
            selectedFactorGroup.addAllFactors(movedFactors);

            // remove duplicated from factor group
            FactorGroup rootFactorGroup = sensitivityTabUI.getSimulAction().getFactorGroup();
            if (!rootFactorGroup.equals(selectedFactorGroup)) {
                rootFactorGroup.removeAll(movedFactors);
            }
            for (int index = 0 ; index < rootFactorGroup.size(); ++index) {
                Factor factor = rootFactorGroup.get(index);
                if (factor instanceof FactorGroup) {
                    FactorGroup factorGroup = (FactorGroup)factor;
                    if (!factorGroup.equals(selectedFactorGroup)) {
                        factorGroup.removeAll(movedFactors);
                    }
                }
            }
            sensitivityTabUI.setFactorModel();
        } catch (IllegalArgumentException ex) {
            JOptionPane.showMessageDialog(sensitivityTabUI, t("isisfish.sensitivity.moveillegal"),
                    t("isisfish.sensitivity.title"), JOptionPane.ERROR_MESSAGE);
        }
    }
    
    /**
     * Mouse click on factors tree.
     * 
     * <ul>
     *  <li>normal click : factor edit</li>
     *  <li>right click : popup menu</li>
     * </ul>
     * 
     * @param sensitivityTabUI ui
     * @param e mouse event
     */
    public void factorsTreeMouseClicked(final SensitivityTabUI sensitivityTabUI, MouseEvent e) {
        // clic droit
        if (e.getButton() == MouseEvent.BUTTON3) {
            JPopupMenu menu = new JPopupMenu();
            JMenuItem menuItemDelete = new JMenuItem(t("isisfish.common.delete"));
            menuItemDelete.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    deleteSelectedFactors(sensitivityTabUI);
                }
            });
            menu.add(menuItemDelete);
            menu.show(e.getComponent(), e.getX(), e.getY());
        }
        else if (e.getButton() == MouseEvent.BUTTON1 && e.getClickCount() == 2) {
            // autre double clic
            factorSelected(sensitivityTabUI);
        }
    }

    /**
     * Factor selection, display modification wizard.
     * 
     * @param sensitivityTabUI
     */
    protected void factorSelected(SensitivityTabUI sensitivityTabUI) {
        // get selected factor
        TreePath selectedPath = sensitivityTabUI.getFactorsTree().getSelectionPath();
        
        // method appelee au clic, donc pas forcement de selection
        if (selectedPath != null) {
            Object[] pathWay = selectedPath.getPath();
            Object selectedObject = pathWay[pathWay.length - 1];
            if (selectedObject != null) {
                if (!(selectedObject instanceof FactorGroup)) {
                    Factor selectedFactor = (Factor)selectedObject;
                    FactorWizardUI factorWizardUI = new FactorWizardUI(sensitivityTabUI);
                    SensitivityWizardHandler handler = factorWizardUI.getHandler();
                    handler.initExistingFactor(factorWizardUI, selectedFactor);
                    factorWizardUI.pack();
                    factorWizardUI.setLocationRelativeTo(sensitivityTabUI);
                    factorWizardUI.setVisible(true);
                }
            }
        }
    }

    /**
     * Delete selection factors.
     */
    protected void deleteSelectedFactors(SensitivityTabUI sensitivityTabUI) {
        // get selected factor
        TreePath[] selectedPaths = sensitivityTabUI.getFactorsTree().getSelectionPaths();
        if (!ArrayUtils.isEmpty(selectedPaths)) { // can happen
            for (TreePath selectedPath : selectedPaths) {
                Object[] pathWay = selectedPath.getPath();
                // > 2 : can't delete root
                if (pathWay.length >= 2) {
                    Object selectedObject = pathWay[pathWay.length - 1];
                    if (selectedObject != null) {
                        if (selectedObject instanceof Factor) {
                            Factor selectedFactor = (Factor)selectedObject;
                            FactorGroup selectedFactorGroup = (FactorGroup)pathWay[pathWay.length - 2];
                            if (log.isDebugEnabled()) {
                                log.debug("Deleting factor " + selectedFactor.getName());
                            }
                            selectedFactorGroup.remove(selectedFactor);
                            sensitivityTabUI.setFactorModel();
                        }
                    }
                }
            }
        }
    }

    /**
     * Return if type can be defined as a factor.
     * 
     * @param type type
     * @return {@code true} if type can be defined as a factor
     */
    public boolean canBeFactor(Class type) {
        boolean result = false;
        
        if (TopiaEntity.class.isAssignableFrom(type)) {
            result = true;
        } else if (double.class.isAssignableFrom(type)) {
            result = true;
        } else if (Number.class.isAssignableFrom(type)) {
            result = true;
        } else if (TimeStep.class.isAssignableFrom(type)) {
            result = true;
        } else if (Month.class.isAssignableFrom(type)) {
            result = true;
        }
        
        return result;
    }

    /**
     * Return value in swing component that could be next used into factor
     * for discrete factor values.
     * 
     * @param component component
     * @return factor value
     */
    protected Object getComponentValue(JComponent component) {
        Object result = null;
        if (component instanceof JTextComponent) {
            result = ((JTextComponent) component).getText();
        } else if (component instanceof MatrixPanelEditor) {
            result = ((MatrixPanelEditor) component).getMatrix();
        } else if (component instanceof InputOneEquationUI) {
            result = ((InputOneEquationUI) component).getEditor().getText();
        } else if (component instanceof RuleChooser) {
            result = ((RuleChooser)component).getRulesList();
        }
        if (log.isDebugEnabled()) {
            log.debug("Value for component : " + component.getClass().getSimpleName() + " is " + result);
        }

        return result;
    }

    /**
     * Export factors list to xml mexico file.
     * 
     * @since 4.1.1.2
     */
    public void exportFactorsToMexico(SensitivityTabUI view) {
        File xmlFile = FileUtil.getFile(t("isisfish.sensitivity.mexico.exporttoxml.title"),
                t("isisfish.sensitivity.mexico.exporttoxml.approve"), view,
                new String[]{
                    ".*\\.xml",
                    t("isisfish.sensitivity.mexico.exporttoxml.filter")
                });

        if (xmlFile != null) {
            
            // try to add ".xml" extension if no present
            if (!FilenameUtils.isExtension(xmlFile.getAbsolutePath(), "xml")) {
                xmlFile = new File(xmlFile.getAbsolutePath() + ".xml");
            }

            SimulAction simulAction = view.getContextValue(SimulAction.class);
            
            DesignPlan designPlan = new DesignPlan();
            designPlan.setFactorGroup(simulAction.getFactorGroup());
            String xml = MexicoHelper.getDesignPlanAsXML(designPlan);
            MexicoHelper.writeDesignPlanToFile(xmlFile, designPlan);
        }
    }
}
