/*
 * #%L
 * IsisFish
 * 
 * $Id: SimulationParameterCache.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL: https://svn.codelutin.com/isis-fish/tags/isis-fish-4.3.1.2/src/main/java/fr/ifremer/isisfish/simulator/SimulationParameterCache.java $
 * %%
 * Copyright (C) 2010 - 2012 Ifremer, Code Lutin, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.lang.ref.SoftReference;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.io.IOUtils;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaException;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.entities.Observation;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.Strategy;
import fr.ifremer.isisfish.export.Export;
import fr.ifremer.isisfish.export.SensitivityExport;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityAnalysis;

/**
 * SimulationParameter soft reference implementation.
 * 
 * Store simulation file on disk, and a soft reference of delegate real
 * SimulationParameter.
 * 
 * If soft reference has been cleared by garbage collector, reload it from
 * disk.
 * 
 * Warning, use only this class for parameter reading. Using setters cause
 * parameters to be written on disk. Use with caution.
 * 
 * @author chatellier
 * @version $Revision: 3969 $
 * 
 * Last update : $Date: 2014-04-17 18:48:13 +0200 (Thu, 17 Apr 2014) $
 * By : $Author: echatellier $
 */
public class SimulationParameterCache implements SimulationParameter {

    /** Parameter file (disk storage). */
    protected File paramFile;

    /** Parameter memory reference. */
    protected SoftReference<SimulationParameter> ref;

    /**
     * 
     * @param param param to cache
     */
    public SimulationParameterCache(SimulationParameter param) {

        try {
            paramFile = File.createTempFile("isis-fish-param", ".properties");
            paramFile.deleteOnExit();
            store(param);
        } catch (IOException eee) {
            throw new IsisFishRuntimeException("Can't cache param to disk", eee);
        }

        ref = new SoftReference<SimulationParameter>(param);
    }

    protected void store(SimulationParameter param) {
        Properties prop = param.toProperties();
        Writer writer = null;
        try {
            writer = new BufferedWriter(new FileWriter(paramFile));
            prop.store(writer, "Parameter cache");
        } catch (IOException eee) {
            throw new IsisFishRuntimeException("Can't cache param to disk", eee);
        } finally {
            IOUtils.closeQuietly(writer);
        }
    }
    
    protected SimulationParameter getParam() {
        SimulationParameter result = ref.get();
        if (result == null) {
            
            Reader reader = null;
            try {
                reader = new BufferedReader(new FileReader(paramFile));
                Properties prop = new Properties();
                prop.load(reader);
                result = new SimulationParameterImpl();
                result.fromProperties(prop);
                ref = new SoftReference<SimulationParameter>(result);
            } catch (IOException eee) {
                throw new IsisFishRuntimeException("Can't cache param to disk", eee);
            } finally {
                IOUtils.closeQuietly(reader);
            }
            
        }
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#check()
     */
    @Override
    public List<String> check() {
        SimulationParameter param = getParam();
        return param.check();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getIsisFishVersion()
     */
    @Override
    public String getIsisFishVersion() {
        SimulationParameter param = getParam();
        return param.getIsisFishVersion();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setIsisFishVersion(java.lang.String)
     */
    @Override
    public void setIsisFishVersion(String isisFishVersion) {
        SimulationParameter param = getParam();
        param.setIsisFishVersion(isisFishVersion);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getDescription()
     */
    @Override
    public String getDescription() {
        SimulationParameter param = getParam();
        return param.getDescription();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setDescription(java.lang.String)
     */
    @Override
    public void setDescription(String description) {
        SimulationParameter param = getParam();
        param.setDescription(description);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getRegion()
     */
    @Override
    public RegionStorage getRegion() {
        SimulationParameter param = getParam();
        return param.getRegion();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getNumberOf(fr.ifremer.isisfish.entities.Population)
     */
    @Override
    public MatrixND getNumberOf(Population pop) {
        SimulationParameter param = getParam();
        return param.getNumberOf(pop);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getExtraRules()
     */
    @Override
    public List<String> getExtraRules() {
        SimulationParameter param = getParam();
        return param.getExtraRules();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#addExtraRules(java.lang.String[])
     */
    @Override
    public void addExtraRules(String... extraRules) {
        SimulationParameter param = getParam();
        param.addExtraRules(extraRules);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getPopulations()
     */
    @Override
    public List<Population> getPopulations() {
        SimulationParameter param = getParam();
        return param.getPopulations();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setPopulations(java.util.List)
     */
    @Override
    public void setPopulations(List<Population> populations) {
        SimulationParameter param = getParam();
        param.setPopulations(populations);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getStrategies()
     */
    @Override
    public List<Strategy> getStrategies() {
        SimulationParameter param = getParam();
        return param.getStrategies();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setStrategies(java.util.List)
     */
    @Override
    public void setStrategies(List<Strategy> strategies) {
        SimulationParameter param = getParam();
        param.setStrategies(strategies);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#addSimulationPlan(fr.ifremer.isisfish.simulator.SimulationPlan)
     */
    @Override
    public void addSimulationPlan(SimulationPlan plan) {
        SimulationParameter param = getParam();
        param.addSimulationPlan(plan);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#removeSimulationPlan(fr.ifremer.isisfish.simulator.SimulationPlan)
     */
    @Override
    public boolean removeSimulationPlan(SimulationPlan plan) {
        SimulationParameter param = getParam();
        boolean result = param.removeSimulationPlan(plan);
        store(param);
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimulationPlans()
     */
    @Override
    public List<SimulationPlan> getSimulationPlans() {
        SimulationParameter param = getParam();
        return param.getSimulationPlans();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulationPlans(java.util.List)
     */
    @Override
    public void setSimulationPlans(List<SimulationPlan> plans) {
        SimulationParameter param = getParam();
        param.setSimulationPlans(plans);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isIndependentPlan()
     */
    @Override
    public boolean isIndependentPlan() {
        SimulationParameter param = getParam();
        return param.isIndependentPlan();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#clearPlans()
     */
    @Override
    public void clearPlans() {
        SimulationParameter param = getParam();
        param.clearPlans();
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#addRule(fr.ifremer.isisfish.rule.Rule)
     */
    @Override
    public void addRule(Rule rule) {
        SimulationParameter param = getParam();
        param.addRule(rule);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#removeRule(fr.ifremer.isisfish.rule.Rule)
     */
    @Override
    public boolean removeRule(Rule rule) {
        SimulationParameter param = getParam();
        boolean result = param.removeRule(rule);
        store(param);
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getRules()
     */
    @Override
    public List<Rule> getRules() {
        SimulationParameter param = getParam();
        return param.getRules();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setRules(java.util.List)
     */
    @Override
    public void setRules(List<Rule> rules) {
        SimulationParameter param = getParam();
        param.setRules(rules);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#clearRules()
     */
    @Override
    public void clearRules() {
        SimulationParameter param = getParam();
        param.clearRules();
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getUseOptimization()
     */
    @Override
    public boolean getUseOptimization() {
        SimulationParameter param = getParam();
        return param.getUseOptimization();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setUseOptimization(boolean)
     */
    @Override
    public void setUseOptimization(boolean useOptimization) {
        SimulationParameter param = getParam();
        param.setUseOptimization(useOptimization);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getUseStatistic()
     */
    @Override
    public boolean getUseStatistic() {
        SimulationParameter param = getParam();
        return param.getUseStatistic();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setUseStatistic(boolean)
     */
    @Override
    public void setUseStatistic(boolean useStatistic) {
        SimulationParameter param = getParam();
        param.setUseStatistic(useStatistic);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getExportNames()
     */
    @Override
    public List<String> getExportNames() {
        SimulationParameter param = getParam();
        return param.getExportNames();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setExportNames(java.util.List)
     */
    @Override
    public void setExportNames(List<String> exportNames) {
        SimulationParameter param = getParam();
        param.setExportNames(exportNames);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getNumberOfSensitivitySimulation()
     */
    @Override
    public int getNumberOfSensitivitySimulation() {
        SimulationParameter param = getParam();
        return param.getNumberOfSensitivitySimulation();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setNumberOfSensitivitySimulation(int)
     */
    @Override
    public void setNumberOfSensitivitySimulation(
            int numberOfSensitivitySimulation) {
        SimulationParameter param = getParam();
        param.setNumberOfSensitivitySimulation(numberOfSensitivitySimulation);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSensitivityAnalysis()
     */
    @Override
    public SensitivityAnalysis getSensitivityAnalysis() {
        SimulationParameter param = getParam();
        return param.getSensitivityAnalysis();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSensitivityAnalysis(fr.ifremer.isisfish.simulator.sensitivity.SensitivityAnalysis)
     */
    @Override
    public void setSensitivityAnalysis(SensitivityAnalysis sensitivityAnalysis) {
        SimulationParameter param = getParam();
        param.setSensitivityAnalysis(sensitivityAnalysis);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSensitivityExport()
     */
    @Override
    public List<SensitivityExport> getSensitivityExport() {
        SimulationParameter param = getParam();
        return param.getSensitivityExport();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSensitivityExport(java.util.List)
     */
    @Override
    public void setSensitivityExport(List<SensitivityExport> sensitivityExport) {
        SimulationParameter param = getParam();
        param.setSensitivityExport(sensitivityExport);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getUseSimulationPlan()
     */
    @Override
    public boolean getUseSimulationPlan() {
        SimulationParameter param = getParam();
        return param.getUseSimulationPlan();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setUseSimulationPlan(boolean)
     */
    @Override
    public void setUseSimulationPlan(boolean useSimulationPlan) {
        SimulationParameter param = getParam();
        param.setUseSimulationPlan(useSimulationPlan);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimualtionPlanNumber()
     */
    @Override
    public int getSimulationPlanNumber() {
        SimulationParameter param = getParam();
        return param.getSimulationPlanNumber();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulationPlanNumber(int)
     */
    @Override
    public void setSimulationPlanNumber(int simulationPlanNumber) {
        SimulationParameter param = getParam();
        param.setSimulationPlanNumber(simulationPlanNumber);
        store(param);
    }

    @Override
    public boolean isSensitivityAnalysisOnlyKeepFirst() {
        SimulationParameter param = getParam();
        return param.isSensitivityAnalysisOnlyKeepFirst();
    }

    @Override
    public void setSensitivityAnalysisOnlyKeepFirst(boolean onlyKeepFirst) {
        SimulationParameter param = getParam();
        param.setSensitivityAnalysisOnlyKeepFirst(onlyKeepFirst);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getNumberOfYear()
     */
    @Override
    public int getNumberOfYear() {
        SimulationParameter param = getParam();
        return param.getNumberOfYear();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setNumberOfYear(int)
     */
    @Override
    public void setNumberOfYear(int numberOfYear) {
        SimulationParameter param = getParam();
        param.setNumberOfYear(numberOfYear);
        store(param);
        
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getUsePreScript()
     */
    @Override
    public boolean getUsePreScript() {
        SimulationParameter param = getParam();
        return param.getUsePreScript();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setUsePreScript(boolean)
     */
    @Override
    public void setUsePreScript(boolean usePreScript) {
        SimulationParameter param = getParam();
        param.setUsePreScript(usePreScript);
        store(param);
        
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getPreScript()
     */
    @Override
    public String getPreScript() {
        SimulationParameter param = getParam();
        return param.getPreScript();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setPreScript(java.lang.String)
     */
    @Override
    public void setPreScript(String preScript) {
        SimulationParameter param = getParam();
        param.setPreScript(preScript);
        store(param);

    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getGeneratedPreScript()
     */
    @Override
    public String getGeneratedPreScript() {
        SimulationParameter param = getParam();
        return param.getGeneratedPreScript();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setGeneratedPreScript(java.lang.String)
     */
    @Override
    public void setGeneratedPreScript(String preScript) {
        SimulationParameter param = getParam();
        param.setGeneratedPreScript(preScript);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getRegionName()
     */
    @Override
    public String getRegionName() {
        SimulationParameter param = getParam();
        return param.getRegionName();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setRegionName(java.lang.String)
     */
    @Override
    public void setRegionName(String regionName) {
        SimulationParameter param = getParam();
        param.setRegionName(regionName);
        store(param);
        
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimulatorName()
     */
    @Override
    public String getSimulatorName() {
        SimulationParameter param = getParam();
        return param.getSimulatorName();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulatorName(java.lang.String)
     */
    @Override
    public void setSimulatorName(String simulatorName) {
        SimulationParameter param = getParam();
        param.setSimulatorName(simulatorName);
        store(param);
        
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getResultEnabled()
     */
    @Override
    public Collection<String> getResultEnabled() {
        SimulationParameter param = getParam();
        return param.getResultEnabled();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setResultEnabled(java.util.Collection)
     */
    @Override
    public void setResultEnabled(Collection<String> resultEnabled) {
        SimulationParameter param = getParam();
        param.setResultEnabled(resultEnabled);
        store(param);
        
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getTagValue()
     */
    @Override
    public Map<String, String> getTagValue() {
        SimulationParameter param = getParam();
        return param.getTagValue();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setTagValue(java.util.Map)
     */
    @Override
    public void setTagValue(Map<String, String> tagValue) {
        SimulationParameter param = getParam();
        param.setTagValue(tagValue);
        store(param);
        
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimulLogLevel()
     */
    @Override
    public String getSimulLogLevel() {
        SimulationParameter param = getParam();
        return param.getSimulLogLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulLogLevel(java.lang.String)
     */
    @Override
    public void setSimulLogLevel(String logLevel) {
        SimulationParameter param = getParam();
        param.setSimulLogLevel(logLevel);
        store(param);
        
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getScriptLogLevel()
     */
    @Override
    public String getScriptLogLevel() {
        SimulationParameter param = getParam();
        return param.getScriptLogLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setScriptLogLevel(java.lang.String)
     */
    @Override
    public void setScriptLogLevel(String logLevel) {
        SimulationParameter param = getParam();
        param.setScriptLogLevel(logLevel);
        store(param);
        
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getLibLogLevel()
     */
    @Override
    public String getLibLogLevel() {
        SimulationParameter param = getParam();
        return param.getLibLogLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setLibLogLevel(java.lang.String)
     */
    @Override
    public void setLibLogLevel(String logLevel) {
        SimulationParameter param = getParam();
        param.setLibLogLevel(logLevel);
        store(param);
        
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulErrorLevel()
     */
    @Override
    public boolean isSimulErrorLevel() {
        SimulationParameter param = getParam();
        return param.isSimulErrorLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulWarnLevel()
     */
    @Override
    public boolean isSimulWarnLevel() {
        SimulationParameter param = getParam();
        return param.isSimulErrorLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulInfoLevel()
     */
    @Override
    public boolean isSimulInfoLevel() {
        SimulationParameter param = getParam();
        return param.isScriptInfoLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulDebugLevel()
     */
    @Override
    public boolean isSimulDebugLevel() {
        SimulationParameter param = getParam();
        return param.isSimulDebugLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptErrorLevel()
     */
    @Override
    public boolean isScriptErrorLevel() {
        SimulationParameter param = getParam();
        return param.isScriptErrorLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptWarnLevel()
     */
    @Override
    public boolean isScriptWarnLevel() {
        SimulationParameter param = getParam();
        return param.isScriptWarnLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptInfoLevel()
     */
    @Override
    public boolean isScriptInfoLevel() {
        SimulationParameter param = getParam();
        return param.isScriptInfoLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptDebugLevel()
     */
    @Override
    public boolean isScriptDebugLevel() {
        SimulationParameter param = getParam();
        return param.isScriptDebugLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibErrorLevel()
     */
    @Override
    public boolean isLibErrorLevel() {
        SimulationParameter param = getParam();
        return param.isLibErrorLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibWarnLevel()
     */
    @Override
    public boolean isLibWarnLevel() {
        SimulationParameter param = getParam();
        return param.isLibWarnLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibInfoLevel()
     */
    @Override
    public boolean isLibInfoLevel() {
        SimulationParameter param = getParam();
        return param.isLibInfoLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibDebugLevel()
     */
    @Override
    public boolean isLibDebugLevel() {
        SimulationParameter param = getParam();
        return param.isLibDebugLevel();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibDebugLevel()
     */
    @Override
    public void setProperty(String key, String value) {
        SimulationParameter param = getParam();
        param.setProperty(key, value);
    }
    
    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#copy()
     */
    @Override
    public SimulationParameter copy() {
        SimulationParameter param = getParam();
        return param.copy();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#deepCopy()
     */
    @Override
    public SimulationParameter deepCopy() {
        SimulationParameter param = getParam();
        return param.deepCopy();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#toProperties()
     */
    @Override
    public Properties toProperties() {
        SimulationParameter param = getParam();
        return param.toProperties();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#fromProperties(java.util.Properties)
     */
    @Override
    public void fromProperties(Properties props) {
        SimulationParameter param = getParam();
        param.fromProperties(props);
        store(param);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#reloadContextParameters()
     */
    @Override
    public void reloadContextParameters() throws TopiaException {
        SimulationParameter param = getParam();
        param.reloadContextParameters();
        store(param);
    }

    @Override
    public Optimization getOptimization() {
        SimulationParameter param = getParam();
        return param.getOptimization();
    }

    @Override
    public Objective getObjective() {
        SimulationParameter param = getParam();
        return param.getObjective();
    }

    @Override
    public void setOptimization(Optimization optimization) {
        SimulationParameter param = getParam();
        param.setOptimization(optimization);
        store(param);
    }

    @Override
    public void setObjective(Objective objective) {
        SimulationParameter param = getParam();
        param.setObjective(objective);
        store(param);
    }

    @Override
    public Map<Export, Observation> getOptimizationExportsObservations() {
        SimulationParameter param = getParam();
        return param.getOptimizationExportsObservations();
    }

    @Override
    public void setOptimizationExportsObservations(Map<Export, Observation> exportsObservations) {
        SimulationParameter param = getParam();
        param.setOptimizationExportsObservations(exportsObservations);
        store(param);
    }

    @Override
    public boolean getUseCache() {
        SimulationParameter param = getParam();
        return param.getUseCache();
    }

    @Override
    public void setUseCache(boolean useCache) {
        SimulationParameter param = getParam();
        param.setUseCache(useCache);
        store(param);
    }
}
