/*
 * #%L
 * IsisFish
 * 
 * $Id: PopulationGroupImpl.java 4179 2015-01-27 17:41:11Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2011 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.entities;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.LazyInitializationException;

import fr.ifremer.isisfish.types.Month;

/**
 * PopulationGroupImpl.
 *
 * Created: 3 janv. 2006 15:26:53
 *
 * @author poussin
 * @version $Revision: 4179 $
 *
 * Last update: $Date: 2015-01-27 18:41:11 +0100 (Tue, 27 Jan 2015) $
 * by : $Author: echatellier $
 */
public class PopulationGroupImpl extends PopulationGroupAbstract {

    /** Logger for this class. */
    private static final Log log = LogFactory.getLog(PopulationGroupImpl.class);

    /** serialVersionUID. */
    private static final long serialVersionUID = 1L;

    /*
     * @see fr.ifremer.isisfish.entities.PopulationGroup#getAge()
     */
    @Override
    public double getAge() {
        double result = 0;
        if (getPopulation() != null && getPopulation().getSpecies() != null) {
            if (getPopulation().getSpecies().getAgeGroupType()) {
                result = super.getAge();
            } else {
                result = getPopulation().getAge(getLength(), this)
                        / Month.NUMBER_OF_MONTH;
            }
        }
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.entities.PopulationGroup#getLongueurMin()
     */
    @Override
    public double getLength() {
        double result = 0.0;
        if (getPopulation() != null && getPopulation().getSpecies() != null) {
            if (!getPopulation().getSpecies().getAgeGroupType()) {
                result = (getMinLength() + getMaxLength()) / 2.0;
            } else {
                result = getPopulation().getLength(
                        getAge() * Month.NUMBER_OF_MONTH, this);
            }
        }
        return result;
    }

    @Override
    public boolean isInLength(double length) {
        boolean result = getMinLength() <= length && length <= getMaxLength();
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.entities.PopulationGroup#getMeanWeight()
     */
    @Override
    public double getMeanWeight() {
        double result = 0;
        try {
            Equation eq = getPopulation().getMeanWeight();
            result = eq.evaluate("group", this);
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Error in mean weight equation");
            }
            if (log.isDebugEnabled()) {
                log.debug("StackTrace", eee);
            }
        }
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.entities.PopulationGroup#getNaturalDeathRate()
     */
    @Override
    public double getNaturalDeathRate(Zone zone) {
        double result = 0;
        try {
            Equation eq = getPopulation().getNaturalDeathRate();
            if (eq != null) {
                result = eq.evaluate("pop", this.getPopulation(), "group",
                        this, "zone", zone);
            }
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Error in natural death rate equation");
            }
            if (log.isDebugEnabled()) {
                log.debug("StackTrace", eee);
            }
        }
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.entities.PopulationGroup#getPrice()
     */
    @Override
    public double getPrice() {
        double result = 0;
        try {
            Equation eq = getPopulation().getPrice();
            if (eq != null) {
                result = eq.evaluate("group", this);
            }
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Error in price equation");
            }
            if (log.isDebugEnabled()) {
                log.debug("StackTrace", eee);
            }
        }
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.entities.PopulationGroup#getMaturityOgive()
     */
    @Override
    public double getMaturityOgive() {
        double result = 0;
        try {
            Equation eq = getPopulation().getMaturityOgiveEquation();
            if (eq != null) {
                result = eq.evaluate("group", this);
            }
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Error in maturityOgive equation");
            }
            if (log.isDebugEnabled()) {
                log.debug("StackTrace", eee);
            }
        }
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.entities.PopulationGroup#getReproductionRate()
     */
    @Override
    public double getReproductionRate() {
        double result = 0;
        try {
            Equation eq = getPopulation().getReproductionRateEquation();
            if (eq != null) {
                result = eq.evaluate("group", this);
            }
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Error in reproductionRate equation");
            }
            if (log.isDebugEnabled()) {
                log.debug("StackTrace", eee);
            }
        }
        return result;
    }

    private String toStringCache;
    @Override
    public String toString() {
        // trop couteux d'appeler i18n, appele des millions de fois
        // et est vraiment raisonnable d'avoir des string != suivant le langage
        // quel impact dans les scripts utilisateurs ?
//        return t("isisfish.populationGroup.toString", this.getPopulation(),
//                this.getId());
        if (toStringCache == null) {
            try {
                toStringCache = this.getPopulation() + " Group " + this.getId();
            } catch (LazyInitializationException e) {
                // cette exception se produit quand le toString est appelé par exemple lors
                // su lancement d'une population mais que l'entité d'origine a été rechargée
                // depuis une précédente transation fermé (rechargement d'une simulation
                // avec ses facteurs)
                toStringCache = "Unknow Population Group " + this.getId();
            }
        }
        return toStringCache;
    }
}
