/*
 * #%L
 * IsisFish
 * 
 * $Id: SSHAgent.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, Code Lutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.util.ssh;

import static org.nuiton.i18n.I18n.t;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPasswordField;

import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.KeyPair;

/**
 * Isis SSH agent.
 * 
 * This class centralize, ssh key and passphrase
 * management.
 *
 * This is a singleton class.
 * 
 * @author chatellier
 * @version $Revision: 3969 $
 * 
 * Last update : $Date: 2014-04-17 18:48:13 +0200 (jeu., 17 avril 2014) $
 * By : $Author: echatellier $
 */
public class SSHAgent {

    /** JSSch instance. */
    protected JSch jsch;

    /**
     * Passphrase associated to key file path.
     * 
     * private key file path =&lt; passphrase.
     * 
     * Ensure that passphrase are valid before add into.
     */
    protected Map<String, String> passphraseForKeys;

    /** Unique agent. */
    private static SSHAgent agent = new SSHAgent();

    /**
     * Constructor.
     */
    private SSHAgent() {
        jsch = new JSch();
        passphraseForKeys = new HashMap<String, String>();
    }

    /**
     * Get agent instance.
     * 
     * @return agent
     */
    public static SSHAgent getAgent() {
        return agent;
    }

    /**
     * Get passphrase for key.
     * 
     * @param privatekeyFile private key file
     * @return <tt>null</tt> if there is no passphrase
     * @throws InvalidPassphraseException if user cancel authentication
     */
    public String getPassphrase(File privatekeyFile)
            throws InvalidPassphraseException {
        String passphrase = getPassphrase(privatekeyFile.getAbsolutePath());
        return passphrase;
    }

    /**
     * Get passphrase for key.
     * 
     * @param privatekeyFile private key file path
     * @return <tt>null</tt> if there is no passphrase
     * @throws InvalidPassphraseException if user cancel authentication
     */
    public String getPassphrase(String privatekeyFile)
            throws InvalidPassphraseException {

        String passphrase = passphraseForKeys.get(privatekeyFile);

        if (passphrase == null) {
            try {
                KeyPair kpair = KeyPair.load(jsch, privatekeyFile);

                if (kpair.isEncrypted()) {

                    boolean isValid = false;
                    String message = t("isisfish.ssh.askpassphrase.message",
                            privatekeyFile);
                    do {
                        passphrase = askPassphrase(privatekeyFile, message);

                        if (kpair.decrypt(passphrase)) {
                            isValid = true;
                            passphraseForKeys.put(privatekeyFile, passphrase);
                        } else {
                            message = t("isisfish.ssh.askpassphrase.wrongpassphrase",
                                    privatekeyFile);
                        }
                    } while (!isValid);
                }
            } catch (JSchException e) {
                throw new RuntimeException("Can't find key : " + privatekeyFile);
            }
        }

        return passphrase;
    }

    /**
     * Ask for passphrase.
     * 
     * @param privatekeyFile private key file path
     * @param message message
     * @return entrered passphrase
     * @throws InvalidPassphraseException if user cancel authentication
     */
    protected String askPassphrase(String privatekeyFile, String message)
            throws InvalidPassphraseException {

        JPasswordField passwordField = new JPasswordField();
        JLabel labelField = new JLabel(message);

        Object[] objs = { labelField, passwordField };

        int option = JOptionPane.showConfirmDialog(null, objs,
                t("isisfish.ssh.askpassphrase.title"),
                JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);

        if (option == JOptionPane.CANCEL_OPTION) {
            throw new InvalidPassphraseException("User cancel passphrase ask");
        }

        return String.valueOf(passwordField.getPassword());
    }

}
