/*
 * #%L
 * $Id: InputHandler.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL: http://svn.codelutin.com/isis-fish/trunk/src/main/java/fr/ifremer/isisfish/ui/input/InputHandler.java $
 * %%
 * Copyright (C) 2011 - 2012 Ifremer, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.input;

import static org.nuiton.i18n.I18n.n;
import static org.nuiton.i18n.I18n.t;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.SwingUtilities;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import jaxx.runtime.JAXXContext;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntityContextable;
import org.nuiton.util.FileUtil;

import fr.ifremer.isisfish.IsisFishDAOHelper;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.datastore.StorageException;
import fr.ifremer.isisfish.datastore.update.ImportFromV2;
import fr.ifremer.isisfish.entities.Cell;
import fr.ifremer.isisfish.entities.FisheryRegion;
import fr.ifremer.isisfish.entities.Gear;
import fr.ifremer.isisfish.entities.Metier;
import fr.ifremer.isisfish.entities.Observation;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.Port;
import fr.ifremer.isisfish.entities.SetOfVessels;
import fr.ifremer.isisfish.entities.Species;
import fr.ifremer.isisfish.entities.SpeciesDAO;
import fr.ifremer.isisfish.entities.Strategy;
import fr.ifremer.isisfish.entities.TripType;
import fr.ifremer.isisfish.entities.VesselType;
import fr.ifremer.isisfish.entities.Zone;
import fr.ifremer.isisfish.mexico.export.RegionExplorer;
import fr.ifremer.isisfish.mexico.export.RegionExport;
import fr.ifremer.isisfish.mexico.export.RegionExportFactorXML;
import fr.ifremer.isisfish.ui.CommonHandler;
import fr.ifremer.isisfish.ui.NavigationUI;
import fr.ifremer.isisfish.ui.input.tree.FisheryDataProvider;
import fr.ifremer.isisfish.ui.input.tree.FisheryTreeHelper;
import fr.ifremer.isisfish.ui.input.tree.FisheryTreeNode;
import fr.ifremer.isisfish.ui.input.tree.FisheryTreeRenderer;
import fr.ifremer.isisfish.ui.input.tree.FisheryTreeSelectionModel;
import fr.ifremer.isisfish.ui.input.tree.loadors.PopulationsNodeLoador;
import fr.ifremer.isisfish.ui.models.common.GenericComboModel;
import fr.ifremer.isisfish.ui.sensitivity.wizard.FactorWizardUI;
import fr.ifremer.isisfish.ui.sensitivity.wizard.SensitivityWizardHandler;
import fr.ifremer.isisfish.ui.simulator.filter.SimulationFilterUtil;
import fr.ifremer.isisfish.vcs.VCSException;

/**
 * Main handler for fishery edition action.
 * Next, each ui as his own handler.
 * 
 * In context : 
 * <ul>
 *  <li>FisheryRegion
 *  <li>RegionStorage
 * </ul>
 * 
 * @author chatellier
 * @version $Revision: 3969 $
 * 
 * Last update : $Date: 2014-04-17 18:48:13 +0200 (jeu., 17 avril 2014) $
 * By : $Author: echatellier $
 */
public class InputHandler extends CommonHandler {

    /** Class logger. */
    private static Log log = LogFactory.getLog(InputHandler.class);
 
    /**
     * Cache pour n'instancier les ui qu'une seule fois
     * et eviter que l'affichage saute pour l'utilsateur.
     */
    protected Map<Class<?>, InputContentUI<?>> uiInstanceCache = new HashMap<Class<?>, InputContentUI<?>>();

    /**
     * Load region by region name, set it into jaxx context and refresh ui.
     * 
     * Before loading region, try to close old one.
     * 
     * @param inputUI input ui
     * @param name region name to load
     */
    protected void loadRegion(InputUI inputUI, String name) {
        if (log.isDebugEnabled()) {
            log.debug("Load region " + name);
        }

        if (name == null) {
            // show empty region ui
            inputUI.getCardlayoutPrincipal().show(inputUI.getInputPanePrincipal(), "none");
            TreeModel model = new DefaultTreeModel(null);
            inputUI.getFisheryRegionTree().setModel(model);
        }
        else {
            FisheryRegion fisheryRegion = null;
            RegionStorage regionStorage = null;
            TopiaContext topiaContext = null;

            // load region
            try {
                regionStorage = RegionStorage.getRegion(name);
                topiaContext = regionStorage.getStorage().beginTransaction();
                fisheryRegion = RegionStorage.getFisheryRegion(topiaContext);
            } catch (TopiaException ex) {
                throw new IsisFishRuntimeException("Can't load region", ex);
            } catch (StorageException ex) {
                throw new IsisFishRuntimeException("Can't load region", ex);
            }

            // TODO echatellier 20110323 voir pour remplacer le binding
            inputUI.setRegionLoaded(fisheryRegion != null);

            // init tree model loader with fishery region
            FisheryTreeHelper fisheryTreeHelper = new FisheryTreeHelper();
            FisheryDataProvider dataProvider = new FisheryDataProvider(fisheryRegion);
            fisheryTreeHelper.setDataProvider(dataProvider);
            TreeModel fisheryTreeModel = fisheryTreeHelper.createTreeModel(fisheryRegion);
            inputUI.getFisheryRegionTree().setCellRenderer(null); // fix region opening after import
            inputUI.getFisheryRegionTree().setModel(fisheryTreeModel);
            inputUI.getFisheryRegionTree().setCellRenderer(new FisheryTreeRenderer(dataProvider));
            inputUI.getFisheryRegionTree().setSelectionModel(new FisheryTreeSelectionModel(inputUI));
            fisheryTreeHelper.setUI(inputUI.getFisheryRegionTree(), true, false, null);

            // global context value : fisheryRegion, regionStorage, treeHelper
            inputUI.setContextValue(fisheryRegion);
            inputUI.setContextValue(regionStorage);
            inputUI.setContextValue(fisheryTreeHelper);
            inputUI.setContextValue(fisheryTreeModel);
            inputUI.setContextValue(topiaContext);

            inputUI.getCardlayoutPrincipal().show(inputUI.getInputPanePrincipal(),"normale");
            
            // autoselect root node (fire some event)
            fisheryTreeHelper.selectNode((FisheryTreeNode)fisheryTreeModel.getRoot());
        }
    }

    /**
     * Reload current loaded fishery tree.
     * 
     * @param inputUI inputUI
     */
    public void reloadFisheryTree(InputUI inputUI) {
        FisheryTreeHelper fisheryTreeHelper = inputUI.getContextValue(FisheryTreeHelper.class);
        TreeModel fisheryTreeModel = inputUI.getContextValue(TreeModel.class);
        fisheryTreeHelper.refreshNode((FisheryTreeNode)fisheryTreeModel.getRoot(), true);
    }

    /**
     * Main ui fishery region selection changed.
     * 
     * @param inputUI inputUI
     * @param e event
     */
    public void regionChange(final InputUI inputUI, ItemEvent e) {
        // event launched twice with itemchange listener
        if (e.getStateChange() == ItemEvent.SELECTED) {
            final String name = (String)inputUI.getFieldCurrentRegion().getSelectedItem();
            if (log.isDebugEnabled()) {
                log.debug("New region selected " + name);
            }

            // long operation, run status bar
            setStatusMessage(inputUI, t("isisfish.message.loading.region", name), true);
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    loadRegion(inputUI, name);
                    setStatusMessage(inputUI, t("isisfish.message.load.finished"));
                }
            });
        }
    }

    /**
     * Create new region (called if region name is not empty).
     * 
     * @param inputUI inputUI
     */
    public void createNewRegion(InputUI inputUI) {
        String name = inputUI.getFieldNewRegion().getText();
        setStatusMessage(inputUI, t("isisfish.message.creating.region", name), true);
        
        if (RegionStorage.getRegionNames().contains(name)) {
            JOptionPane.showMessageDialog(inputUI, t("isisfish.error.region.already.exists"));
        }
        else {
            try {
                RegionStorage.create(name);
            } catch (StorageException ex) {
                throw new IsisFishRuntimeException("Can't create region", ex);
            }
            inputUI.getFieldNewRegion().setText("");
            refreshRegionList(inputUI, name);
        }

        setStatusMessage(inputUI, t("isisfish.message.creation.finished"));
    }

    /**
     * Rafraichit la liste des regions.
     * 
     * @param inputUI inputUI
     * @param selectedItem region to select
     */
    protected void refreshRegionList(InputUI inputUI, String selectedItem) {
        // refresh region list
        GenericComboModel regionsModel = new GenericComboModel(RegionStorage.getRegionNames());
        inputUI.getFieldCurrentRegion().setModel(regionsModel);
        // event is fired by setSelectedItem to call #regionChange
        if (selectedItem != null) {
            inputUI.getFieldCurrentRegion().setSelectedItem(selectedItem);
        }
        else {
            loadRegion(inputUI, null);
        }
    }

    /**
     * Exporter la region dans un zip.
     *
     * @param inputUI inputUI
     */
    public void importRegion(InputUI inputUI) {
        setStatusMessage(inputUI, t("isisfish.message.import.zip"), true);

        try {
            File file = FileUtil.getFile(".*.zip$",
                    t("isisfish.message.import.region.zipped"));
            if (file != null) {
                RegionStorage.importZip(file);
                
                refreshRegionList(inputUI, null);
            }

        } catch (Exception eee) {
            throw new IsisFishRuntimeException(t("isisfish.error.region.import"), eee);
        }
        
        setStatusMessage(inputUI, t("isisfish.message.import.finished"));
    }

    /**
     * Exporter la region dans un zip.
     *
     * @param inputUI inputUI
     */
    public void importRegionAndRename(InputUI inputUI) {
        setStatusMessage(inputUI, t("isisfish.message.import.zip"), true);

        try {
            File file = FileUtil.getFile(".*.zip$",
                    t("isisfish.message.import.region.zipped"));
            if (file != null) {

                String newName = JOptionPane
                        .showInputDialog(t("isisfish.message.name.imported.region"));
                RegionStorage.importAndRenameZip(file, newName);
                
                refreshRegionList(inputUI, null);
            }
        } catch (Exception eee) {
            throw new IsisFishRuntimeException(t("isisfish.error.region.import"), eee);
        }
        
        setStatusMessage(inputUI, t("isisfish.message.import.finished"));
    }

    /**
     * Importer la region depuis un fichier XML de la version 2.
     * 
     * @param inputUI inputUI
     */
    public void importV2Region(InputUI inputUI) {
        
        setStatusMessage(inputUI, t("isisfish.message.import.xml.v2.file"), true);

        try {
            File file = FileUtil.getFile(".*.xml$", t("isisfish.message.import.region.xml"));
            if (file != null) {
                new ImportFromV2(true).importXML(file);
                
                refreshRegionList(inputUI, null);
            }
        } catch (Exception eee) {
            throw new IsisFishRuntimeException(t("isisfish.error.region.import"), eee);
        }
        
        setStatusMessage(inputUI, t("isisfish.message.import.finished"));
    }

    /**
     * Extract from a simulation the region, and rename it with name given
     * by user.
     *
     * @param inputUI inputUI
     */
    public void importRegionFromSimulation(InputUI inputUI) {

        setStatusMessage(inputUI, t("isisfish.message.import"), true);

        // first step select a simulation and new region name
        try {
            //TODO Should use a shared model ?
            List<String> simulationNames = SimulationStorage.getSimulationNames();
            String simulationName = SimulationFilterUtil.selectSimulation(simulationNames);
            if (simulationName != null) {
                
                if (log.isInfoEnabled()) {
                    log.info("simulation used " + simulationName);
                }
                
                // ask new region name
                String regionName = JOptionPane.showInputDialog(
                        t("isisfish.message.import.region.name"), "region from "
                                + simulationName);
                
                if (StringUtils.isNotBlank(regionName)) {
                    if (RegionStorage.getRegionNames().contains(regionName)) {
                        JOptionPane.showMessageDialog(inputUI, t("isisfish.error.region.already.exists"));
                    }
                    else {
                        SimulationStorage.getSimulation(simulationName).extractRegion(regionName);
                    }
                }
            }

        } catch (Exception eee) {
            throw new IsisFishRuntimeException(t("isisfish.error.no.select.simulation"), eee);
        }

        setStatusMessage(inputUI, t("isisfish.message.export.done"));
    }

    /**
     * Exporter la region dans un zip
     *
     * Call only if region is loaded.
     * 
     * @param inputUI inputUI
     */
    public void exportRegion(InputUI inputUI) {
        try {
            File file = FileUtil.getFile(".*.zip$",
                    t("isisfish.message.import.region.zipped"));
            
            if (file != null) {
                // add .zip extension is not set
                if (!file.getAbsolutePath().endsWith(".zip")) {
                    file = new File(file.getAbsolutePath() + ".zip");
                }

                int resp = JOptionPane.YES_OPTION;
                if (file.exists()) {
                    resp = JOptionPane.showConfirmDialog(inputUI,
                            t("isisfish.message.file.overwrite"));
                }
                if (resp == JOptionPane.YES_OPTION) {
                    RegionStorage regionStorage = inputUI.getContextValue(RegionStorage.class);
                    regionStorage.createZip(file);
                }
            }
        } catch (IOException eee) {
            throw new IsisFishRuntimeException("Can't export region", eee);
        }
    }
    
    /**
     * Remove region.
     *
     * @param inputUI input ui
     * @param vcsDelete if true delete region in CVS too
     */
    public void removeRegion(InputUI inputUI, boolean vcsDelete) {
        try {
            RegionStorage regionStorage = inputUI.getContextValue(RegionStorage.class);
            int resp = JOptionPane.showConfirmDialog(inputUI,
                    t("isisfish.message.confirm.remove.region", regionStorage.getName()));
            if (resp == JOptionPane.YES_OPTION) {
                regionStorage.delete(vcsDelete);
                refreshRegionList(inputUI, null);
            }
        } catch (StorageException eee) {
            throw new IsisFishRuntimeException("Can't delete region", eee);
        }
    }
    
    /**
     * Copy la region avec un autre nom.
     *
     * @param inputUI input ui
     */
    public void copyRegion(InputUI inputUI) {
        try {
            String newName = JOptionPane
                    .showInputDialog(t("isisfish.message.new.region.name"));

            if (StringUtils.isNotEmpty(newName)) {
                RegionStorage regionStorage = inputUI.getContextValue(RegionStorage.class);
                File zip = regionStorage.createZip();
                RegionStorage.importAndRenameZip(zip, newName);
            }

        } catch (IOException eee) {
            throw new IsisFishRuntimeException("Can't copy region", eee);
        } catch (StorageException eee) {
            throw new IsisFishRuntimeException("Can't copy region", eee);
        }
    }
    
    /**
     * Commit region.
     *
     * @param inputUI input ui
     */
    public void commitRegionInCVS(InputUI inputUI) {
        try {
            RegionStorage regionStorage = inputUI.getContextValue(RegionStorage.class);
            setStatusMessage(inputUI, t("isisfish.message.commiting.region", regionStorage.getName()), true);
            String msg = regionStorage.getCommentForNextCommit();
            JTextArea text = new JTextArea(msg);
            int resp = JOptionPane.showOptionDialog(null,
                    new JScrollPane(text), t("isisfish.commit.message"),
                    JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE,
                    null, // icon
                    null, null);
            if (resp == JOptionPane.OK_OPTION) {
                regionStorage.commit(text.getText());
                regionStorage.clearCommentForNextCommit();
                setStatusMessage(inputUI, t("isisfish.message.region.commited"));
            } else {
                setStatusMessage(inputUI, t("isisfish.message.commit.region.canceled"));
            }
        } catch (VCSException eee) {
            throw new IsisFishRuntimeException("Can't commit region", eee);
        } catch (IOException eee) {
            throw new IsisFishRuntimeException("Can't commit region", eee);
        }
        setStatusMessage(inputUI, t("isisfish.message.export.done"));
    }
    
    /**
     * Explore region and export any enabled sensitivity factor name with value.
     * 
     * @param inputUI input ui
     */
    public void exportRegionSensitivityFactors(InputUI inputUI) {

        try {
            File exportFile = FileUtil.getFile(t("isisfish.input.sensitivity.export.title"),
                    t("isisfish.common.ok"), inputUI, ".*\\.xml", "XML Files");

            if (exportFile != null) {
                // make sur that filename ends with ".xml"
                if (!exportFile.getAbsolutePath().endsWith(".xml")) {
                    exportFile = new File(exportFile.getAbsolutePath() + ".xml");
                }

                FisheryRegion fisheryRegion = inputUI.getContextValue(FisheryRegion.class);
                setStatusMessage(inputUI, t("isisfish.input.sensitivity.export.running"), true);
                RegionExplorer explorer = new RegionExplorer();
                RegionExport exportXML = new RegionExportFactorXML(exportFile);
                explorer.explore(fisheryRegion, exportXML);
                setStatusMessage(inputUI, t("isisfish.input.sensitivity.export.complete"), true);
            }
            else {
                setStatusMessage(inputUI, t("isisfish.input.sensitivity.export.cancel"), true);
            }
        } catch (TopiaException eee) {
            throw new IsisFishRuntimeException("Can't export sensitivity factors", eee);
        }
    }

    /**
     * Changement de selection dans l'arbre de la pecherie.
     * 
     * @param inputUI
     * @param event
     */
    public void nodeSelectionChanged(InputUI inputUI, TreeSelectionEvent event) {

        TreePath newTreePath = event.getNewLeadSelectionPath();
        if (newTreePath != null) {
            Object lastTreePath = newTreePath.getLastPathComponent();
            if (lastTreePath instanceof FisheryTreeNode) {
                FisheryTreeNode isisTreeNode = (FisheryTreeNode)lastTreePath;

                Class<?> internalClass = isisTreeNode.getInternalClass();
                TopiaEntityContextable topiaEntity = null;
                String topiaId = isisTreeNode.getId();

                try {
                    // noeud qui n'en charge pas d'autres (= un bean)
                    if (isisTreeNode.isStaticNode()) {
                        FisheryRegion fisheryRegion = inputUI.getContextValue(FisheryRegion.class);
                        TopiaContext topiaContext = fisheryRegion.getTopiaContext();
                        topiaEntity = (TopiaEntityContextable)topiaContext.findByTopiaId(topiaId);
                    }

                    InputContentUI inputContentUI = getUIInstanceForBeanClass(internalClass, inputUI);
                    inputContentUI.getSaveVerifier().setInputContentUI(inputContentUI);

                    // mandatory set
                    inputContentUI.getSaveVerifier().reset(); // before set bean !!!
                    if (topiaEntity != null) {
                        inputContentUI.getSaveVerifier().addCurrentEntity(topiaEntity);
                    }

                    inputContentUI.setBean(topiaEntity);
                    inputContentUI.setActive(topiaEntity != null);

                    // add initialized ui to panel
                    inputUI.getCardlayoutPrincipal().show(inputUI.getInputPanePrincipal(), "normale");
                    inputUI.getInputPane().removeAll();
                    inputUI.getInputPane().add(inputContentUI, BorderLayout.CENTER);
                    inputUI.getInputPane().repaint();
                    inputUI.getInputPane().validate();
                } catch (Exception ex) {
                    throw new IsisFishRuntimeException("Can't display bean " + topiaId, ex);
                }
            }
        }
    }

    /**
     * Get ui that can display internalClass.
     * 
     * @param internalClass internal class to get ui
     * @return ui for class
     * @throws Exception 
     */
    protected InputContentUI<?> getUIInstanceForBeanClass(Class<?> internalClass, NavigationUI navigationUI) throws Exception {

        Class<? extends InputContentUI<?>> uiClass = null;
        if (FisheryRegion.class.isAssignableFrom(internalClass)) {
            uiClass = FisheryRegionUI.class;
        } else if (Cell.class.isAssignableFrom(internalClass)) {
            uiClass = CellUI.class;
        } else if (Gear.class.isAssignableFrom(internalClass)) {
            uiClass = GearUI.class;
        } else if (Metier.class.isAssignableFrom(internalClass)) {
            uiClass = MetierUI.class;
        } else if (Population.class.isAssignableFrom(internalClass)) {
            uiClass = PopulationUI.class;
        } else if (Port.class.isAssignableFrom(internalClass)) {
            uiClass = PortUI.class;
        } else if (SetOfVessels.class.isAssignableFrom(internalClass)) {
            uiClass = SetOfVesselsUI.class;
        } else if (Species.class.isAssignableFrom(internalClass)) {
            uiClass = SpeciesUI.class;
        } else if (Strategy.class.isAssignableFrom(internalClass)) {
            uiClass = StrategyUI.class;
        } else if (TripType.class.isAssignableFrom(internalClass)) {
            uiClass = TripTypeUI.class;
        } else if (VesselType.class.isAssignableFrom(internalClass)) {
            uiClass = VesselTypeUI.class;
        } else if (Zone.class.isAssignableFrom(internalClass)) {
            uiClass = ZoneUI.class;
        } else if (Observation.class.isAssignableFrom(internalClass)) {
            uiClass = ObservationUI.class;
        }

        // use map to implement UI cache
        InputContentUI<?> result = uiInstanceCache.get(uiClass);
        if (result == null) {
            Constructor<?> constructor = uiClass.getConstructor(JAXXContext.class);
            result = (InputContentUI<?>)constructor.newInstance(navigationUI);

            uiInstanceCache.put(uiClass, result);
        }

        return result;
    }

    /**
     * Change tree selection with new node id.
     * 
     * Called by specific UI (CellUI to change node).
     * 
     * @param inputUI context ui (to get context value tree helper...)
     * @param nodeId node id to select
     */
    public void setTreeSelection(InputContentUI<?> inputUI, String nodeId) {
        setTreeSelection(inputUI, null, nodeId);
    }
    
    /**
     * Change tree selection with new node id.
     * 
     * Called by specific UI (CellUI to change node).
     * 
     * @param inputUI context ui (to get context value tree helper...)
     * @param parentNodeId find node to select from this node
     * @param nodeId node id to select
     */
    public void setTreeSelection(InputContentUI<?> inputUI, String parentNodeId, String nodeId) {
        FisheryTreeHelper fisheryTreeHelper = inputUI.getContextValue(FisheryTreeHelper.class);
        TreeModel fisheryTreeModel = inputUI.getContextValue(TreeModel.class);
        FisheryTreeNode fromNode = (FisheryTreeNode)fisheryTreeModel.getRoot();
        if (parentNodeId != null) {
            fromNode = fisheryTreeHelper.findNode(fromNode, parentNodeId);
        }
        FisheryTreeNode newSelectNode = fisheryTreeHelper.findNode(fromNode, nodeId);
        if (newSelectNode != null) {
            fisheryTreeHelper.selectNode(newSelectNode);
        }
    }

    /**
     * Delete tree node referenced by parent, and auto select parent node.
     * 
     * @param inputUI ui containing tree
     * @param topiaId node id to delete
     */
    public void deleteTreeNode(InputUI inputUI, String topiaId) {
        FisheryTreeHelper fisheryTreeHelper = inputUI.getContextValue(FisheryTreeHelper.class);
        TreeModel fisheryTreeModel = inputUI.getContextValue(TreeModel.class);
        FisheryTreeNode newSelectNode = fisheryTreeHelper.findNode((FisheryTreeNode)fisheryTreeModel.getRoot(), topiaId);
        FisheryTreeNode parentNode = newSelectNode.getParent();
        fisheryTreeHelper.selectNode(parentNode);
        fisheryTreeHelper.removeNode(newSelectNode);
    }

    /**
     * Insert new tree node and select it.
     * 
     * @param inputUI ui containing tree
     * @param nodeClass node type to create
     * @param topiaEntity node to insert
     */
    public void insertTreeNode(InputUI inputUI, Class nodeClass, TopiaEntityContextable topiaEntity) {
        FisheryTreeHelper fisheryTreeHelper = inputUI.getContextValue(FisheryTreeHelper.class);

        // on part du principe que pour ne pas compliquer les ui est les lier
        // à l'arbre, on ajoute un nouveau noeud en fils de celui selectionné
        // (ou son parent si la creation à lieu à partir d'un noeud bean)
        FisheryTreeNode selectedNode = fisheryTreeHelper.getSelectedNode();
        if (selectedNode.isStaticNode()) {
            selectedNode = selectedNode.getParent();
        }

        // must use loador to properly load species node
        /*FisheryTreeNodeLoador typeNodeLoador = fisheryTreeHelper.getLoadorFor(nodeClass);
        FisheryTreeNode newNode = null;
        if (newNode != null) {
            newNode = (FisheryTreeNode)typeNodeLoador.createNode(topiaEntity, null);
        }
        else {
            // FIXME echatellier 20110418 cas non résolu du node "population"
            // pour lequel on n'a pas le loador
            newNode = new FisheryTreeNode(nodeClass, topiaEntity.getTopiaId(), null, null);
        }*/

        // FIXME echatellier 20110429 Hack en dur pour population
        FisheryTreeNode newNode = new FisheryTreeNode(nodeClass, topiaEntity.getTopiaId(), null, null);
        fisheryTreeHelper.insertNode(selectedNode, newNode);
        if (nodeClass.equals(Species.class)) {
            FisheryTreeNode newPopNode = new FisheryTreeNode(
                    Population.class, n("isisfish.input.tree.populations"),
                    null, new PopulationsNodeLoador((Species)topiaEntity));
            fisheryTreeHelper.insertNode(newNode, newPopNode);
        }
        fisheryTreeHelper.selectNode(newNode);
    }

    /**
     * Update tree node for topiaId.
     * 
     * @param inputUI ui containing tree
     * @param topiaId node id to update
     */
    public void updateTreeNode(InputUI inputUI, String topiaId) {
        FisheryTreeHelper fisheryTreeHelper = inputUI.getContextValue(FisheryTreeHelper.class);
        TreeModel fisheryTreeModel = inputUI.getContextValue(TreeModel.class);
        FisheryTreeNode newSelectNode = fisheryTreeHelper.findNode((FisheryTreeNode)fisheryTreeModel.getRoot(), topiaId);
        fisheryTreeHelper.refreshNode(newSelectNode, false);
    }
    
    /**
     * Action appelée lors du clic sur un layer (sensitivity).
     * 
     * @param inputContentUI inputContentUI
     * @param e l'event initial intersepté par le layer
     */
    public void accept(InputContentUI<?> inputContentUI, ActionEvent e) {

        // get clicked component info
        JComponent source = (JComponent) e.getSource();
        Class<? extends TopiaEntityContextable> sensitivityBeanClass =
                (Class<? extends TopiaEntityContextable>)source.getClientProperty("sensitivityBean");
        String sensitivityBeanID = (String)source.getClientProperty("sensitivityBeanID");
        String sensitivityMethod = (String)source.getClientProperty("sensitivityMethod");

        if (log.isDebugEnabled()) {
            log.debug("Event intercepted " + source);
            log.debug(" client property (bean) : " + sensitivityBeanClass);
            log.debug(" client property (beanID) : " + sensitivityBeanID);
            log.debug(" client property (method) : " + sensitivityMethod);
        }

        displayFactorWizard(inputContentUI, sensitivityBeanClass, sensitivityBeanID, sensitivityMethod);
    }
    
    public void displayFactorWizard(InputContentUI<?> inputContentUI, Class<? extends TopiaEntityContextable> sensitivityBeanClass,
            String sensitivityBeanID, String sensitivityMethod) {

        // get bean for component class info
        TopiaEntityContextable bean = null;
        if (sensitivityBeanID == null) {
            bean = inputContentUI.getSaveVerifier().getEntity(sensitivityBeanClass);
        }
        else {
            bean = inputContentUI.getSaveVerifier().getEntity(sensitivityBeanClass, sensitivityBeanID);
        }

        if (bean != null) {
            FactorWizardUI factorWizardUI = new FactorWizardUI(inputContentUI);
            SensitivityWizardHandler handler = factorWizardUI.getHandler();
            handler.initNewFactor(factorWizardUI, bean, sensitivityMethod);
            factorWizardUI.pack();
            factorWizardUI.setLocationRelativeTo(inputContentUI);
            factorWizardUI.setVisible(true);
        }
        else {
            if (log.isErrorEnabled()) {
                log.error("Can't find bean in current verifier (sensitivityBeanClass = " + sensitivityBeanClass + ", sensitivityBeanID = " + sensitivityBeanID + ")");
            }
        }
    }

    /**
     * Dans le cas d'une creation de population, on doit la creer dans
     * une espèce. On doit rechercher celle qui est sélectionnée dans l'arbre.
     * 
     * @param inputUI input ui
     * @return selected species
     */
    public Species findSpecies(InputUI inputUI) {
        // get selected node
        FisheryTreeHelper treeHelper = inputUI.getContextValue(FisheryTreeHelper.class);
        FisheryTreeNode selectedNode = treeHelper.getSelectedNode();
        if (selectedNode == null) {
            throw new IsisFishRuntimeException("Not selected tree node");
        }

        // look for parent node (Species type)
        String speciesId = null;
        FisheryTreeNode parentNode = selectedNode.getParent();
        if (parentNode.getInternalClass().equals(Species.class)) {
            // cas selection du noeud type "Population"
            speciesId = parentNode.getId();
        }
        else if (parentNode.getParent().getInternalClass().equals(Species.class)) {
            // cas où on est deja sur une population
            speciesId = parentNode.getParent().getId();
        }
        else {
            throw new IsisFishRuntimeException("Not selected tree node");
        }

        // look for entities
        TopiaContext topiaContext = inputUI.getContextValue(TopiaContext.class);
        Species result = null;
        try {
            SpeciesDAO dao = IsisFishDAOHelper.getSpeciesDAO(topiaContext);
            result = dao.findByTopiaId(speciesId);
        } catch (TopiaException e) {
            throw new IsisFishRuntimeException("Can't find ");
        }
        
        return result;
    }
}
