/*
 * #%L
 * IsisFish
 * 
 * $Id: AspectJUrlClassLoader.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL: http://svn.codelutin.com/isis-fish/trunk/src/main/java/fr/ifremer/isisfish/aspect/AspectJUrlClassLoader.java $
 * %%
 * Copyright (C) 2013 Ifremer, Code Lutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.aspect;

import java.io.IOException;
import java.net.URL;
import java.security.CodeSource;
import java.util.HashMap;
import java.util.Map;

import org.aspectj.bridge.AbortException;
import org.aspectj.weaver.bcel.ExtensibleURLClassLoader;
import org.aspectj.weaver.tools.Trace;
import org.aspectj.weaver.tools.TraceFactory;
import org.aspectj.weaver.tools.WeavingClassLoader;

public class AspectJUrlClassLoader extends ExtensibleURLClassLoader implements WeavingClassLoader {

    public static final String WEAVING_CLASS_PATH = "aj.class.path";
    public static final String WEAVING_ASPECT_PATH = "aj.aspect.path";

    private URL[] aspectURLs;
    private AspectJWeavingAdaptor adaptor;
    private boolean initializingAdaptor;
    private Map generatedClasses = new HashMap(); /* String -> byte[] */

    private static Trace trace = TraceFactory.getTraceFactory().getTrace(AspectJUrlClassLoader.class);

    public AspectJUrlClassLoader(URL[] urls, ClassLoader parent) {
        super(urls, parent);
        if (trace.isTraceEnabled())
            trace.enter("<init>", this, new Object[] { urls, parent });
        if (trace.isTraceEnabled())
            trace.exit("<init>");
        
        adaptor = new AspectJWeavingAdaptor(getParent(), this);
    }

    private static String getAspectPath() {
        return System.getProperty(WEAVING_ASPECT_PATH, "");
    }

    private static String getClassPath() {
        return System.getProperty(WEAVING_CLASS_PATH, "");
    }

    protected void addURL(URL url) {
        adaptor.addURL(url);
        super.addURL(url);
    }
    
    /**
     * Override to weave class using WeavingAdaptor
     */
    protected Class defineClass(String name, byte[] b, CodeSource cs) throws IOException {
        if (trace.isTraceEnabled())
            trace.enter("defineClass", this, new Object[] { name, b, cs });
        // System.err.println("? WeavingURLClassLoader.defineClass(" + name + ", [" + b.length + "])");
        byte orig[] = b;
        /* Avoid recursion during adaptor initialization */
        if (!initializingAdaptor) {

            /* Need to defer creation because of possible recursion during constructor execution */
            if (adaptor == null && !initializingAdaptor) {
                //createAdaptor();
            }

            try {
                b = adaptor.weaveClass(name, b, false);
            } catch (AbortException ex) {
                trace.error("defineClass", ex);
                throw ex;
            } catch (Throwable th) {
                trace.error("defineClass", th);
            }
        }
        Class clazz;
        
        // On error, define the original form of the class and log the issue
        try { 
            clazz= super.defineClass(name, b, cs);
        } catch (Throwable th) {
            trace.error("Weaving class problem. Original class has been returned. The error was caused because of: " + th, th);
            clazz= super.defineClass(name, orig, cs);
        }
        if (trace.isTraceEnabled())
            trace.exit("defineClass", clazz);
        return clazz;
    }

    /**
     * Override to find classes generated by WeavingAdaptor
     */
    protected byte[] getBytes(String name) throws IOException {
        byte[] bytes = super.getBytes(name);

        if (bytes == null) {
            // return adaptor.findClass(name);
            return (byte[]) generatedClasses.remove(name);
        }

        return bytes;
    }

    /**
     * Implement method from WeavingClassLoader
     */
    public URL[] getAspectURLs() {
        return aspectURLs;
    }

    public void acceptClass (String name, byte[] classBytes, byte[] weavedBytes) {
        generatedClasses.put(name, weavedBytes);
    }

    public void deploy(Class<?> aspectClass) {
        byte[] result = adaptor.deploy(aspectClass);
        // this classloader need to be implemented in "local first" strategy
        // and aspectized class must be forced into current classloader
        // otherwize, aspectj will complain about missing 'aspectOf' methods
        defineClass(aspectClass.getName(), result, 0, result.length);
    }

    /**
     * Mark classloader as configured with aspects and tell to prepare for
     * weaving.
     * 
     * Must call this method to avoid NPE in aspectj weaver.
     */
    public void prepare() {
        adaptor.prepare();
    }
}
