/*
 * #%L
 * IsisFish
 * 
 * $Id: OffsetReader.java 3798 2012-10-30 10:39:32Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2002 - 2010 Ifremer, CodeLutin, Benjamin Poussin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.logging.io;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.regex.Pattern;

/**
 * Abstract implementation of {@link OffsetReader} giving generic
 * algorithms to extract offsets from a {@link LineReader}.
 *
 * @author chemit
 */
public abstract class OffsetReader {

    /** number of lines managed by the reader */
    protected long nbLines;

    /**
     * Save the offset of the line for a given position
     *
     * @param position current line position
     * @param offset   current offset of line
     * @throws IOException if any problem while storing
     */
    protected abstract void storeOffset(long position, long offset) throws IOException;

    /**
     * @return <code>true</code> if the reader need to create his internal
     *         datas.
     * @throws IOException if any problme while creating
     */
    protected abstract boolean needCreate() throws IOException;


    /**
     * Obtain an offset for a position
     *
     * @param position position of the line
     * @return the offset of the first car of the given line
     * @throws java.io.IOException todo
     */
    public abstract long getOffset(long position) throws IOException;

    /**
     * Open the reader
     *
     * @param reader LineReader linked with this reader
     * @throws IOException if any problem while opening
     */
    public void open(LineReader reader) throws IOException {
        if (needCreate()) {
            if (reader.getParent() != null) {
                // first use, create the offsets file from parent one
                this.nbLines = createOffsets(reader, reader.getParent());
            } else {
                // first use, create the offsets file
                this.nbLines = createOffsets(reader);
            }
        }
    }

    /**
     * Close the reader
     *
     * @throws java.io.IOException if any problem while closing
     */
    public void close() throws IOException {
    }

    /** @return the number of lines registered in this reader */
    public long getNbLines() {
        return nbLines;
    }

    /**
     * Update the reader
     *
     * @param lineReader lineReader linked with this reader
     * @throws IOException if any problem while updating
     */
    public synchronized void update(LineReader lineReader) throws IOException {

        long newLength = lineReader.getFile().length();
        //long newModified = lineReader.getFile().lastModified();

        lineReader.close();

        // goto end of file (from old length)
        RandomAccessFile reader = new RandomAccessFile(lineReader.getFile(), "r");
        long nbNewLines = 0;
        String line;
        try {
            long offset = newLength;
            reader.seek(offset);

            while ((line = reader.readLine()) != null) {

                if (lineReader.match(line)) {
                    nbNewLines++;
                    // offset of the current line is offset
                    storeOffset(nbLines + nbNewLines, offset);
                }

                int lineSize = line.length();
                offset += lineSize;

                // must find how many cars after the last car of the line
                reader.seek(offset);

                int next = reader.read();

                int nbCharAtEnd = 1;
                if (next == '\r') {
                    // we have a '\r'
                    next = reader.read();
                    if (next == '\n') {
                        nbCharAtEnd = 2;
                    }
                }
                // increment new offset
                offset += nbCharAtEnd;
                reader.seek(offset);
            }
        } finally {
            reader.close();
        }
    }

    private static Pattern LOG_ENTRY_PATTERN = Pattern.compile("(WARN|ERROR|INFO|DEBUG|TRACE).*");

    protected boolean isLogEntry(String line) throws IOException {
        boolean result = false;
        int index = line.indexOf('|');
        if (index == -1) {
            return false;
        }
        result = LOG_ENTRY_PATTERN.matcher(line.substring(0, index)).matches();
        return result;

    }

    /**
     * read a line from reader including terminaison caracters
     *
     * @param reader reader to use
     * @return the line with the terminaison
     * @throws IOException if any problem while reading
     */
    protected String readLine(BufferedReader reader) throws IOException {
        StringBuilder sb = new StringBuilder();
        int c;
        while (true) {
            c = reader.read();
            switch (c) {
                case -1:
                    // end of file
                    return sb.toString();
                case '\n':
                    // end of line
                    sb.append((char) c);
                    return sb.toString();
                default:
                    sb.append((char) c);
            }
        }
    }

    protected long createOffsets(LineReader lineReader) throws IOException {

        BufferedReader reader = new BufferedReader(new FileReader(lineReader.getFile()));
        long nbLines = 0;

        try {
            long offset = 0;
            String line;
            while (!(line = readLine(reader)).isEmpty()) {
                int lineSize = line.length();
                if (lineReader.match(line)) {
                    storeOffset(nbLines, offset);
                    nbLines++;
                }
                offset += lineSize;
            }
        } finally {
            reader.close();
        }
        return nbLines;
    }

    protected long createOffsetsForLog(LineReader lineReader) throws IOException {

        BufferedReader reader = new BufferedReader(new FileReader(lineReader.getFile()));
        long nbLines = 0;

        try {
            long offset = 0;
            String line;
            StringBuilder currentEntry = null;
            while (!(line = readLine(reader)).isEmpty()) {
                if (!isLogEntry(line)) {
                    if (currentEntry==null) {
                        //skip!
                        continue;
                    }
                    System.out.println("no a log entry "+line);
                    currentEntry.append(line);
                    continue;
                }
                if (currentEntry != null) {
                    // save the old entry
                    String toSave = currentEntry.toString();
                    System.out.println("[" + nbLines + "] save entry " + line);
                    int lineSize = toSave.length();
                    if (lineReader.match(toSave)) {
                        storeOffset(nbLines, offset);
                        nbLines++;
                    }
                    offset += lineSize;
                }
                // begin a new entry
                currentEntry = new StringBuilder(line);
            }
        } finally {
            reader.close();
        }
        return nbLines;
    }

    protected long createOffsets(LineReader lineReader, LineReader parent) throws IOException {
        long nbLines = 0;
        for (long i = 0, max = parent.getNbLines(); i < max; i++) {
            String line = parent.readLine(i);
            if (lineReader.match(line)) {
                // store the match
                long offset = parent.getOffsetReader().getOffset(i);
                storeOffset(nbLines, offset);
                nbLines++;
            }
        }
        return nbLines;
    }

    /** delete offset resources */
    public void deleteOffsetFile() {
        // no file used by default
    }

}
