/*
 * #%L
 * IsisFish
 * 
 * $Id: ExportHelper.java 3798 2012-10-30 10:39:32Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2011 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.export;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.Writer;
import java.util.List;
import java.util.Properties;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;

import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.datastore.StorageHelper;

/**
 * Helper for exports manipulation.
 * 
 * Created: 20 janv. 2006 01:52:04
 *
 * @author poussin
 * @version $Revision: 3798 $
 *
 * Last update: $Date: 2012-10-30 11:39:32 +0100 (Tue, 30 Oct 2012) $
 * by : $Author: echatellier $
 */
public class ExportHelper extends StorageHelper {

    /** Class logger. */
    private static Log log = LogFactory.getLog(ExportHelper.class);

    /**
     * Permet de faire les exports pour une simulation.
     * 
     * @param simulation La simulation pour lequel il faut faire les exports
     * @param destdir le repertoire dans lequel il faut ecrire les exports
     * @param exports les instances des exports à faire
     * @param compileDir le nom du répertoire ou les classes d'export sont compilées
     */
    public static void doExport(SimulationStorage simulation, File destdir,
            List<Export> exports, File compileDir) {

        // on ne compte plus ici les temps d'instanciations
        // deplacer dans SimulationExportResultWrapper#afterSimulation(SimulationContext)
        
        long timeStart = System.currentTimeMillis();
        for (Export export : exports) {
            String exportName = ExportStorage.getName(export);
            long time = System.currentTimeMillis();
            try {
                exportToFile(simulation, destdir, export);
            } catch (Exception eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't export object: " + exportName, eee);
                }
            }
            simulation.getInformation().addExportTime(exportName,
                    System.currentTimeMillis() - time);
        }
        simulation.getInformation().addExportTime("All export",
                System.currentTimeMillis() - timeStart);
    }

    /**
     * Do single export.
     * 
     * @param simulation la simulation pour lequel il faut faire les exports
     * @param destdir le repertoire dans lequel il faut ecrire les exports
     * @param export le nom des exports a faire
     * @throws Exception si une erreur survient
     */
    protected static void exportToFile(SimulationStorage simulation,
            File destdir, Export export) throws Exception {
        String filename = export.getExportFilename();
        String extension = export.getExtensionFilename();

        File file = new File(destdir, filename + extension);
        // prevent two export with same name
        // name MyExport.csv become MyExport_1.csv
        int val = 0;
        while (file.exists()) {
            val++;
            file = new File(destdir, filename + extension + "_" + val);
        }

        Writer out = null;
        try {
            out = new BufferedWriter(new FileWriter(file));
            export.export(simulation, out);
        } finally {
            IOUtils.closeQuietly(out);
        }
    }

    /**
     * Recupere dans prop les valeurs des champs specifique au export et met a
     * jour les champs de l'export.
     * 
     * @param exportIndex l'index de l'export
     * @param export l'export a mettre à jour
     * @param context le topia context dont on a besoin
     * @param props les proprietes contenant les parametre de l'export
     */
    public static void populateSensitivityExport(int exportIndex, TopiaContext context,
            SensitivityExport export, Properties props) {
        populateStorageParams(exportIndex, context, export, props, "sensitivityexport");
    }
    
    /**
     * Permet de mettre les parametres de l'export sous une forme String pour
     * pouvoir les relire ensuite.
     *
     * @param sensitivityExportIndex l'index de l'export
     * @param context le topia context dont on a besoin
     * @param sensitivityExport l'export dont on souhaite mettre les parametres dans l'objet
     *        Properties retourné
     * @return L'objet Properties contenant les valeurs des parametres de l'export
     */
    public static Properties getSensitivityExportAsProperties(
            int sensitivityExportIndex, TopiaContext context, SensitivityExport sensitivityExport) {
        return getParamsAsProperties(sensitivityExportIndex, context, sensitivityExport, "sensitivityexport");
    }
}
