/*
 * #%L
 * $Id: ResultHandler.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL: https://svn.codelutin.com/isis-fish/tags/isis-fish-4.3.0.0/src/main/java/fr/ifremer/isisfish/ui/result/ResultHandler.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.result;

import static org.nuiton.i18n.I18n.t;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyVetoException;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.Writer;

import javax.swing.JMenuItem;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.math.matrix.viewer.MatrixViewerPanel;
import org.nuiton.math.matrix.viewer.renderer.MatrixChartRenderer;
import org.nuiton.math.matrix.viewer.renderer.MatrixPanelRenderer;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.FileUtil;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.datastore.ResultStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.datastore.StorageException;
import fr.ifremer.isisfish.entities.FisheryRegion;
import fr.ifremer.isisfish.export.Export;
import fr.ifremer.isisfish.logging.SimulationLoggerUtil;
import fr.ifremer.isisfish.simulator.launcher.SimulationJob;
import fr.ifremer.isisfish.simulator.launcher.SimulationService;
import fr.ifremer.isisfish.simulator.launcher.SimulationServiceListener;
import fr.ifremer.isisfish.ui.CommonHandler;
import fr.ifremer.isisfish.ui.models.common.GenericComboModel;

/**
 * Handler for result UIs.
 * 
 * @author chatellier
 * @version $Revision: 3969 $
 * 
 * Last update : $Date: 2014-04-17 18:48:13 +0200 (Thu, 17 Apr 2014) $
 * By : $Author: echatellier $
 */
public class ResultHandler extends CommonHandler {

    private static Log log = LogFactory.getLog(ResultHandler.class);

    protected SimulationServiceListener listener = null;

    /**
     * Initialise la vue avec les liste des simulations disponible.
     * 
     * @param resultUI
     */
    public void init(final ResultUI resultUI) {
        initSimulationList(resultUI);

        // reference must stay allocated
        listener = new SimulationServiceListener() {
            @Override
            public void simulationStart(SimulationService simService, SimulationJob job) {
            }
            @Override
            public void simulationStop(SimulationService simService, SimulationJob job) {
                GenericComboModel<String> model = (GenericComboModel)resultUI.getSimulationComboBox().getModel();
                model.addElement(job.getId());
            }
            @Override
            public void clearJobDone(SimulationService simService) {
            }
        };
        SimulationService.getService().addSimulationServiceListener(listener);
    }

    /**
     * Reload simulation combo box model.
     * 
     * @param resultUI resultUI
     */
    protected void initSimulationList(ResultUI resultUI) {
        java.util.List<String> value = SimulationStorage.getSimulationNames();
        
        //resultUI.getSimulationComboBox().setEditable(false);
        GenericComboModel model = new GenericComboModel(value);
        resultUI.getSimulationComboBox().setModel(model);
        resultUI.getSimulationComboBox().setSelectedItem(null);
    }

    /**
     * Open selected simulation item in new internal frame.
     * 
     * @param resultUI resultUI
     */
    public void openNewSimulation(ResultUI resultUI) {
        String selected = (String)resultUI.getSimulationComboBox().getSelectedItem();
        SimulationStorage simulation = SimulationStorage.getSimulation(selected);

        ResultSimulationFrame internalFrame = new ResultSimulationFrame(resultUI);
        internalFrame.setTitle(simulation.getName());
        internalFrame.setSimulationStorage(simulation);
        init(internalFrame);
        internalFrame.setSize(800, 600);
        resultUI.getSimulationDesktopPane().add(internalFrame);
        try {
            internalFrame.setSelected(true);
        } catch (PropertyVetoException ex) {
            if (log.isWarnEnabled()) {
                log.warn("Can't auto selected internal frame", ex);
            }
        }
    }

    /**
     * Delete selected simulation.
     * 
     * @param resultUI resultUI
     */
    public void deleteSimulation(ResultUI resultUI) {
        String selected = (String)resultUI.getSimulationComboBox().getSelectedItem();
        SimulationStorage simulation = SimulationStorage.getSimulation(selected);
        try {
            simulation.delete(false);
        } catch (StorageException ex) {
            throw new IsisFishRuntimeException("Can't delete simulation", ex);
        }
        initSimulationList(resultUI);
    }

    /**
     * Display simulation log.
     * 
     * @param resultUI resultUI
     */
    public void showLog(ResultUI resultUI) {
        String selected = (String)resultUI.getSimulationComboBox().getSelectedItem();
        try {
            SimulationLoggerUtil.showSimulationLogConsole(selected);
        } catch (Exception ex) {
            throw new IsisFishRuntimeException("Can't display simulation log", ex);
        }
    }

    /**
     * Init ResultSimulationFrame.
     * 
     * @param resultSimulationFrame resultSimulationFrame
     */
    public void init(final ResultSimulationFrame resultSimulationFrame) {
        SimulationStorage simulation = resultSimulationFrame.getSimulationStorage();
        try {
            // init menu
            for (String exportName : ExportStorage.getExportNames()) {
                JMenuItem item = new JMenuItem(t(exportName));
                item.addActionListener(new ExportActionListener(simulation, exportName));
                resultSimulationFrame.getExportMenu().add(item);
            }

            resultSimulationFrame.topiaContext = simulation.getStorage().beginTransaction();
            final ResultStorage resultStorage = simulation.getResultStorage();
            FisheryRegion fisheryRegion = SimulationStorage.getFisheryRegion(resultSimulationFrame.topiaContext);

            // init viewer panel
            final MatrixViewerPanel matrixViewerPanel = new MatrixViewerPanel();
            matrixViewerPanel.addMatrixDimentionAction(new YearSumDimensionAction());
            matrixViewerPanel.addMatrixRenderer(new MatrixSummaryRenderer(simulation, resultStorage, resultSimulationFrame.topiaContext), true);
            matrixViewerPanel.addMatrixRenderer(new MatrixChartRenderer());
            matrixViewerPanel.addMatrixRenderer(new MatrixMapRenderer(fisheryRegion));
            matrixViewerPanel.addMatrixRenderer(new MatrixPanelRenderer());
            matrixViewerPanel.addMatrixFilter(new ResultMatrixFilter(resultStorage));
            matrixViewerPanel.addMatrixFilter(new SumByYearMatrixFilter());

            // init available results list
            GenericComboModel model = new GenericComboModel(resultStorage.getResultName());
            resultSimulationFrame.getResultsComboBox().setModel(model);
            resultSimulationFrame.getResultsComboBox().addItemListener(new ItemListener() {
                @Override
                public void itemStateChanged(ItemEvent e) {
                    if (e.getStateChange() == ItemEvent.SELECTED) {
                        String selectedMatrixName = (String)e.getItem();
                        MatrixND matrix = resultStorage.getMatrix(selectedMatrixName, resultSimulationFrame.topiaContext);
                        matrixViewerPanel.setMatrix(matrix);
                    }
                }
            });

            // force summary display
            matrixViewerPanel.updateSelectedRenderingComponent();
            resultSimulationFrame.getMatrixViewerContainer().add(matrixViewerPanel, BorderLayout.CENTER);
            
            if (model.getSize() > 0) {
                model.setSelectedItem(model.getElementAt(0));
            }
        } catch (TopiaException ex) {
            throw new IsisFishRuntimeException("Can't open simulation", ex);
        } catch (StorageException ex) {
            throw new IsisFishRuntimeException("Can't open simulation", ex);
        }
    }

    protected class ExportActionListener implements ActionListener {

        protected SimulationStorage simulationStorage;

        protected String exportName;

        public ExportActionListener(SimulationStorage simulationStorage, String exportName) {
            this.simulationStorage = simulationStorage;
            this.exportName = exportName;
        }

        public void actionPerformed(ActionEvent e) {
            try {
                File file = FileUtil.getFile(".+\\.csv", t("isisfish.result.export.file"));
                // add csv extension if not set
                if (!file.getName().endsWith(".csv")) {
                    file = new File(file.getAbsolutePath() + ".csv");
                }
                if (file != null) {
                    Writer out = null;
                    try {
                        out = new BufferedWriter(new FileWriter(file));
                        ExportStorage storage = ExportStorage.getExport(exportName);
                        Export export = storage.getNewInstance();
                        export.export(simulationStorage, out);
                    }
                    finally {
                        IOUtils.closeQuietly(out);
                    }
                }                
            } catch(Exception eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Erreur lors de l'export ", eee);
                }
            }
        }
    }
}
