/*
 * #%L
 * $Id: EntityVariableHandler.java 3713 2012-08-08 15:17:34Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/isis-fish/tags/isis-fish-4.1.0.3/src/main/java/fr/ifremer/isisfish/ui/input/variable/EntityVariableHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.input.variable;

import static org.nuiton.i18n.I18n._;

import java.awt.CardLayout;
import java.util.List;

import javax.swing.JList;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntity;

import fr.ifremer.isisfish.IsisFishDAOHelper;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.entities.Variable;
import fr.ifremer.isisfish.entities.VariableDAO;
import fr.ifremer.isisfish.entities.VariableType;
import fr.ifremer.isisfish.ui.input.model.TopiaEntityListModel;

/**
 * Handler for generic variable ui.
 * 
 * @author chatellier
 * @version $Revision: 3713 $
 * 
 * Last update : $Date: 2012-08-08 17:17:34 +0200 (Wed, 08 Aug 2012) $
 * By : $Author: echatellier $
 */
public class EntityVariableHandler {

    /** Class logger. */
    private static Log log = LogFactory.getLog(EntityVariableHandler.class);

    /**
     * Init view with currently entity variables.
     * 
     * @param view view
     */
    public void initView(EntityVariableUI view) {

        // clear previous
        view.getVariablesList().clearSelection();

        // fill current list
        List<Variable> variables = null;
        TopiaEntityListModel model = (TopiaEntityListModel)view.getVariablesList().getModel();
        TopiaEntity bean = view.getBean();
        if (bean != null) {
            try {
                TopiaContext context = view.getTopiaContext();
                VariableDAO dao = IsisFishDAOHelper.getVariableDAO(context);
                variables = dao.findAllByEntityId(bean.getTopiaId());
            } catch (TopiaException ex) {
                throw new IsisFishRuntimeException("Can't init view", ex);
            }
            
            // try to set entity name in label
            try {
                String name = BeanUtils.getProperty(view.getBean(), "name");
                view.getVariableEntityName().setText(name);
            } catch (Exception ex) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't get entity name", ex);
                }
            }
        } else {
            view.getVariableEntityName().setText("");
        }
        
        model.setEntities(variables);
    }

    /**
     * Add new variable into list.
     * 
     * @param view view
     */
    public void addNewVariable(EntityVariableUI view) {
        try {
            TopiaContext context = view.getTopiaContext();
            VariableDAO dao = IsisFishDAOHelper.getVariableDAO(context);

            Variable variable = dao.create(
                    Variable.PROPERTY_ENTITY_ID, view.getBean().getTopiaId(),
                    Variable.PROPERTY_NAME, _("isisfish.variables.defaultname"));

            TopiaEntityListModel model = (TopiaEntityListModel)view.getVariablesList().getModel();
            List<Variable> variables = (List<Variable>)model.getElements();
            variables.add(variable);
            model.setEntities(variables);

            // auto select
            view.getVariablesList().setSelectedValue(variable, true);
        } catch (TopiaException ex) {
            throw new IsisFishRuntimeException("Can't add variable", ex);
        }
    }

    /**
     * Delete selected variable.
     * 
     * @param view view
     */
    public void deleteVariable(EntityVariableUI view) {
        JList variableList = view.getVariablesList();
        Variable variable = (Variable)variableList.getSelectedValue();

        try {
            // delete in db
            TopiaContext context = view.getTopiaContext();
            VariableDAO dao = IsisFishDAOHelper.getVariableDAO(context);
            dao.delete(variable);
            context.commitTransaction();
            
            // refresh ui
            view.getVariablesList().clearSelection(); // fix event bug
            TopiaEntityListModel model = (TopiaEntityListModel)view.getVariablesList().getModel();
            List<Variable> variables = (List<Variable>)model.getElements();
            variables.remove(variable);
            model.setEntities(variables);
        } catch (TopiaException ex) {
            throw new IsisFishRuntimeException("Can't delete variable", ex);
        }
    }

    /**
     * Display selected variable for edition.
     * 
     * @param view view
     */
    public void showSelectedVariable(EntityVariableUI view) {

        JList variableList = view.getVariablesList();
        Variable variable = (Variable)variableList.getSelectedValue();
        view.setVariable(variable);
        view.getSaveVerifier().addCurrentEntity(variable);

        /*if (variable != null) {
            view.getVariableNameField().setText(variable.getName());
            
            String type = variable.getType();
            view.getVariableTypeCombo().setSelectedItem(type); // fire showSelectedType
            
            if ("double".equals(type)) {
                view.getVariableDoubleValue().setText(String.valueOf(variable.getDoubleValue()));
            } else if ("matrix".equals(type)) {
                //view.get
            } else if ("equation".equals(type)) {
                //cardLayout.show(view.getVariableTypePanel(), "equationtype");
            }
        }*/
    }

    /**
     * Display card layout part associated with selected type.
     * 
     * @param view view
     */
    public void showSelectedType(EntityVariableUI view) {
        CardLayout cardLayout = view.getVariableTypeLayout();
        VariableType type = (VariableType)view.getVariableTypeCombo().getSelectedItem();
        view.getVariable().setType(type);
        switch (type) {
        case DOUBLE:
            cardLayout.show(view.getVariableTypePanel(), "doubletype");
            break;
        case EQUATION:
            cardLayout.show(view.getVariableTypePanel(), "equationtype");
            break;
        case MATRIX:
            cardLayout.show(view.getVariableTypePanel(), "matrixtype");
            break;
        }
    }

    /**
     * Save current edited variable.
     * 
     * @param view view
     */
    public void saveVariable(EntityVariableUI view) {
        JList variableList = view.getVariablesList();
        Variable variable = (Variable)variableList.getSelectedValue();
        variable.setName(view.getVariableNameField().getText().trim());

        VariableType type = (VariableType)view.getVariableTypeCombo().getSelectedItem();

        switch (type) {
        case DOUBLE:
            try {
                double v = Double.parseDouble(view.getVariableDoubleValue().getText().trim());
                variable.setDoubleValue(v);
            } catch (NumberFormatException ex) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't parse double value as double", ex);
                }
            }
            break;
        case MATRIX:
            variable.setMatrixValue(view.getMatrixPanel().getMatrix());
            break;
        } // equation already set by ui component

        //try {
            // save in db
            //TopiaContext context = view.getTopiaContext();
            //VariableDAO dao = IsisFishDAOHelper.getVariableDAO(context);
            //dao.update(variable);
            //context.commitTransaction();
            view.getSaveVerifier().save();

            // refresh ui (name change)
            TopiaEntityListModel model = (TopiaEntityListModel)view.getVariablesList().getModel();
            List<Variable> variables = (List<Variable>)model.getElements();
            model.setEntities(variables);
        /*} catch (TopiaException ex) {
            throw new IsisFishRuntimeException("Can't save variable", ex);
        }*/
    }
}
