/*
 * #%L
 * $Id: FilterableComboBox.java 3565 2011-12-16 15:43:35Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/isis-fish/tags/isis-fish-4.0.0.1/src/main/java/fr/ifremer/isisfish/ui/widget/FilterableComboBox.java $
 * %%
 * Copyright (C) 2011 Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.widget;

import static org.nuiton.i18n.I18n._;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ComboBoxModel;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import org.jdesktop.swingx.JXComboBox;
import org.jdesktop.swingx.JXTextField;
import org.nuiton.util.Resource;

/**
 * A filterable combobox using an additional text field and reset button.
 * 
 * @author chatellier
 * @version $Revision: 3565 $
 * 
 * Last update : $Date: 2011-12-16 16:43:35 +0100 (ven., 16 déc. 2011) $
 * By : $Author: echatellier $
 */
public class FilterableComboBox extends JPanel {

    /** serialVersionUID. */
    private static final long serialVersionUID = -6669320998140538372L;

    protected JComboBox dataBox;
    protected JTextField filterField;
    protected JButton resetButton;

    protected FilterComboBoxModel filterModel;

    protected List<ActionListener> actionListeners;

    protected boolean filtering = false;

    /*
     * Il faut obligatoirement que la liste filtrées soit mise en cache, car
     * l'appeler dans le getElementAt() comme c'était fait avant dégrade énormement
     * les perfs, un truc de dingue.
     */
    protected List<Object> itemCache;

    public FilterableComboBox() {
        actionListeners = new ArrayList<ActionListener>();
        buildLayout();
    }

    public FilterableComboBox(ComboBoxModel model) {
        this();
        setModel(model);
    }

    class FilterComboBoxModel extends DefaultComboBoxModel {

        /** serialVersionUID. */
        private static final long serialVersionUID = 6256289874593024153L;

        protected ComboBoxModel model;

        public FilterComboBoxModel(ComboBoxModel model) {
            this.model = model;
        }

        @Override
        public int getSize() {
            int size = 0;
            if (itemCache != null) {
                size = itemCache.size();
            } else {
                size = model.getSize();
            }
            return size;
        }

        @Override
        public Object getElementAt(int index) {
            Object element = null;
            if (itemCache != null) {
                element = itemCache.get(index);
            } else {
                element = model.getElementAt(index);
            }
            return element;
        }

        public void fireDataChanged() {
            filtering = true;
            super.fireContentsChanged(this, 0, model.getSize());
            
            // filter list
            int delegateSize = model.getSize();
            itemCache = new ArrayList<Object>(delegateSize);
            for (int i = 0; i < delegateSize; ++i) {
                Object element = model.getElementAt(i);
                if (element.toString().matches(".*" + filterField.getText() + ".*")) {
                    itemCache.add(element);
                }
            }
            filtering = false;
        }
        
        
    }
    
    class FilterDocumentListener implements DocumentListener {
        @Override
        public void insertUpdate(DocumentEvent e) {
            updateFilter();
        }

        @Override
        public void removeUpdate(DocumentEvent e) {
            updateFilter();
        }

        @Override
        public void changedUpdate(DocumentEvent e) {
            updateFilter();
        }
        
        protected void updateFilter() {
            ((FilterComboBoxModel)dataBox.getModel()).fireDataChanged();
            resetButton.setEnabled(!filterField.getText().isEmpty());
        }
    }

    class FilterActionListener implements ActionListener {
        /*
         * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            filterField.setText("");
        }
    }

    class ComboBoxActionListener implements ActionListener {
        /*
         * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            if (!filtering) {
                for (ActionListener a : actionListeners) {
                    a.actionPerformed(e);
                }
            }
        }
    }

    /**
     * Build layout.
     */
    private void buildLayout() {
        dataBox = new JXComboBox();
        dataBox.addActionListener(new ComboBoxActionListener());
        filterField = new JXTextField(_("isisfish.common.filter"));
        // fix size
        filterField.setPreferredSize(new Dimension(200, 0));
        resetButton = new JButton(Resource.getIcon("/icons/cancel.png"));
        resetButton.addActionListener(new FilterActionListener());
        resetButton.setEnabled(false);
        filterField.getDocument().addDocumentListener(new FilterDocumentListener());

        setLayout(new BorderLayout());
        add(dataBox, BorderLayout.CENTER);

        JPanel filterPanel = new JPanel(new BorderLayout());
        filterPanel.add(filterField, BorderLayout.CENTER);
        filterPanel.add(resetButton, BorderLayout.EAST);

        add(filterPanel, BorderLayout.EAST);
    }

    /**
     * Change model.
     * 
     * @param model model to set
     */
    public void setModel(ComboBoxModel model) {
        dataBox.setModel(new FilterComboBoxModel(model));
    }

    public Object getSelectedItem() {
        return dataBox.getSelectedItem();
    }
    
    public void setSelectedItem(Object object) {
        dataBox.setSelectedItem(object);
    }

    public boolean addActionListener(ActionListener e) {
        return actionListeners.add(e);
    }

    public boolean removeActionListener(ActionListener o) {
        return actionListeners.remove(o);
    }
}
