/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2005 - 2011 Ifremer, Code Lutin, Cedric Pineau, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.simulator;

import static org.nuiton.i18n.I18n._;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;

import jaxx.runtime.SwingUtil;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.math.matrix.gui.MatrixPanelEditor;
import org.nuiton.topia.TopiaContext;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.datastore.RuleStorage;
import fr.ifremer.isisfish.datastore.ScriptStorage;
import fr.ifremer.isisfish.datastore.SensitivityAnalysisStorage;
import fr.ifremer.isisfish.datastore.SensitivityExportStorage;
import fr.ifremer.isisfish.datastore.SimulationPlanStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.datastore.SimulatorStorage;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.Species;
import fr.ifremer.isisfish.entities.Strategy;
import fr.ifremer.isisfish.export.Export;
import fr.ifremer.isisfish.export.SensitivityExport;
import fr.ifremer.isisfish.mexico.MexicoHelper;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.simulator.SimulationParameter;
import fr.ifremer.isisfish.simulator.SimulationParameterImpl;
import fr.ifremer.isisfish.simulator.SimulationPlan;
import fr.ifremer.isisfish.simulator.launcher.SimulationService;
import fr.ifremer.isisfish.simulator.launcher.SimulatorLauncher;
import fr.ifremer.isisfish.simulator.sensitivity.DesignPlan;
import fr.ifremer.isisfish.simulator.sensitivity.Factor;
import fr.ifremer.isisfish.simulator.sensitivity.FactorGroup;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityAnalysis;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityException;
import fr.ifremer.isisfish.simulator.sensitivity.domain.ContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.DiscreteDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.EquationContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.EquationDiscreteDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.MatrixContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.RuleDiscreteDomain;
import fr.ifremer.isisfish.ui.SimulationUI;
import fr.ifremer.isisfish.ui.sensitivity.FactorWizardUI;
import fr.ifremer.isisfish.ui.sensitivity.SensitivityChooserUI;
import fr.ifremer.isisfish.ui.sensitivity.SensitivityInputHandler;
import fr.ifremer.isisfish.ui.util.ErrorHelper;
import fr.ifremer.isisfish.ui.widget.editor.ScriptParameterDialog;

/**
 * SimulAction.
 *
 * Created: 1 aout 2005 18:37:25 CEST
 *
 * @author Benjamin POUSSIN <poussin@codelutin.com>
 * @version $Revision: 3523 $
 *
 * Last update: $Date: 2011-11-08 15:11:46 +0100 (mar., 08 nov. 2011) $
 * by : $Author: echatellier $
 */
public class SimulAction {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(SimulAction.class);

    private static final SimpleDateFormat DATEFORMAT = new SimpleDateFormat("yyyy-MM-dd-HH-mm");

    /** Les parametres de simulation (commun a tous les onglet de l'interface de simulation). */
    protected SimulationParameter param = null;

    protected RegionStorage regionStorage = null;
    protected SimulationStorage simulStorage = null;
    protected List<String> oldSimulNames = null;
    protected String simulName = null;
    protected SensitivityAnalysisStorage sensitivityStorage = null;

    /**
     * List de facteur sous forme d'arbre (factor group).
     * {@code null} name for compatibility with 3.3.0.0.
     */
    protected FactorGroup factorGroup = new FactorGroup(null);

    public SimulAction() {
        init();
    }

    /**
     * cree et initialise param avec les valeurs par defaut de l'utilisateur
     */
    public void init() {
        if (log.isDebugEnabled()) {
            log.debug("Init ");
        }
        try {
            param = new SimulationParameterImpl();
            List<String> resultNames = getResultNames();
            // put default value in param
            param.setSimulatorName(IsisFish.config.getSimulatorClassfile());
            param.setTagValue(IsisFish.config.getDefaultTagValueAsMap());

            param.setExportNames(IsisFish.config.getDefaultExportNamesAsList());
            List<String> defaultResultNames = IsisFish.config
                    .getDefaultResultNamesAsList();
            if (defaultResultNames != null) {
                param.setResultEnabled(defaultResultNames);
            } else {
                param.setResultEnabled(resultNames);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't init SimulationParameter", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.initsimulaction"), eee);
        }
    }

    protected void setName(String name) {
        simulName = name;
    }

    /**
     * Load simulation parameter file.
     * 
     * @param f
     * 
     * TODO public just for sensitivity
     */
    public void importSimulation(File f) {
        FileInputStream fos = null;
        try {
            fos = new FileInputStream(f);
            Properties proper = new Properties();
            proper.load(fos);
            param.fromProperties(proper);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't import simulation", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.importparameter"), e);
        } finally {
            IOUtils.closeQuietly(fos);
        }
    }

    /**
     * Save current simulation to given file.
     * 
     * @param f file to save simulation to
     * 
     * TODO public just for sensitivity
     */
    public void saveSimulation(File f) {
        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(f);
            param.toProperties().store(fos, simulName);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't save simulation", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.savesimulation"), e);
        } finally {
            IOUtils.closeQuietly(fos);
        }
    }

    /**
     * Looking for ResultName.java script and read all fields and fill
     * list of field value and doc
     * @return the list of result names found in ResultName script
     */
    protected List<String> getResultNames() {
        List<String> result = new ArrayList<String>();
        try {
            ScriptStorage storage = ScriptStorage.getScript("ResultName.java");
            Class<?> resultNameClass = storage.getCodeClass();
            // TODO: if we can't find class, perhaps try to parser Java source ?
            Field[] fields = resultNameClass.getFields();
            for (Field f : fields) {
                String fieldName = (String) f.get(null);
                result.add(fieldName);
            }
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't find result name script", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.getresultname"), eee);
        }
        return result;
    }

    /**
     * Change region in simulation launcher
     * 
     * @param paramsUI paramsUI
     * @param regionName region name
     */
    public void regionChange(ParamsUI paramsUI, String regionName) {
        regionStorage = RegionStorage.getRegion(regionName);
        paramsUI.getParentContainer(SimulationUI.class).setContextValue(regionStorage);
        paramsUI.getParentContainer(SimulationUI.class).setRegionStorage(regionStorage);
        param.setRegionName(regionName);
    }

    /**
     * Load old simulation.
     * 
     * Reset some field to empty default values:
     * <ul>
     *  <li>params</li>
     *  <li>simulation plans</li>
     *  <li>factors list</li>
     * <ul>
     * 
     * Open old simulation:
     * <ul>
     *  <li>params copy</li>
     *  <li>factors</li>
     * </ul>
     * 
     * @param simulName name of simulation to load
     */
    public void loadOldSimulation(String simulName) {
        if (log.isDebugEnabled()) {
            log.debug("call loadOldSimulation: " + simulName);
        }
        try {
            this.simulName = simulName;
            simulStorage = SimulationStorage.getSimulation(simulName);
            param = simulStorage.getParameter().copy();
            // all time reset number after load
            param.setSimulationPlanNumber(-1);
            regionStorage = param.getRegion();

            // Chargement des facteurs
            // clear list even if mexico file doesn't exists
            factorGroup.clearFactors();
            File f = SimulationStorage.getMexicoDesignPlan(SimulationStorage.getSimulationDirectory(simulName));
            if (f != null && f.canRead()) {
                if (log.isInfoEnabled()) {
                    log.info("Import design plan from : " + f.getAbsolutePath());
                }
                TopiaContext topiaContext = regionStorage.getStorage();
                DesignPlan designPlan = MexicoHelper.getDesignPlanFromXML(f, topiaContext);
                for (Factor factor : designPlan.getFactors()) {
                    if (log.isDebugEnabled()) {
                        log.debug("Find factor : " + factor.getName());
                    }
                    //factors.put(factor.getPath() + factor.getName(), factor);
                    factorGroup = designPlan.getFactorGroup();
                }
            }
            else {
                if (log.isInfoEnabled()) {
                    log.info("No xml design plan file found");
                }
            }
        } catch (Exception eee) {
            throw new IsisFishRuntimeException(_("isisfish.error.simulation.loadoldsimulation"), eee);
        }
    }

    /**
     * Called by RuleChooser component before rule deletion.
     * Used to remove factor associated to rule to delete.
     * 
     * Factor path reference rule with factor path containing rule index
     * in rule list :
     * for example :
     * <pre>
     * parameters.rule.2.parameter.tacPoids
     * </pre>
     * 
     * Must also rename all next indices.
     * 
     * @param ruleIndex rule index to to delete
     */
    public void preRemoveRule(int ruleIndex) {
        preRemoveRule(factorGroup, ruleIndex);
    }

    /**
     * Recursive rename and delete rule factor path.
     * 
     * @param ruleIndex rule index to to delete
     */
    protected void preRemoveRule(FactorGroup factorGroup, int ruleIndex) {
        Collection<Factor> factorCopy = new ArrayList<Factor>(factorGroup.getFactors());
        for (Factor factor : factorCopy) {
            if (factor instanceof FactorGroup) {
                preRemoveRule((FactorGroup)factor, ruleIndex);
            }
            else {
                Pattern factorPathPattern = Pattern.compile("^(parameters\\.rule\\.)(\\d+)(.*)$");
                Matcher factorPathMatcher = factorPathPattern.matcher(factor.getPath());
                if (factorPathMatcher.find()) {
                    Integer index = Integer.parseInt(factorPathMatcher.group(2));
                    if (index == ruleIndex) {
                        // meme index, suppression
                        if (log.isDebugEnabled()) {
                            log.debug("Removing factor for index " + ruleIndex + " : " + factor.getPath());
                        }
                        factorGroup.remove(factor);
                    }
                    else if (index > ruleIndex) {
                        // index supérieur, renommage
                        // avec un index de moins
                        String factorPath = factorPathMatcher.group(1) +
                                String.valueOf(index - 1) + factorPathMatcher.group(3);
                        if (log.isDebugEnabled()) {
                            log.debug("Renammed factor for index " + ruleIndex + " : " + factor.getPath());
                        }
                        factor.setPath(factorPath);
                    }
                }
            }
        }
    }
    /**
     * Get strategies list to fill Jlist in ParamUI.
     * 
     * @return strategies list
     */
    public List<Strategy> getStrategies() {
        List<Strategy> result = new ArrayList<Strategy>();
        try {
            TopiaContext tx = param.getRegion().getStorage().beginTransaction();
            result = RegionStorage.getFisheryRegion(tx).getStrategy();
            tx.rollbackTransaction();
            tx.closeContext();
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get strategies", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.liststrategies"), e);
        }
        return result;
    }

    /**
     * Set parameters strategies.
     * 
     * @param strategies
     */
    public void setStrategies(Object[] strategies) {
        List<Strategy> result = new ArrayList<Strategy>();
        for (Object o : strategies) {
            result.add((Strategy) o);
        }
        param.setStrategies(result);
    }

    /**
     * Get population list to fill JList in paramUI.
     * 
     * @return populations list
     */
    public List<Population> getPopulations() {

        List<Population> result = new ArrayList<Population>();
        try {
            TopiaContext tx = param.getRegion().getStorage().beginTransaction();
            List<Species> species = RegionStorage.getFisheryRegion(tx)
                    .getSpecies();
            for (Species s : species) {
                Collection<Population> populations = s.getPopulation();

                // FIXME initialiaze lazy hibernate collection
                for (Population p : populations) {
                    p.getPopulationGroup().size();
                    p.getPopulationZone().size();
                }

                result.addAll(populations);
            }
            tx.rollbackTransaction();
            tx.closeContext();
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get population", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.listpopulation"), e);
        }
        return result;
    }

    /**
     * Set parameter population.
     * 
     * @param populations populations to set
     */
    public void setPopulations(Object[] populations) {
        List<Population> result = new ArrayList<Population>();
        for (Object o : populations) {
            result.add((Population) o);
        }
        param.setPopulations(result);
    }

    // Years
    public int getNumberOfYear() {
        return param.getNumberOfYear();
    }

    /**
     * Change number of simulation year by parsing string value.
     * Default to 1 if value is not parsable.
     * 
     * @param years number of years to set
     */
    public void setNumberOfYear(String years) {
        try {
            param.setNumberOfYear(Integer.parseInt(years));
        }
        catch (NumberFormatException ex) {
            // defaut to one year
            param.setNumberOfYear(1);
        }
    }

    public List<String> getSimulatorNames() {
        return SimulatorStorage.getSimulatorNames();
    }

    /**
     * Return old simulations.
     *
     * @return old simulations
     */
    public List<String> getOldSimulationItem() {

        //try {
            oldSimulNames = new ArrayList<String>();

            //FilterModel<SimulationProperties, String> filterModel;
            // keep in context list of old simulation names (for filter process)
            oldSimulNames.addAll(SimulationStorage.getSimulationNames());
            // create filter model
            //filterModel = SimulationFilterUtil.createFilterModel(oldSimulNames);
            // to used directly model.getFilteredResult() in xml
            // we must fill filterModel result with original items
            //filterModel.selectAll();

            // chatellier : Ajout d'une entrée vide pour forcer la sélection
            // et que le changement lance un evenement sur la liste
            // Add empty item after, otherwise, un " " directory
            // will be created
            oldSimulNames.add(0, " ");

            return oldSimulNames;
        /*} catch (ParseException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get old simulation item", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.listoldsimulation"), e);
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get old simulation item", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.listoldsimulation"), e);
        }
        return null;*/
    }

    public List<String> getFilteredOldSimulatorNames(boolean force) {
        if (oldSimulNames == null || force) {
            oldSimulNames = getOldSimulationItem();
        }
        return oldSimulNames;
    }

    public void setOldSimulatorNames(List<String> sn) {
        oldSimulNames = sn;
    }

    public void resetOldSimulatorNames() {
        oldSimulNames = getOldSimulationItem();
    }

    /*
     ************
     * AdvancedParameterUI
     ************
     */

    // Gestion des TagValues
    public void addTagValue(String tag, String value) {
        param.getTagValue().put(tag, value);
    }

    public void removeTagValue(String tag) {
        if (log.isDebugEnabled()) {
            log.debug("removeTagValue: " + tag);
        }
        param.getTagValue().remove(tag);
    }

    public void saveTagValue(String simulatorName) {
        Map<String, String> tagValues = param.getTagValue();
        if (log.isDebugEnabled()) {
            log.debug("call saveTagValue: " + tagValues);
        }
        IsisFish.config.setDefaultTagValues(tagValues);

        IsisFish.config.setSimulatorClassfile(simulatorName);
    }

    /*
     * ExportUI
     */

    public List<String> getExportNames() {
        List<String> result = new ArrayList<String>();
        List<String> exportNames = ExportStorage.getExportNames();
        for (String export : exportNames) {
            result.add(export);
        }
        return result;
    }

    /**
     * Save current parameters exports names in configuration.
     */
    public void saveExport() {

        if (param.getExportNames() != null) {
            
            if (log.isDebugEnabled()) {
                log.debug("Set exports in configuration : " + param.getExportNames());
            }
            
            IsisFish.config.setDefaultExportNames(param.getExportNames());
        }
    }

    /*
     * ResultUI
     */

    public void saveParametersResultNames(Object[] resultNames) {
        List<String> resultNamesString = new ArrayList<String>();
        for (Object o : resultNames) {
            resultNamesString.add(o.toString());
        }
        param.setResultEnabled(resultNamesString);
        
        if (log.isDebugEnabled()) {
            log.debug("Set simulation result names : " + resultNamesString);
        }
    }

    public void saveConfigurationResultNames(Object[] resultNames) {
        List<String> resultNamesString = new ArrayList<String>();
        for (Object o : resultNames) {
            resultNamesString.add(o.toString());
        }
        IsisFish.config.setDefaultResultNames(resultNamesString);
        
        if (log.isDebugEnabled()) {
            log.debug("Set configuration result names : " + resultNamesString);
        }
    }

    public List<String> getDefaultResultNames() {
        return IsisFish.config.getDefaultResultNamesAsList();
    }

    /**
     * Return simulation plan names list.
     * 
     * @return simulation plan names
     */
    public List<String> getSimulationPlanNames() {
        return SimulationPlanStorage.getSimulationPlanNames();
    }

    public List<SimulationPlan> getParamSimulationPlans() {
        return param.getSimulationPlans();
    }

    public void addSimulationPlan(SensUI sensUI, String name) {
        try {
            SimulationPlan sp = SimulationPlanStorage.getSimulationPlan(name).getNewSimulationPlanInstance();
            
            // add it after autoconfiguration (if enabled)
            sp = (SimulationPlan)ScriptParameterDialog.displayConfigurationFrame(sensUI, sp);
            if (sp != null) {
                getSimulationParameter().addSimulationPlan(sp);
            }
        } catch (IsisFishException ex) {
            throw new IsisFishRuntimeException("Can't add simulation plan", ex);
        }
    }

    public void removeSimulationPlan(SimulationPlan sp) {
        getSimulationParameter().removeSimulationPlan(sp);
    }

    public void clearSimulationPlans() {
        getSimulationParameter().clearPlans();
    }

    public Map<String, Class<?>> getSimulationPlanParameterName(SimulationPlan sp) {
        Map<String, Class<?>> result = null;
        if (sp != null) {
            result = SimulationPlanStorage.getParameterNames(sp);
        }
        return result;
    }

    public Object getSimulationPlanParameterValue(String paramName, SimulationPlan sp) {
        Object result = null;
        if (sp != null) {
            try {
                result = SimulationPlanStorage.getParameterValue(sp, paramName);
            } catch (IsisFishException ex) {
                throw new IsisFishRuntimeException("Can't get simulation plan param value", ex);
            }
        }
        return result;
    }

    public void setSimulationPlanParameterValue(String paramName, SimulationPlan sp,
            Object value) {
        if (log.isDebugEnabled()) {
            log.debug("paramName : " + paramName + " simulationPlanName : " + sp
                    + " value : " + value);
        }
        try {
            SimulationPlanStorage.setParameterValue(sp, paramName, value);
        } catch (IsisFishException ex) {
            throw new IsisFishRuntimeException("Can't set simulation plan param value", ex);
        }
    }

    /*
     * Sensitivity
     */
    public List<String> getSensitivityExportNames() {
        List<String> result = new ArrayList<String>();
        List<String> exportNames = SensitivityExportStorage.getSensitivityExportNames();
        for (String export : exportNames) {
            result.add(export);
        }
        return result;
    }

    public List<SensitivityExport> getSensitivityExports() {
        List<SensitivityExport> result = param.getSensitivityExport();
        return result;
    }

    public void addSensitivityExport(SensitivityChooserUI sensitivityChooserUI, String name) {
        try {
            SensitivityExportStorage storage = SensitivityExportStorage.getSensitivityExport(name);
            SensitivityExport sensitivityExport = storage.getNewSensitivityExportInstance();
            
            // add it after autoconfiguration (if enabled)
            sensitivityExport = (SensitivityExport)ScriptParameterDialog.displayConfigurationFrame(sensitivityChooserUI, sensitivityExport);
            if (sensitivityExport != null) {
                param.getSensitivityExport().add(sensitivityExport);
            }
        } catch (IsisFishException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't add sensitivity export", e);
            }
        }
    }

    /**
     * Remove an export.
     * 
     * @param export export to remove
     */
    public void removeSensitivityExport(SensitivityExport export) {
        param.getSensitivityExport().remove(export);
    }

    public void clearSensitivityExport() {
        param.getSensitivityExport().clear();
    }

    public Map<String, Class<?>> getSensitivityExportParameterNames(
            SensitivityExport export) {
        return SensitivityExportStorage.getParameterNames(export);
    }

    public void setSensitivityExportParameterValue(Export export,
            String paramName, Object value) {
        if (value != null) {
            try {
                SensitivityExportStorage.setParameterValue(export, paramName, value);
            } catch (IsisFishException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't set sensitivity export param value", e);
                }
            }
        }
    }

    public Object getSensitivityExportParameterValue(SensitivityExport sensitivityExport, String paramName) {
        Object result = null;
        try {
            result = SensitivityExportStorage.getParameterValue(sensitivityExport, paramName);
        } catch (IsisFishException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get sensitivity export param value", e);
            }
        }
        return result;
    }
    
    /**
     * Return sensitivity analysis name without .java extension.
     * 
     * @return sensitivity analysis names list
     */
    public List<String> getSensitivityAnalysisNames() {
        List<String> result = new ArrayList<String>();
        for (String r : SensitivityAnalysisStorage.getSensitivityAnalysisNames()) {
            // there is some non java files in sensitivity directory
            if (r.endsWith(".java")) {
                // Remove .java extention
                // for example SensitivityStorage.getRuleName(String)
                result.add(r.substring(0, r.length() - 5));
            }
        }
        return result;
    }

    /**
     * Get current sensitivity calculator instance.
     * @return sensitivity calculator
     */
    public SensitivityAnalysis getSensitivityAnalysis() {
        return param.getSensitivityAnalysis();
    }

    /**
     * Build a new sensitivity calculator instance by his name.
     * 
     * @param name calculator name
     * @return instance
     */
    public SensitivityAnalysis getSensitivityAnalysisInstance(String name) {
        SensitivityAnalysis sensitivityAnalysis = null;
        try {
            sensitivityStorage = SensitivityAnalysisStorage.getSensitivityAnalysis(name);
            sensitivityAnalysis = sensitivityStorage
                    .getNewSensitivityAnalysisInstance();
        } catch (IsisFishException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't set sensitivity analysis", e);
            }
        }
        return sensitivityAnalysis;
    }

    /**
     * Set calculator instance to use.
     * 
     * @param sensitivityAnalysis new instance
     */
    public void setSensitivityAnalysis(SensitivityAnalysis sensitivityAnalysis) {
        param.setSensitivityAnalysis(sensitivityAnalysis);
    }

    /**
     * Get current simulation factor list.
     * 
     * @return factor list
     */
    public FactorGroup getFactorGroup() {
        return factorGroup;
    }
    
    /**
     * Search factor in factor group tree by path.
     * 
     * @param factorPath factor path to search
     * @return found factor
     */
    public Factor getFactor(String factorPath) {
        return getFactor(factorGroup, factorPath);
    }

    /**
     * Recursive search for factor in factor group by path.
     * 
     * @param factorGroup factor group to search to
     * @param factorPath factor path to search
     * @return found factor
     */
    protected Factor getFactor(FactorGroup factorGroup, String factorPath) {
        Factor result = null;
        for (Factor factor : factorGroup.getFactors()) {
            if (factor instanceof FactorGroup) {
                result = getFactor((FactorGroup)factor, factorPath);
            }
            if (factorPath.equals(factor.getPath())) {
                result = factor;
            }
        }
        return result;
    }
    
    /**
     * Remove factor in factor group tree by path.
     * 
     * @param factorPath factor path to remove
     */
    public void removeFactor(String factorPath) {
        removeFactor(factorGroup, factorPath);
    }

    /**
     * Recursive remove for factor in factor group by path.
     * 
     * @param factorGroup factor group to search to
     * @param factorPath factor path to remove
     */
    protected void removeFactor(FactorGroup factorGroup, String factorPath) {
        Collection<Factor> factorCopy = new ArrayList<Factor>(factorGroup.getFactors());
        for (Factor factor : factorCopy) {
            if (factor instanceof FactorGroup) {
                removeFactor((FactorGroup)factor, factorPath);
            }
            if (factorPath.equals(factor.getPath())) {
                factorGroup.remove(factor);
            }
        }
    }

    public void addFactor(Factor f) {
        if (log.isDebugEnabled()) {
            log.debug("Add factor (" + f.getName() + ") : " +f.getPath());
        }
        factorGroup.addFactor(f);
    }

    /**
     * Ajout d'un facteur continue de type (min/max).
     * 
     * @param name
     * @param comment
     * @param path
     * @param min
     * @param max
     * @param exist
     */
    public void addContinuousFactor(String name, String comment, String path, Double min,
            Double max, boolean exist) {
        Factor f = new Factor(name);
        ContinuousDomain domain = new ContinuousDomain();
        domain.setMinBound(min);
        domain.setMaxBound(max);
        f.setDomain(domain);
        f.setComment(comment);
        f.setPath(path);
        if (exist) {
            removeFactor(path);
        }
        addFactor(f);
    }

    /**
     * Ajout d'un facteur continue de type pourcentage.
     * 
     * @param name
     * @param comment
     * @param path
     * @param referenceValue
     * @param coefficient
     * @param exist
     */
    public void addContinuousPercentageFactor(String name, String comment, String path, Double referenceValue,
            Double coefficient, boolean exist) {
        Factor f = new Factor(name);
        ContinuousDomain domain = new ContinuousDomain(true);
        domain.setReferenceValue(referenceValue);
        domain.setCoefficient(coefficient);
        f.setDomain(domain);
        f.setComment(comment);
        f.setPath(path);
        if (exist) {
            removeFactor(path);
        }
        addFactor(f);
    }

    public void addDiscreteFactor(String name, String comment, String path,
            List<Object> values, boolean exist) {
        addDiscreteFactor(new DiscreteDomain(), name, comment, path, values, exist);
    }
    
    public void addDiscreteRuleFactor(String name, String comment, String path,
            List<Object> values, boolean exist) {
        addDiscreteFactor(new RuleDiscreteDomain(), name, comment, path, values, exist);
    }
    
    public void addDiscreteEquationFactor(String name, String comment, String path,
            List<Object> values, boolean exist) {
        addDiscreteFactor(new EquationDiscreteDomain(), name, comment, path, values, exist);
    }
    
    protected void addDiscreteFactor(DiscreteDomain domain, String name, String comment, String path,
            List<Object> values, boolean exist) {
        Factor f = new Factor(name);
        SortedMap<Object, Object> domainValues = new TreeMap<Object, Object>();
        int label = 0;
        for (Object value : values) {
            // FIXME test when integer
            // Don't work with String ;(
            domainValues.put(label, value);
            // and start at 0
            label++;
        }
        domain.setValues(domainValues);
        f.setDomain(domain);
        f.setComment(comment);
        f.setPath(path);
        if (exist) {
            removeFactor(path);
        }
        addFactor(f);
    }

    public void addContinuousEquationFactor(String name, String comment, String path,
            EquationContinuousDomain domain, boolean exist) {
        // factor name need to be composed
        Factor f = new Factor(name + "." + domain.getVariableName());
        f.setDomain(domain);
        f.setComment(comment);
        // don't modify path, need to be valid identifier
        //f.setPath(path + domain.getVariableName());
        f.setPath(path);
        if (exist) {
            removeFactor(path);
        }
        addFactor(f);
    }

    public void addContinuousMatrixFactor(String name, String comment, String path,
            MatrixND referenceValue, Double coef, boolean exist) {
        Factor f = new Factor(name);
        MatrixContinuousDomain domain = new MatrixContinuousDomain(true);
        domain.setReferenceValue(referenceValue);
        domain.setCoefficient(coef);
        f.setComment(comment);
        f.setDomain(domain);
        f.setPath(path);
        if (exist) {
            removeFactor(path);
        }
        addFactor(f);
    }
    
    public void addContinuousMatrixFactor(String name, String comment, String path,
            MatrixND minBound, MatrixND maxBound, boolean exist) {
        Factor f = new Factor(name);
        MatrixContinuousDomain domain = new MatrixContinuousDomain();
        domain.setMinBound(minBound);
        domain.setMaxBound(maxBound);
        f.setComment(comment);
        f.setDomain(domain);
        f.setPath(path);
        if (exist) {
            removeFactor(path);
        }
        addFactor(f);
    }

    /*
     ************
     * General
     ************
     */

    public SimulationParameter getSimulationParameter() {
        return param;
    }

    public void setSimulationParameter(SimulationParameter p) {
        param = p;
    }

    public SimulationStorage getSimulationStorage() {
        return simulStorage;
    }

    public RegionStorage getRegionStorage() {
        return regionStorage;
    }

    public List<SimulatorLauncher> getSimulationLauncher() {
        return SimulationService.getService().getSimulationLaunchers();
    }

    /**
     * Launch automatically the simulation, when is possible (no other simulation)
     * or wait for the last automatically simulation ended.
     *
     * @param simulationId id of the simulation to simulate
     * @param launcher launcher to use
     * 
     * @see SimulatorLauncher
     */
    public void launchSimulation(String simulationId, SimulatorLauncher launcher) {

        String fullSimulationId = "sim_" + simulationId + "_"
                + DATEFORMAT.format(new java.util.Date());

        // log
        if (log.isDebugEnabled()) {
            log.debug("Launch simulation with custom launcher "
                    + launcher.toString());
        }

        try {
            if ("".equals(fullSimulationId)
                    || SimulationStorage.localyExists(fullSimulationId)
                    || SimulationService.getService().exists(fullSimulationId)) {
                ErrorHelper.showErrorDialog(_("isisfish.simulator.simulaction.badid",
                        fullSimulationId), null);
            } else {
                SimulationService.getService().submit(fullSimulationId, param,
                        launcher, 0);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't start simulation", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.launchsimulation"), eee);
        }
    }

    /**
     * Launch simulation with factors variations parameters.
     * 
     * @param simulationId id of the simulation to simulate
     * @param launcher launcher to use
     * @param sensitivityAnalysis sensitivity analysis to use
     * 
     * @see SimulatorLauncher
     * @see SensitivityAnalysis
     * @see DesignPlan
     */
    public void launchSimulation(String simulationId,
            SimulatorLauncher launcher, SensitivityAnalysis sensitivityAnalysis) {

        String fullSimulationId = "as_" + simulationId + "_"
                + DATEFORMAT.format(new java.util.Date());

        // log
        if (log.isDebugEnabled()) {
            log.debug("Launch factor simulation with custom launcher "
                    + launcher.toString());
            log.debug("Using sensitivityCalculator : "
                    + sensitivityAnalysis.getDescription());
        }

        try {
            if ("".equals(fullSimulationId)
                    || SimulationStorage.localyExists(fullSimulationId)
                    || SimulationService.getService().exists(fullSimulationId)) {
                ErrorHelper.showErrorDialog(_("isisfish.simulator.simulaction.badid",
                        fullSimulationId), null);
            } else {
                DesignPlan designPlan = new DesignPlan();
                designPlan.setFactorGroup(factorGroup);
                SimulationService.getService().submit(fullSimulationId, param,
                        launcher, 0, sensitivityAnalysis, designPlan);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't start simulation", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.launchsimulation"), eee);
        }
    }

    public void launchSimulationWithSensibility(String simulationId,
            SimulatorLauncher launcher) {
        launchSimulation(simulationId, launcher, getSensitivityAnalysis());
    }

    /**
     * Lance la second passe d'une analyse de sensibilité.
     * 
     * @param masterSensitivitySimulationName nom du prefix de toutes les simulations (without _)
     */
    public void runSensitivitySecondPass(String masterSensitivitySimulationName) {

        // sensitivity analysis found
        SensitivityAnalysis sensitivityAnalysis = null;
        //List<SimulationStorage> simulationStorageForAnalyze = new ArrayList<SimulationStorage>();
        SortedMap<Integer, SimulationStorage> simulationStorageForAnalyze = new TreeMap<Integer, SimulationStorage>();
        File simuationDirectory = SimulationStorage.getSimulationDirectory();
        for (File simuation : simuationDirectory.listFiles()) {
            if (simuation.isDirectory()
                    && simuation.getName().startsWith(
                            masterSensitivitySimulationName + "_")) {
                SimulationStorage storage = SimulationStorage
                        .getSimulation(simuation.getName());
                String suffix = simuation.getName().substring(
                        simuation.getName().lastIndexOf("_") + 1);
                simulationStorageForAnalyze.put(Integer.valueOf(suffix),
                        storage);

                if (sensitivityAnalysis == null) {
                    // try to find find calculator name in one storage
                    SimulationParameter params = storage.getParameter();
                    sensitivityAnalysis = params.getSensitivityAnalysis();
                }
            }
        }

        if (sensitivityAnalysis != null) {
            try {
                // build master sensitivity export directory
                File masterExportDirectory = new File(SimulationStorage.getSensitivityResultsDirectory(),
                        masterSensitivitySimulationName);
                if (!masterExportDirectory.isDirectory()) {
                    masterExportDirectory.mkdirs();
                }
                List<SimulationStorage> simulationStorageForAnalyzeList = new ArrayList<SimulationStorage>(
                        simulationStorageForAnalyze.values());
                sensitivityAnalysis.analyzeResult(
                        simulationStorageForAnalyzeList, masterExportDirectory);
            } catch (SensitivityException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't call analyse result", e);
                }
            }
        } else {
            if (log.isWarnEnabled()) {
                log.warn("Can't run sensitivity second pass");
            }
        }
    }

    /**
     * Renvoie les resultats de la seconde pass.
     * 
     * @param masterSensitivitySimulationName nom du prefix de toutes les simulations (without _)
     * 
     * @return les fichiers genere lors de la seconde passe (seulement ceux qui ne commence pas par ".")
     */
    public List<File> getSensitivitySecondPassResults(
            String masterSensitivitySimulationName) {

        List<File> result = new ArrayList<File>();

        // build master sensitivity export directory
        File masterExportDirectory = new File(SimulationStorage.getSensitivityResultsDirectory(), masterSensitivitySimulationName);
        if (masterExportDirectory.isDirectory()) {
            for (File exportFile : masterExportDirectory.listFiles()) {
                if (exportFile.isFile()
                        && !exportFile.getName().startsWith(".")) {
                    result.add(exportFile);
                }
            }
        }

        return result;
    }
    
    /**
     * Population selection changed.
     *
     * Fill matrix panel with population effective of selected population.
     * 
     * @param paramsUI paramsUI
     */
    public void populationSelected(final ParamsUI paramsUI) {
        CardLayout layout = (CardLayout)paramsUI.getPopulationEffectivesPanel().getLayout();
        
        Object[] selectedPopulationsValues = paramsUI.getListSimulParamsPopulations().getSelectedValues();
        if (ArrayUtils.isNotEmpty(selectedPopulationsValues)) {
            paramsUI.getPopulationEffectivesTabbedPane().removeAll();
            setPopulations(selectedPopulationsValues);

            for (Object selectedPopulationValue : selectedPopulationsValues) {
                Population selectedPopulation = (Population)selectedPopulationValue;
                final MatrixPanelEditor matrixPanel = new MatrixPanelEditor();
                MatrixND populationEffectives = getSimulationParameter().getNumberOf(selectedPopulation);
                matrixPanel.setMatrix(populationEffectives);

                JPanel matrixPanelComponent = new JPanel(new BorderLayout());
                matrixPanelComponent.add(matrixPanel, BorderLayout.CENTER);

                // add addFactorButton with matrixPanel (just in sensitivity cas)
                if (paramsUI.isSensitivity()) {
                    matrixPanel.putClientProperty("sensitivityPopulation", selectedPopulation);
                    // TODO add another thing that action
                    JButton addFactorButton = new JButton();
                    addFactorButton.setAction(new AbstractAction() {
                        public void actionPerformed(ActionEvent e) {
                            addFactorWithComponent(paramsUI, matrixPanel);
                        }
                        @Override
                        public Object getValue(String key) {
                            Object result = null;
                            if (key.equals(Action.SMALL_ICON)) {
                                result = SwingUtil.createImageIcon("building_add.png");
                            }
                            return result;
                        }
                    });
                    matrixPanelComponent.add(addFactorButton, BorderLayout.EAST);
                }

                paramsUI.getPopulationEffectivesTabbedPane().add(matrixPanelComponent,
                        _("isisfish.params.populationEffectives", selectedPopulation.getName()));
            }
            
            layout.show(paramsUI.getPopulationEffectivesPanel(), "specific");
        }
        else {
            layout.show(paramsUI.getPopulationEffectivesPanel(), "default");
        }
    }

    /**
     * Action appelée lors du clic sur les boutons a coté des composants factorisables.
     *
     * Contrairement à l'interface 'input', on edite ici des facteurs existants,
     * (créé à la première demande).
     *
     * @param paramsUI paramsUI
     * @param source la source de l'event
     */
    public void addFactorWithComponent(ParamsUI paramsUI, JComponent source) {

        if (log.isDebugEnabled()) {
            log.debug("Event intercepted on " + source);
        }

        Factor selectedFactor = null;

        // new factor with rule domain
        if (source instanceof RuleChooser) {
            String factorPath = "parameters.rules";
            selectedFactor = getFactor(factorPath);
            if (selectedFactor == null) {
                selectedFactor = new Factor(_("isisfish.sensitivity.rulesfactorname"));
                selectedFactor.setPath(factorPath);
                selectedFactor.setDomain(new RuleDiscreteDomain());
            }
        }

        // new factor with matrix continous domain
        else if (source instanceof MatrixPanelEditor) {
            Population population = (Population)source.getClientProperty("sensitivityPopulation");
            String factorPath = "parameters.population." + population.getName() + ".number";
            selectedFactor = getFactor(factorPath);
            if (selectedFactor == null) {
                selectedFactor = new Factor(_("isisfish.sensitivity.populationfactorname", population.getName()));
                selectedFactor.setPath(factorPath);
                MatrixContinuousDomain factorDomain = new MatrixContinuousDomain();
                MatrixND populationEffectives = getSimulationParameter().getNumberOf(population);
                factorDomain.setReferenceValue(populationEffectives.copy());
                factorDomain.setCoefficient(0.0);
                selectedFactor.setDomain(factorDomain);
            }
        }

        if (selectedFactor != null) {
            FactorWizardUI wizard = new FactorWizardUI(paramsUI);
            SensitivityInputHandler handler = wizard.getHandler();
            handler.initExistingFactor(wizard, selectedFactor);
            wizard.pack();
            wizard.setLocationRelativeTo(paramsUI);
            wizard.setVisible(true);
        }
    }

    /**
     * Ajout d'un nouveau facteur sur un parametres de règles.
     * 
     * Le facteur est directement crée avec un nom et un path correct.
     * 
     * @param ruleChooser rule chooser
     * @param rule rule
     * @param paramName rule parameter name
     */
    public void addRuleParameterFactor(RuleChooser ruleChooser, Rule rule, String paramName) {

        // get sensitivity handler (this break package rules)
        // but no other choice :(
        SensitivityInputHandler handler = ruleChooser.getContextValue(SensitivityInputHandler.class);

        // get index of rule in rule list
        // warning, factor path must always be cohérent
        // with rule list, if a rule is deleted, factor on it must
        // be deleted too, et next factor must be renamed
        int index = ruleChooser.getRulesList().indexOf(rule);
        String factorPath = "parameters.rule." + index + ".parameter." + paramName;
        try {

            // on a besoin de la valeur pour savoir s'il peut être continue
            Object paramValue = RuleStorage.getParameterValue(rule, paramName);

            if (!handler.canBeContinue(paramValue)) {
                // dans le cas ou il ne peut pas être continue, on le gere
                // differement et on utilise sont type comme
                // valeur
                Class paramType = RuleStorage.getParameterType(rule, paramName);
                factorPath += "." + paramType.getName();
                paramValue = paramType;
            }

            // init new factor wizard ui
            FactorWizardUI factorWizardUI = new FactorWizardUI(ruleChooser);
            factorWizardUI.getFactorNameField().setText(_("isisfish.sensitivity.ruleparameterfactorname",
                    rule.getClass().getSimpleName(), paramName));
            factorWizardUI.setFactorPath(factorPath);
            handler.initNewFactorWithValue(factorWizardUI, paramValue);
            factorWizardUI.pack();
            factorWizardUI.setLocationRelativeTo(ruleChooser);
            factorWizardUI.setVisible(true);
        } catch (IsisFishException ex) {
            throw new IsisFishRuntimeException("Can't add factor on rule", ex);
        }
    }
}
