/*
 * #%L
 * IsisFish
 * 
 * $Id: SimulationService.java 3567 2011-12-19 14:38:07Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2002 - 2011 Ifremer, Code Lutin, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator.launcher;

import static org.nuiton.i18n.I18n._;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableInt;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.FileUtil;
import org.nuiton.util.ListenerSet;
import org.nuiton.util.ObjectUtil;
import org.nuiton.util.StringUtil;
import org.nuiton.util.ZipUtil;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.datastore.RuleStorage;
import fr.ifremer.isisfish.datastore.ScriptStorage;
import fr.ifremer.isisfish.datastore.SensitivityAnalysisStorage;
import fr.ifremer.isisfish.datastore.SensitivityExportStorage;
import fr.ifremer.isisfish.datastore.SimulationPlanStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.datastore.SimulatorStorage;
import fr.ifremer.isisfish.export.SensitivityExport;
import fr.ifremer.isisfish.mexico.MexicoHelper;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.rule.RuleHelper;
import fr.ifremer.isisfish.simulator.SimulationControl;
import fr.ifremer.isisfish.simulator.SimulationException;
import fr.ifremer.isisfish.simulator.SimulationParameter;
import fr.ifremer.isisfish.simulator.SimulationPlan;
import fr.ifremer.isisfish.simulator.SimulationPlanContext;
import fr.ifremer.isisfish.simulator.sensitivity.DesignPlan;
import fr.ifremer.isisfish.simulator.sensitivity.Domain;
import fr.ifremer.isisfish.simulator.sensitivity.Factor;
import fr.ifremer.isisfish.simulator.sensitivity.FactorGroup;
import fr.ifremer.isisfish.simulator.sensitivity.Scenario;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityAnalysis;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityScenarios;
import fr.ifremer.isisfish.simulator.sensitivity.domain.EquationContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.EquationDiscreteDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.RuleDiscreteDomain;
import fr.ifremer.isisfish.util.CompileHelper;
import fr.ifremer.isisfish.util.ConverterUtil;

/**
 * Cette classe est responsable de conservation de toutes les simulations faites
 * ou a faire. Pour ajouter une nouvelle simulation on appelle une des méthodes
 * {@code submit}.
 * <p>
 * Cette classe sert aussi de modele pour le moniteur de queue
 * <p>
 * Il existe une instance unique pour toute l'application
 * <p>
 * Lors de l'instanciation de la classe, l'ensemble des
 * {@link SimulatorLauncher} disponible est recherche dans la configuration
 * et un executor est cree pour chaque.
 * <p>
 * Si un {@link SimulatorLauncher} genere trop d'erreur (RemoteException)
 * Il est alors suspendu pour ne plus etre utilise pour les simulations.
 * 
 * @author poussin
 * @version $Revision: 3567 $
 * 
 * Last update : $Date: 2011-12-19 15:38:07 +0100 (lun., 19 déc. 2011) $
 * By : $Author: echatellier $
 */
public class SimulationService {

    public static final String SIMULATION_LAUNCHER = "simulation.launcher";
    /** nombre maximal de simulation autoriser pour un plan */
    // FIXME a rendre configurable MAX_PLAN_SIMULATION
    public static final int MAX_PLAN_SIMULATION = 20000;

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(SimulationService.class);

    protected static SimulationService instance = new SimulationService();

    /**
     * Retourne l'instance du {@link SimulationService} a utiliser
     * @return l'instance a utiliser
     */
    public static SimulationService getService() {
        return instance;
    }

    protected PropertyChangeSupport propertyListeners = new PropertyChangeSupport(
            this);
    protected ListenerSet<SimulationServiceListener> listeners = new ListenerSet<SimulationServiceListener>();

    // FIXME pouvoir configurer ceci en fichier de config
    // FIXME ainsi que la mise a false, lors de la simulation de la derniere
    // (sans doute ajoute un boolean, simulationListAsQueue = true|false)
    protected boolean autoLaunch = true;

    /** L'executor utilise pour creer toutes les sous simulations des plans
     * independants */
    protected ExecutorService subSimulationComputationExecutor = Executors
            .newSingleThreadExecutor();
    /** Tous les types de {@link SimulatorLauncher} disponibles, et leur
     * executors associe */
    protected Map<SimulatorLauncher, SimulationExecutor> executors = new LinkedHashMap<SimulatorLauncher, SimulationExecutor>();
    /** Le nombre d'erreur pour les SimulatorLauncher */
    protected Map<SimulatorLauncher, MutableInt> launcherError = new HashMap<SimulatorLauncher, MutableInt>();
    /** La queue contenant toutes les simulations a faire */
    protected SimulationQueue queue = new SimulationQueue();

    /** Contient les identifiants des simulations presentes dans {@link #jobs}*/
    protected Set<String> idJobs = new HashSet<String>();
    /** La liste des jobs existant (queue + job demarre) */
    protected Set<SimulationJob> jobs = new TreeSet<SimulationJob>();

    /** La liste des jobs termines */
    protected Set<SimulationJob> jobDones = new TreeSet<SimulationJob>();

    /**
     * Cree une instance et initialise les executors ainsi que le launcherError
     */
    protected SimulationService() {
        // on cree un executor par type SimulatorLauncher
        Properties prop = IsisFish.config
                .getOptionStartsWith(SIMULATION_LAUNCHER);

        // sort simulation names by... names
        // this solve "local" "sub", "remote" order
        // default "local" is in first in UI.
        List<String> simulationKeys = new ArrayList<String>(prop
                .stringPropertyNames());
        Collections.sort(simulationKeys);

        for (String key : simulationKeys) {
            String value = prop.getProperty(key);
            try {
                SimulatorLauncher sl = (SimulatorLauncher) ObjectUtil
                        .newInstance(value);
                addSimulationLauncher(sl);
            } catch (Exception eee) {
                log.warn(_("Can't instantiate %s", value), eee);
            }
        }

        // fait un appel au moniteur
        // pour reprendre le monitoring des simulation en cours
        // TODO check this, don't restart monitoring on non ui launch
        if (IsisFish.config.isLaunchUI()) {
            SimulationMonitor.getInstance().reloadConfig(this);
        }
    }

    /**
     * Permet d'ajouter un nouveau SimulatorLauncher. Cela cree automatiquement
     * un executor pour ce SimulatorLauncher. S'il y avait deja un
     * SimulatorLauncher de ce type un nouveau est ajoute.
     * @param sl le SimulatorLauncher a ajouter
     */
    public void addSimulationLauncher(SimulatorLauncher sl) {
        SimulationQueue executorQueue = new SimulationQueue(queue);
        SimulationExecutor se = new SimulationExecutor(this, sl, executorQueue);
        executors.put(sl, se);
        launcherError.put(sl, new MutableInt(0));
    }

    /**
     * @return les SimulatorLauncher et leurs noms
     */
    public List<SimulatorLauncher> getSimulationLaunchers() {
        List<SimulatorLauncher> result = new ArrayList<SimulatorLauncher>(
                executors.keySet());
        return result;
    }

    public Collection<SimulationExecutor> getSimulationExecutors() {
        return executors.values();
    }

    public synchronized Set<SimulationJob> getJobs() {
        return jobs;
    }

    public synchronized Set<SimulationJob> getJobDones() {
        return jobDones;
    }

    /**
     * Retourne la liste de tous les {@link SimulatorLauncher} disponible.
     * 
     * @return {@link SimulatorLauncher}s
     */
    protected Set<SimulatorLauncher> getSimulatorLaunchers() {
        return executors.keySet();
    }

    public void addSimulationServiceListener(SimulationServiceListener l) {
        listeners.add(l);
    }

    public void removeSimulationServiceListener(SimulationServiceListener l) {
        listeners.remove(l);
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propertyListeners.addPropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName,
            PropertyChangeListener listener) {
        propertyListeners.addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propertyListeners.removePropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(String propertyName,
            PropertyChangeListener listener) {
        propertyListeners.removePropertyChangeListener(propertyName, listener);
    }

    /**
     * Ajoute le job au job en cours previent s'il n'etait pas deja present
     * les listeners
     * {@link SimulationServiceListener#simulationStart(SimulationService, SimulationJob)}.
     * @param job
     */
    protected synchronized void fireStartEvent(SimulationJob job) {
        // on ajoute au cas on il n'y serait pas
        if (jobs.add(job)) {
            idJobs.add(job.getItem().getControl().getId());
            for (SimulationServiceListener l : listeners) {
                l.simulationStart(this, job);
            }
        }
    }

    /**
     * Supprime le job de la liste des jobs en cours et l'ajoute le job si
     * besoin a la liste des jobs faits (s'il y a une erreur ou qu'il ne sagit
     * pas d'une simulation lancer par un plan de simulation).
     * Les listener sont prevenu par la méthode
     * {@link SimulationServiceListener#simulationStop(SimulationService, SimulationJob)}.
     * @param job
     */
    protected synchronized void fireStopEvent(SimulationJob job) {
        jobs.remove(job);
        idJobs.remove(job.getItem().getControl().getId());

        // on ajout au jobDones que les simulations avec erreur ou les
        // simulation reellement demande par l'utilisateur (pas de simulation
        // de plan
        jobDones.add(job);
        for (SimulationServiceListener l : listeners) {
            l.simulationStop(this, job);
        }
    }

    /**
     * Nettoie la liste des jobs faits
     */
    public void clearJobDone() {
        jobDones.clear();
        for (SimulationServiceListener l : listeners) {
            l.clearJobDone(this);
        }
    }

    /**
     * @return Returns the autoLaunch.
     */
    public boolean isAutoLaunch() {
        return this.autoLaunch;
    }

    /**
     * @param autoLaunch The autoLaunch to set.
     */
    public void setAutoLaunch(boolean autoLaunch) {
        synchronized (this) {
            boolean oldValue = this.autoLaunch;
            this.autoLaunch = autoLaunch;
            if (this.autoLaunch) {
                this.notifyAll();
            }
            propertyListeners.firePropertyChange("autoLaunch", oldValue,
                    autoLaunch);
        }
    }

    /**
     * Permet de mettre en attente les threads de simulation si l'utilisateur
     * a suspendu la queue
     */
    protected void waitAutoLaunch() {
        synchronized (this) {
            while (!isAutoLaunch()) {
                try {
                    log.info("autoLaunch is false waiting queue start");
                    this.wait();
                } catch (InterruptedException eee) {
                    log.warn("Error during wait autoLaunch flag", eee);
                }
            }
        }
    }

    /**
     * Permet d'ajouter une nouvelle simulation a la queue. Si la simulation est
     * simple ou avec un plan d'experience dependant, un nouveau job est cree
     * et directement ajout a la queue de jobs. S'il la simulation est un plan
     * ou les simulations sont independantes, toutes les simulations sont
     * genere et toute ajoute a la queue de simulation. Mais dans ce cas seul
     * la simulation de depart est ajoute au jobs en cours, les simulations
     * generees seront ajoutees lors de leur reel execution
     * 
     * @param id l'identifiant de la simulation
     * @param param les parametres de la simulation
     * @param launcher le launcher a utiliser pour lancer la simulation
     * @param priority la priorite de la simulation
     */
    public void submit(String id, SimulationParameter param,
            SimulatorLauncher launcher, int priority) {

        // make deep copy, ui, still work with
        // copy need to be deep for simulation plan to reload
        SimulationParameter localParameters = param.deepCopy();

        // on l'ajoute tout de suite a la liste des simulations demandee
        SimulationControl control = new SimulationControl(id);
        SimulationItem item = new SimulationItem(control, localParameters);
        SimulationJob job = new SimulationJob(this, item, priority);
        job.setLauncher(launcher);
        fireStartEvent(job);

        // Attention, dans le cas d'un plan de simulation, le new PrepareSimulationJob
        // doit etre fait AVANT de faire le prepareSimulationZipFile
        // car, il est possible que l'init des plans ajoute des regles !!!
        PrepareSimulationJob task = null;
        if (localParameters.getUseSimulationPlan()) {
            task = new PrepareSimulationJob(this, job);
        }

        // on construit le zip de la simulation
        File zip = prepareSimulationZipFile(control, localParameters, null, null, true);
        item.setSimulationZip(zip);

        if (localParameters.getUseSimulationPlan() && localParameters.isIndependentPlan()) {
            // c un plan de simulation independant, on construit toute les sous simu
            subSimulationComputationExecutor.execute(task);
        } else {
            // l'item est fini d'etre initialise, on peut l'ajouter a la queue
            // sauf si c'etait un plan independant ou se seront les sous simu
            // qui seront dans la queue
            submit(job);
        }

    }

    /**
     * Permet d'ajouter une simulation avec des factors a prendre en compte.
     * 
     * @param id l'identifiant de la simulation
     * @param param les parametres de la simulation
     * @param launcher le launcher a utiliser pour lancer la simulation
     * @param priority la priorite de la simulation
     * @param sensitivityAnalysis l'implementation du calculateur de sensibilite
     * @param designPlan le design plan qui contient les facteurs
     */
    public void submit(String id, SimulationParameter param,
            SimulatorLauncher launcher, int priority,
            SensitivityAnalysis sensitivityAnalysis, DesignPlan designPlan) {

        // make deep copy, ui, still work with
        // copy need to be deep for simulation plan to reload
        SimulationParameter localParameters = param.deepCopy();

        try {

            // build master sensitivity export directory
            File masterExportDirectory = new File(SimulationStorage.getSensitivityResultsDirectory(), id);
            if (!masterExportDirectory.isDirectory()) {
                masterExportDirectory.mkdirs();
            }

            // export designplan as XML
            // get as XML (before compute)
            // compute do bad modification on design plan
            String xmlDesignPlan = MexicoHelper.getDesignPlanAsXML(designPlan);

            // get a TopiaContext on region database, for discret domain
            // to get original database value (Nominal value)
            RegionStorage regionStorage = RegionStorage.getRegion(param.getRegionName());
            TopiaContext context = null;
            try {
                context = regionStorage.getStorage().beginTransaction();
                fillDesignPlanNominalValue(context, designPlan);
            } finally {
                if (context != null) {
                    context.closeContext();
                }
            }

            // Sensitivity scenario returned contains same factor references
            // as input design plan. Factor group are still present
            SensitivityScenarios sensitivityScenarios = sensitivityAnalysis
                    .compute(designPlan, masterExportDirectory);

            // ajout des parametres de simulation
            localParameters.setSensitivityAnalysis(sensitivityAnalysis);
            localParameters.setNumberOfSensitivitySimulation(sensitivityScenarios.getScenarios().size());

            // on l'ajoute tout de suite a la liste des simulations demandee
            // init with default "id" must been changed for designPlan
            // simulation
            SimulationControl controlJustForZip = new SimulationControl(id);

            // on construit le zip de la simulation
            // in zip, there will be param "without" prescript
            File zip = prepareSimulationZipFile(controlJustForZip, localParameters, xmlDesignPlan, sensitivityScenarios, true);

            // for each simulation, we neeed to launch a specific simulation
            // new simulation ids will be id + "_" + x
            // where x is a int starting at 0
            int simulationIndex = 0;
            for (Scenario scenario : sensitivityScenarios.getScenarios()) {

                // modify modification parameter
                // subParam only convenient for add pre script
                // for launcher
                SimulationParameter subParam = localParameters.copy();
                String preScriptContent = generatePreScript(scenario);
                // usefull next two line ?
                subParam.setUsePreScript(true);
                subParam.setPreScript(preScriptContent);

                // build new simulation id
                String subsimulationId = id + "_" + simulationIndex;
                SimulationControl realSimulationControl = new SimulationControl(subsimulationId);

                // new item, with zip
                SimulationItem item = new SimulationItem(realSimulationControl, subParam);
                item.setSimulationZip(zip);
                item.setStandaloneSimulation(false); // AS = always false
                item.setStandaloneSimulationZip(false); // AS = same zip
                item.setSimulationNumber(simulationIndex);
                item.setLastSimulation(simulationIndex == sensitivityScenarios.getScenarios().size() - 1);

                SimulationJob job = new SimulationJob(this, item, priority);
                job.setLauncher(launcher);
                fireStartEvent(job);

                if (subParam.getUseSimulationPlan()
                        && subParam.isIndependentPlan()) {
                    // c un plan de simulation independant, on construit toute les sous simu
                    Runnable task = new PrepareSimulationJob(this, job);
                    subSimulationComputationExecutor.execute(task);
                } else {
                    // l'item est fini d'etre initialise, on peut l'ajouter a la queue
                    // sauf si c'etait un plan independant ou se seront les sous simu
                    // qui seront dans la queue
                    submit(job);
                }

                simulationIndex++;
            }

        } catch (Exception e) {
            throw new IsisFishRuntimeException("Can't get scenarios from calculator", e);
        }
    }

    /**
     * Explore design plan factor tree, and set factor nominal value
     * for each factor.
     * 
     * Also fix factor name for R (without special chars in name)
     * 
     * @param context opened database context
     * @param designPlan design plan
     */
    protected void fillDesignPlanNominalValue(TopiaContext context, DesignPlan designPlan) {
        // take care only on first level of factor tree
        for (Factor factor : designPlan.getFactors()) {
            // get nominal value
            String path = factor.getPath();
            if (StringUtils.contains(path, "#")) {
                String topiaId = path.substring(0, path.lastIndexOf("#"));
                String propertyName=path.substring(path.lastIndexOf("#") + 1);
                try {
                    TopiaEntity entity = context.findByTopiaId(topiaId);
                    Object result = BeanUtils.getProperty(entity, propertyName);
                    factor.setNominalValue(result);
                } catch (Exception ex) {
                    if (log.isErrorEnabled()) {
                        log.error("An error occurred while trying to get nominal value", ex);
                    }
                }
            } else {
                factor.setNominalValue("NaN");
            }
            
            // also fix factor name for sensitivity calculator
            String name = factor.getName();
            name = StringUtil.unaccent(name);
            name = name.replaceAll("\\W", "_");
            factor.setName(name);
        }
    }

    /** Permet de manipuler un int entre plusieurs appele de methodes recursive. */
    public static class Counter {
        protected int counter = 0;
        public void inc() {
            counter++;
        }
        public int getCounter() {
            return counter;
        }
    }

    /**
     * Generate prescript to set factor values before simulation start.
     * 
     * @param scenario to generate prescript
     * @return simulation pre script with correct values
     */
    protected String generatePreScript(Scenario scenario) {
        ConvertUtilsBean beanUtils = ConverterUtil.getConverter(null);

        // n'utilise plus freemarker, car il y avait plus d'instruction
        // freemarker compliqué que de code a afficher
        StringBuffer buffer = new StringBuffer();
        buffer.append("import org.apache.commons.beanutils.BeanUtils;\n");
        buffer.append("import org.apache.commons.beanutils.ConvertUtilsBean;\n");
        buffer.append("import fr.ifremer.isisfish.util.ConverterUtil;\n");
        buffer.append("import fr.ifremer.isisfish.simulator.SimulationParameter;\n");
        buffer.append("ConvertUtilsBean beanUtils = ConverterUtil.getConverter(db);\n");
        buffer.append("SimulationParameter params = context.getSimulationStorage().getParameter();\n");

        generatePreScript(beanUtils, buffer, new Counter(), scenario.getFactors());
        
        String scriptContent = buffer.toString();
        if (log.isTraceEnabled()) {
            log.trace("Simulation prescript content = " + scriptContent);
        }
        return scriptContent;
    }

    /**
     * Generate prescript for a factor list that can be called recursively to
     * manage factor group.
     * 
     * @param beanUtils beanUtils converter
     * @param buffer buffer to fill
     * @param counter call counter used to avoid variables name collision
     * @param factors factor list to manage
     */
    protected void generatePreScript(ConvertUtilsBean beanUtils, StringBuffer buffer, Counter counter, Collection<Factor> factors) {
        
        for (Factor factor : factors) {
            if (factor instanceof FactorGroup) {
                buffer.append("/* factor group : ").append(factor.getName()).append(" */\n");
                FactorGroup factorGroup = (FactorGroup)factor;
                generatePreScript(beanUtils, buffer, counter, factorGroup.getFactors());
            }
            else {
                int factorIndex = counter.getCounter();
                buffer.append("/* factor : ").append(factor.getName()).append(" */\n");

                // cas special 1 : population de départ
                if (factor.getPath().matches("parameters.population\\.\\w+\\.number")) {
                    String paramName = StringUtils.removeStart(factor.getPath(), "parameters.");
                    // pas de convert, c'est fait comme ca dans
                    // SimulationParameters.toProperties();
                    MatrixND matrix = (MatrixND)factor.getValue();
                    Object stringValue = String.valueOf(matrix.toList());
                    buffer.append("params.setProperty(\"").append(paramName).append("\",\"").append(stringValue).append("\");\n");
                }
                // cas special 2 : regles
                else if (factor.getPath().equals("parameters.rules")) {
                    List<Rule> rules = (List<Rule>)factor.getValue();
                    List<String> rulesNames = new ArrayList<String>();
                    int ruleIndex = 0;
                    for (Rule rule : rules) {
                        rulesNames.add(rule.getClass().getSimpleName());
                        Properties rulesProps = RuleHelper.getRuleAsProperties(ruleIndex++, null, rule);
                        for (String rulesProp : rulesProps.stringPropertyNames()) {
                            String value = rulesProps.getProperty(rulesProp);
                            buffer.append("params.setProperty(\"").append(rulesProp).append("\",\"").append(value).append("\");\n");
                        }
                    }
                    buffer.append("params.setProperty(\"rules\",\"").append(StringUtils.join(rulesNames, ",")).append("\");\n");
                }
                // cas special 3 : facteur sur les parametres des regles
                else if (factor.getPath().startsWith("parameters.rule.")) {
                    // special case for rule parameter with 
                    Pattern pattern = Pattern.compile("^parameters\\.(rule\\.\\d+\\.parameter\\.\\w+)\\..+$");
                    Matcher matcher = pattern.matcher(factor.getPath());
                    String paramName = null;
                    if (matcher.matches()) {
                        paramName = matcher.group(1);
                    } else {
                        paramName = StringUtils.removeStart(factor.getPath(), "parameters.");
                    }
                    String stringValue = ConvertUtils.convert(factor.getValue());
                    buffer.append("params.setProperty(\"").append(paramName).append("\",\"").append(stringValue).append("\");\n");
                }
                // cas special 4 : equation
                else if (factor.getDomain() instanceof EquationContinuousDomain) {
                    buffer.append("context.setComputeValue(\"").append(factor.getName());
                    buffer.append("\",").append(factor.getValue()).append(");\n");
                }
                // cas pas si special
                else {
                    Object value = factor.getValue();
                    String stringValue = beanUtils.convert(value);
                    String escValue = stringValue;
                    if (factor.getDomain() instanceof EquationDiscreteDomain) {
                        // echatellier: equation can contains quotes that break
                        // prescript, only quote, not all java replacements
                        // a ne pas faire pour le reste, pour les
                        // matrice par exemple, ca passe mal
                        escValue = StringUtils.replace(escValue, "\n", "");
                        escValue = StringUtils.replace(escValue, "\r", "");
                        escValue = StringUtils.replace(escValue, "\"", "\\\"");
                    }
                    String path = factor.getPath();
                    String topiaId = path.substring(0, path.lastIndexOf('#'));
                    String property = path.substring(path.lastIndexOf('#') + 1);
                    // Double value123 = beanUtils.convert("mystringvalue", Double.class);
                    buffer.append(value.getClass().getName()).append(" value");
                    buffer.append(factorIndex).append(" = beanUtils.convert(\"");
                    buffer.append(escValue).append("\", ").append(value.getClass().getName());
                    buffer.append(".class);\n");
                    // TopiaEntity entity123 = db.findByTopiaId(topiaId);
                    buffer.append("TopiaEntity entity").append(factorIndex);
                    buffer.append(" = db.findByTopiaId(\"").append(topiaId);
                    buffer.append("\");\n");
                    
                    if (factor.getDomain() instanceof EquationDiscreteDomain) {
                        // BeanUtils.setProperty(entity123, "propertyContent", value123);
                        buffer.append("BeanUtils.setProperty(entity").append(factorIndex);
                        buffer.append(", \"").append(property).append("Content\", ");
                        buffer.append("value").append(factorIndex).append(");\n");
                    } else {
                        // BeanUtils.setProperty(entity123, "property", value123);
                        buffer.append("BeanUtils.setProperty(entity").append(factorIndex);
                        buffer.append(", \"").append(property).append("\", ");
                        buffer.append("value").append(factorIndex).append(");\n");
                    }
                }
            }

            counter.inc();
        }
    }

    protected void submit(SimulationJob job) {
        SimulatorLauncher launcher = job.getLauncher();
        // on ajoute a la queue qui utilise le launcher defini dans le job
        if (launcher != null) {
            for (SimulatorLauncher l : executors.keySet()) {
                if (launcher == l) {
                    SimulationExecutor executor = executors.get(l);
                    executor.execute(job);
                    return;
                }
            }
        }
        // dernier recours on ajoute a la queue sans launcher
        log.info(_("Add to default queue"));
        queue.add(job);
    }

    /**
     * Permet de resoumettre un job qui a ete pris par un thread mais qu'il 
     * ne peut pas traiter. Cela arrive lorsque l'executor est en pause, ou
     * que le launcher de l'executor ne fonctionne plus (il se met en pause
     * tout seul)
     * 
     * @param job l'item a resoumettre
     */
    protected void resubmit(SimulationJob job) {
        submit(job);
    }

    /**
     * Permet de soumettre a la queue un job provenant d'un plan.
     * Ce plan ne doit pas apparaitre dans la console de queue.
     * 
     * @param job
     */
    protected void submitSubJob(SimulationJob job) {
        submit(job);
    }

    /**
     * Resoumet une simulation qui a deja été démarrée, mais
     * on ne faisant que du control de monitoring.
     * 
     * @param job job to submit
     */
    public void submitForCheckOnly(SimulationJob job) {
        job.setOnlyCheckControl(true);
        fireStartEvent(job);
        submit(job);
    }

    /**
     * Restart a job.
     * 
     * This method fire "start" event.
     * 
     * @param job job to retstart
     */
    public void restart(SimulationJob job) {
        // lorsqu'un job est resoumit, il redevient forcement standalone
        job.getItem().setStandaloneSimulation(true);
        // case were we restart a reloaded job
        job.setOnlyCheckControl(false);

        job.getItem().getControl().reset();
        fireStartEvent(job);
        submit(job);
    }

    /**
     * Supprime un job de la queue (annulation d'une simulation). Appele
     * depuis {@link SimulationJob#stop}
     * 
     * @param job le job a annuler
     * @return vrai si la simulation a pu etre annulee avant sont lancement 
     * (encore presente dans la queue), faux si la simulation a deja ete
     * prise par un thread de simulation ou quelle est terminee
     */
    protected boolean cancel(SimulationJob job) {
        boolean result = queue.remove(job);
        return result;
    }

    public boolean exists(String id) {
        boolean result = idJobs.contains(id);
        return result;
    }

    /**
     * Report une erreur pour un launcher, on resoumet le job en supprimant
     * le launcher utilise
     * 
     * @param launcher le launcher posant probleme
     * @param job le job qui n'a pas reussi a se faire
     */
    protected void reportError(SimulatorLauncher launcher, SimulationJob job) {
        MutableInt i = launcherError.get(launcher);
        i.setValue(i.intValue() + 1);
        // si on a plus de N error, on stop l'executor associe
        if (i.intValue() >= 50) {
            log.error(_("Launcher %s will be stopped because there are too many error (%s)",
                            launcher, i.intValue()));
            SimulationExecutor e = executors.get(launcher);
            e.pause();
        }
        // il faut bien penser a supprimer le launcher pour qu'un autre executor
        // puisse y mettre le sien.
        // FIXME: est ce le bon choix si l'utilisateur avait force un launcher particulier, ne faudrait t'il pas prevenir l'utilisateur ?
        // FIXME: disabled since caparmor is not our friend. Retry on caparmor in the limit of 50 errors
        //job.setLauncher(null);
        resubmit(job);
    }

    /**
     * Permet de genere les sous simulations d'un plan de simulation. Pour les
     * plan independant, on l'utilise en Runnable pour genere tous les plans
     * possible et les soumettre a la queue. Pour les plans dependant
     * on l'utilise seulement comme iterator. La methode afterSimulation des plans
     * est appelee automatiquement a la fin de la simulation grace au mecanisme
     * de PostAction sur les {@link SimulationJob}.
     */
    public static class PrepareSimulationJob implements Runnable,
            Iterator<SimulationJob>, SimulationJob.PostAction {

        protected SimulationService simulationService;
        protected SimulationPlanContext planContext;
        protected SimulationJob job;
        protected SimulationJob nextJob;
        protected boolean doNext = true;
        protected String id;
        protected SimulationControl control;
        protected SimulationParameter param;
        protected int done = 0;
        protected List<SimulationPlan> simulationPlan;

        public PrepareSimulationJob(SimulationService simulationService,
                SimulationJob job) {
            this.simulationService = simulationService;
            this.job = job;
            id = job.getItem().getControl().getId();
            control = job.getItem().getControl();
            param = job.getItem().getParameter();
            // take a copy of simulation paln list
            // because they a freed during simulation (soft reference)
            // is there is not enought memory available
            simulationPlan = param.getSimulationPlans();
            this.planContext = new SimulationPlanContext(control.getId(), param);
            
            try {
                // appel de init sur chaque plan
                for (SimulationPlan plan : simulationPlan) {
                    plan.init(planContext);
                }
            } catch (Exception eee) {
                // add manual log
                // we are in a thread, IsisFishRuntimeException is displayed
                // outside a log
                if (log.isErrorEnabled()) {
                    log.error(_("isisfish.error.evaluate.preplan.script"), eee);
                }
                throw new IsisFishRuntimeException(_("isisfish.error.evaluate.preplan.script"), eee);
            }
        }

        /**
         * Genere toutes les sous simulations et les places dans la queue.
         * 
         * Cette methode {@code run()} est appelée seulement dans le cas de la
         * génération de plans indépendants.
         * 
         * @see SimulationJob#run() pour les plans dépendants
         */
        public void run() {
            /* Original code
            while (hasNext()) {
                try {
                    SimulationJob subJob = next();

                    if (log.isInfoEnabled()) {
                        log.info("Simulation generee: " + subJob.getId());
                    }
                    simulationService.submitSubJob(subJob);
                } catch (Exception eee) {
                    if (log.isErrorEnabled()) {
                        log.error(_("Can't add simulation: %s", job.getItem()
                            .getControl().getId()), eee);
                    }
                }
            }*/
            
            // New iteration remember always simulation N and N+1
            // needed to know witch simulation is the last one
            // get simulation N
            SimulationJob subJobN = null;
            if (hasNext()) {
                subJobN = next();
            }

            while (subJobN != null) {
                
                try {
                    if (log.isInfoEnabled()) {
                        log.info("Simulation generated: " + subJobN.getId());
                    }
    
                    // set item additionnal informations
                    SimulationItem itemN = subJobN.getItem();
                    itemN.setStandaloneSimulation(false); // independant plan
                    // - 1 because planContext.getNumber() is set to next simulation to generate
                    itemN.setSimulationNumber(planContext.getNumber() - 1);
                    
                    // job N+1
                    // carefull call this next after itemN.setSimulationNumber()
                    SimulationJob subJobNp1 = next();
                    if (subJobNp1 == null) {
                        // there is no N+1 job, N is the last one
                        itemN.setLastSimulation(true);
                    }
                    
                    simulationService.submitSubJob(subJobN);
                    
                    subJobN = subJobNp1;
                }
                catch (Exception eee) {
                    if (log.isErrorEnabled()) {
                        log.error(_("Can't add simulation: %s", job.getItem()
                            .getControl().getId()), eee);
                    }
                }
            }
        }

        /**
         * Indique s'il y a encore des simulations dans le plan. Par defaut pour
         * Eviter les plans sans fin, le nombre de plan genere par simulation
         * est limite a {@link SimulationService#MAX_PLAN_SIMULATION}
         * 
         * @return <tt>true</tt> if has next
         */
        public boolean hasNext() {
            try {
                // if user request stop simulation, stop all futur planned simulation
                // and if last doNext is false not do next simulation
                boolean result = !control.isStopSimulationRequest() && doNext;
                if (result) {
                    
                    // hasNext() est appelee par un autre thread concurrent
                    // via la methode finished(SimulationJob, SimulationStorage)
                    synchronized (this) {
                    
                        // si deja creer on ne le refait pas
                        if (nextJob == null) {
                            // Prepration de la simulation a faire
                            // create next id simulation
    
                            // this start a 0
                            int planNumber = planContext.getNumber();
    
                            if (planNumber > MAX_PLAN_SIMULATION) {
                                log.error(_("Analyse plan error, too many simulation for %s : %s",
                                                id, planNumber));
                                doNext = false;
                                result = false;
                            } else {
                                String simId = id + "_" + planNumber;
                                param.setSimulationPlanNumber(planNumber);
    
                                File tmpDirectory = FileUtil.createTempDirectory(
                                        "isisfish-simulation-", "-preparation");
                                SimulationStorage sim = SimulationStorage
                                        .importAndRenameZip(tmpDirectory, job
                                                .getItem().getSimulationZip(),
                                                simId);
                                sim.getParameter().setSimulationPlanNumber(planNumber);
    
                                // appel de tous les plans pour modifier la simulation
                                for (SimulationPlan plan : simulationPlan) {
                                    result = result
                                            && plan.beforeSimulation(planContext, sim);
                                    if (!result) {
                                        nextJob = null;
                                        break;
                                    }
                                }
                                doNext = result;
                                if (result) {

                                    File zip = sim.createZip();
                                    SimulationControl childControl = new SimulationControl(simId);
                                    SimulationParameter childParam = param.copy();
                                    SimulationItem item = new SimulationItem(childControl, childParam);
                                    item.setSimulationZip(zip);

                                    nextJob = new SimulationJob(simulationService, job, item, job.getPriority());
                                    nextJob.setLauncher(job.getLauncher());

                                    // FIXME on retire la post action pour les plan dépendants
                                    // sera appelé directement par le job de preparations
                                    if (param.isIndependentPlan()) {
                                        nextJob.addPostAction(this); // pour l'appel des after des plans
                                    }
                                }

                                // close context for plan generator
                                sim.closeStorage();

                                // quoi qu'il arrive on supprime le repertoire temporaire
                                if (!FileUtil.deleteRecursively(tmpDirectory)) {
                                    log.warn(_("isisfish.error.remove.directory",
                                            tmpDirectory));
                                }
                            }

                            // increment number for next simulation job
                            planContext.incNumber();
                        }
                    }
                }
                return result;
            } catch (Exception eee) {
                throw new IsisFishRuntimeException(
                        _("isisfish.error.evalute.plan.script"), eee);
            }

        }

        public SimulationJob next() {
            hasNext(); // pour etre sur qu'il a ete appele au moins une fois
            SimulationJob result = null;
            
            // next est appelee par un autre thread concurrent
            // via la methode finished(SimulationJob, SimulationStorage)
            synchronized (this) {
                result = nextJob;
                nextJob = null;
            }
            
            return result;

        }

        public void remove() {
            throw new UnsupportedOperationException("Not supported.");
        }

        public void finished(SimulationJob job, SimulationStorage sim) {
            // doNext = true;

            // appel de tous les plans pour modifier la simulation
            // EC20090716 : use param.getSimulationPlans() instances,
            // not sim.getParameters().getSimulationPlans() not sames !!!
            for (SimulationPlan plan : simulationPlan) {
                try {
                    boolean result = plan.afterSimulation(planContext, sim);
                    doNext = doNext && result;
                } catch (Exception eee) {
                    log.error(_("Stop simulation plan, because can't call afterSimulation correctly on plan %s",
                                   plan.getClass().getName()), eee);
                    doNext = false;
                }
            }
            
            // une sim vient de se finir, on incremente le compteur
            done++;
            if (!hasNext() && (done + 1 == planContext.getNumber())) {
                // on enleve le master plan des simulations en cours, vu que
                // toutes les simu sont terminees
                simulationService.fireStopEvent(this.job);
            }
        }

        public void exception(SimulationJob job, Throwable eee) {
            // il y a une simulation d'echoue, on ne fait pas les suivantes
            // cela n'impacte pas les plan independant puisque toutes les 
            // simulation on deja ete generee
            doNext = false;
            simulationService.fireStopEvent(this.job);
        }
    }

    /**
     * Prepare les fichiers qui seront utilsé à la simulation:
     * <li> scripts
     * <li> rules
     * <li> exports
     * <li> simulators
     * <li> export de la database de la region
     * <li> simultionplan
     * <p>
     * Le tout est zippé et le zip est retourné, il peut-être directement
     * importé dans le {@link SimulationStorage} (en le renommant comme
     * il faut {@link SimulationStorage#importAndRenameZip(File, String)}.
     * <p>
     * Ce zip est automatiquement supprimé à la fin de l'application.
     * 
     * @param control le controleur
     * @param param les parametre de la simulation
     * @param xmlDesignPlan contenu xml des design plan
     * @param sensitivityScenarios used to add some extra files (such as rules)
     * @param compile si vrai la version compile des fichiers Java est aussi
     * mise dans le fichier zip. Cela peut servir pour les simulations locales
     * pour ne pas recompiler pour chaque simulation avec plan de simulation
     * @return un zip de simulation pour une simulation pret a être faite
     * @throws SimulationException pour tout problème rencontré (IO,Topia...)
     */
    protected File prepareSimulationZipFile(SimulationControl control,
            SimulationParameter param, String xmlDesignPlan, SensitivityScenarios sensitivityScenarios, boolean compile)
            throws SimulationException {
        try {
            File tmpDirectory = FileUtil.createTempDirectory(
                    "isisfish-simulation-", "-preparation");
            //File regionXML = IsisConfig.getDataBackupFile(tmpDirectory);
            File regionXML = new File(tmpDirectory,
                    SimulationStorage.DATA_BACKUP_FILENAME);

            // sauvegarde des parametres
            Properties prop = param.toProperties();
            //File f = IsisConfig.getSimulationParametersFile(tmpDirectory);
            File f = new File(tmpDirectory, SimulationStorage.PARAMETERS_FILENAME);
            FileOutputStream out = null;
            try {
                out = new FileOutputStream(f);
                prop.store(out, "Parameters");
            } finally {
                IOUtils.closeQuietly(out);
            }

            // backup pour toutes les simulations, pour eviter que l'utilisateur
            // ne puisse le modifier en meme temps
            control.setText(_("isisfish.message.backup.database.progress"));
            RegionStorage region = RegionStorage.getRegion(param
                    .getRegionName());
            TopiaContext tc = region.getStorage().beginTransaction();
            tc.backup(regionXML, true);
            tc.closeContext();
            control.setText(_("isisfish.message.backup.database.finished"));

            // copie de toutes regles a utiliser
            List<Rule> rules = param.getRules();

            // FIXME small hack to add rules defined in simulations factors
            if (sensitivityScenarios != null) {
                for (Scenario sensitivityScenario : sensitivityScenarios.getScenarios()) {
                    List<Factor> factors = sensitivityScenario.getFactors();
                    for (Factor factor : factors) {
                        Domain domain = factor.getDomain();
                        if (domain instanceof RuleDiscreteDomain) {
                            rules.addAll((List<Rule>)factor.getValue());
                        }
                    }
                }
            }
            for (Rule rule : rules) {
                String name = RuleStorage.getName(rule);
                File ruleFile = new File(RuleStorage.getRuleDirectory(), name + ".java");
                if (!ruleFile.isFile()) {
                    ruleFile = new File(RuleStorage.getCommunityRuleDirectory(), name + ".java");
                }
                FileUtil.copy(ruleFile, new File(tmpDirectory, RuleStorage.RULE_PATH
                                + File.separator + name + ".java"));
            }

            // copie des regles reclamées par les plans de simulation
            for (String name : param.getExtraRules()) {
                File ruleFile = new File(RuleStorage.getRuleDirectory(), name + ".java");
                if (!ruleFile.isFile()) {
                    ruleFile = new File(RuleStorage.getCommunityRuleDirectory(), name + ".java");
                }
                FileUtil.copy(ruleFile, new File(tmpDirectory, RuleStorage.RULE_PATH
                                + File.separator + name + ".java"));
            }

            // copie de toutes regles a utiliser
            List<SimulationPlan> plans = param.getSimulationPlans();
            for (SimulationPlan plan : plans) {
                String name = SimulationPlanStorage.getName(plan);
                File planFile = new File(SimulationPlanStorage.getSimulationPlanDirectory(), name + ".java");
                if (!planFile.isFile()) {
                    planFile = new File(SimulationPlanStorage.getCommunitySimulationPlanDirectory(), name + ".java");
                }
                FileUtil.copy(planFile, new File(tmpDirectory, SimulationPlanStorage.SIMULATION_PLAN_PATH
                                + File.separator + name + ".java"));
            }

            // copie de tous les exports a utiliser
            for (String name : param.getExportNames()) {
                name = name.endsWith(".java") ? name : name + ".java";
                File exportFile = new File(ExportStorage.getExportDirectory(), name);
                if (!exportFile.isFile()) {
                    exportFile = new File(ExportStorage.getCommunityExportDirectory(), name);
                }
                FileUtil.copy(exportFile,
                        new File(tmpDirectory, ExportStorage.EXPORT_PATH
                                + File.separator + name));
            }
            
            // copie de tous les exports de sensitivity a utiliser
            for (SensitivityExport sensitivity : param.getSensitivityExport()) {
                String name = SensitivityExportStorage.getName(sensitivity);
                name = name.endsWith(".java") ? name : name + ".java";
                File sensitivityFile = new File(SensitivityExportStorage.getSensitivityExportDirectory(), name);
                if (!sensitivityFile.isFile()) {
                    sensitivityFile = new File(SensitivityExportStorage.getCommunitySensitivityExportDirectory(), name);
                }
                FileUtil.copy(sensitivityFile,
                        new File(tmpDirectory, SensitivityExportStorage.SENSITIVITY_EXPORT_PATH
                                + File.separator + name));
            }

            // copie de tous les scripts a utiliser
            // les script officiel prevalent sur les scripts communautés
            FileUtil.copyRecursively(ScriptStorage.getCommunityScriptDirectory(),
                    tmpDirectory, ".*\\.java$");
            FileUtil.copyRecursively(ScriptStorage.getScriptDirectory(),
                    tmpDirectory, ".*\\.java$");

            // copie de tous les simulateurs a utiliser
            File simulatorFile = new File(SimulatorStorage.getSimulatorDirectory(), param.getSimulatorName());
            if (!simulatorFile.isFile()) {
                simulatorFile = new File(SimulatorStorage.getCommunitySimulatorDirectory(), param.getSimulatorName());
            }
            FileUtil.copy(simulatorFile, new File(tmpDirectory,
                    SimulatorStorage.SIMULATOR_PATH + File.separator
                            + param.getSimulatorName()));

            // convert all file to UTF-8
            //convertAllFile(control, tmpDirectory);
            
            if (compile) {
                compileAllFile(control, tmpDirectory);
            }

            // Sauvegarde du design plan en XML
            if (!StringUtils.isEmpty(xmlDesignPlan)) {
                File simulationDesignPlanFile = SimulationStorage.getMexicoDesignPlan(tmpDirectory);
                FileUtil.writeString(simulationDesignPlanFile, xmlDesignPlan, "utf-8");
            }
            
            // creation du zip
            File result = new File(tmpDirectory.getPath() + ".zip");
            result.deleteOnExit();
            ZipUtil.compress(result, tmpDirectory, null);

            // poussin 20071015: remove temp directory
            if (!FileUtil.deleteRecursively(tmpDirectory)) {
                log.warn(_("isisfish.error.remove.directory", tmpDirectory));
            }

            return result;
        } catch (IOException eee) {
            throw new SimulationException(
                    _("isisfish.error.prepare.information.simulation"), eee);
        } catch (TopiaException eee) {
            throw new SimulationException(
                    _("isisfish.error.prepare.information.simulation"), eee);
        }

    }

    /*
     * Convertit tous les fichiers du répertoire directory en UTF-8,
     * pour eviter les erreurs d'encodage du la compilation
     * sur un autre systeme ne supportant pas l'encodage courant.
     *
     * @param control le controleur
     * @param directory le répertoire a convertir
     *
    protected void convertAllFile(SimulationControl control, File directory) {

        control.setText("Converting file to unicode");

        List<File> fileToConvert = new ArrayList<File>();

        List<File> tmp = FileUtil.find(new File(directory,
                ExportStorage.EXPORT_PATH), ".*\\.java$", true);
        fileToConvert.addAll(tmp);
        
        tmp = FileUtil.find(new File(directory,
                SensitivityExportStorage.SENSITIVITY_EXPORT_PATH), ".*\\.java$", true);
        fileToConvert.addAll(tmp);

        tmp = FileUtil.find(new File(directory, RuleStorage.RULE_PATH),
                ".*\\.java$", true);
        fileToConvert.addAll(tmp);

        tmp = FileUtil.find(new File(directory,
                SimulationPlanStorage.SIMULATION_PLAN_PATH), ".*\\.java$", true);
        fileToConvert.addAll(tmp);

        tmp = FileUtil.find(
                new File(directory, SimulatorStorage.SIMULATOR_PATH),
                ".*\\.java$", true);
        fileToConvert.addAll(tmp);

        //
        // Convertion
        //
        //CompileHelper.convertToUnicode(fileToConvert);
    }*/

    /**
     * Compile les fichiers présent dans le répertoire passé en
     * parametre, ce répertoire est hiérarchisé en: rules, exports, simulators
     * et scripts. Seuls les fichiers des 3 premiers répertoires sont compilés
     * les fichiers du dernier sont compilé par les dépendances qu'on les autres 
     * <p>
     * Il permet donc de compiler facilement tous les fichiers pour une
     * simulation
     *  
     * @param control le controleur
     * @param directory le répertoire où compiler
     */
    protected void compileAllFile(SimulationControl control, File directory) {

        control.setText(_("isisfish.simulation.message.scriptscompilation"));
        long currentTime = System.nanoTime();

        //
        // Recherche des fichiers a compiler
        // On ne prend pas les scripts, car ils sont tous copiés mais pas
        // forcément util. Lors de la compilation des autres fichiers, les
        // script servant réellement seront automatiquement compilé

        List<File> fileToCompile = new ArrayList<File>();

        String[] modules = {
                ExportStorage.EXPORT_PATH,
                RuleStorage.RULE_PATH,
                SensitivityExportStorage.SENSITIVITY_EXPORT_PATH,
                SensitivityAnalysisStorage.SENSITIVITY_ANALYSIS_PATH,
                ScriptStorage.SCRIPT_PATH,
                SimulationPlanStorage.SIMULATION_PLAN_PATH,
                SimulatorStorage.SIMULATOR_PATH
        };

        for (String module : modules) {
            List<File> tmp = FileUtil.find(new File(directory, module), ".*\\.java$", true);
            fileToCompile.addAll(tmp);
        }

        //
        // Compilation
        //

        CompileHelper.compile(directory, fileToCompile, directory, null);
        long time = System.nanoTime() - currentTime;
        control.setText(_("isisfish.message.compilation.time",
                DurationFormatUtils.formatDuration(time / 1000000, "s'.'S")));

    }

}
