/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2011 Ifremer, CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.datastore;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityAnalysis;
import fr.ifremer.isisfish.util.Doc;
import fr.ifremer.isisfish.util.Docable;
import fr.ifremer.isisfish.vcs.VCSException;

/**
 * Cette class permet de stocker les fichiers de calculateur de sensibilité.
 * <p/>
 * Gere les fichiers VCS de type {@link SensitivityAnalysis} (package sensitivityanalysis).
 * 
 * Created: 17 août 2005 11:11:51 CEST
 *
 * @author chatellier <chatellier@codelutin.com>
 * @version $Revision: 3436 $
 * Last update: $Date: 2011-07-07 16:33:45 +0200 (jeu., 07 juil. 2011) $ by : $Author: chatellier $
 */
public class SensitivityAnalysisStorage extends JavaSourceStorage { // SensitivityStorage

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(SensitivityAnalysisStorage.class);

    /** Emplacement de stockage des fichiers de sensibilité */
    public final static String SENSITIVITY_ANALYSIS_PATH = "sensitivityanalysis";

    /** Template freemarker pour les scripts de sensibilité. */
    public static final String SENSITIVITY_ANALYSIS_TEMPLATE = "templates/script/sensitivityanalysis.ftl";

    /** Cache. */
    protected static Map<String, SensitivityAnalysisStorage> sensitivityCache = new ReferenceMap();

    /**
     * Build new {@link SensitivityAnalysisStorage}.
     *
     * @param rootSrc   repertoire root de stockage des calculateurs de sensibilité.
     * @param directory le repertoire ou devrait se trouver le calculateur de sensibilité
     * @param name      le nom du calculateur de sensibilité
     */
    protected SensitivityAnalysisStorage(File rootSrc, File directory, String name) {
        super(rootSrc, directory, name);
    }

    /**
     * Get sensitivity storage directory.
     * 
     * Create directory if not exists.
     * 
     * @return sensitivity storage directory
     */
    public static File getSensitivityAnalysisDirectory() {
        File result = new File(getContextDatabaseDirectory(), SENSITIVITY_ANALYSIS_PATH);
        result.mkdirs();
        return result;
    }

    /**
     * Get community VCS sensitivity directory.
     * 
     * @return community sensitivity directory
     */
    public static File getCommunitySensitivityAnalysisDirectory() {
        File result = new File(getCommunityDatabaseDirectory(), SENSITIVITY_ANALYSIS_PATH);
        result.mkdirs();
        return result;
    }

    /**
     * Retourne le nom de tous les calculateurs de sensibilité existant.
     *
     * @return le nom de tous les calculateurs de sensibilité existans en local
     */
    public static List<String> getSensitivityAnalysisNames() {
        List<String> result = getStorageNames(getSensitivityAnalysisDirectory());
        result.addAll(getStorageNames(getCommunitySensitivityAnalysisDirectory()));
        return result;
    }

    /**
     * Retourne le storage pour le calculateur demandé.
     *
     * @param name le nom du calculateur souhaité
     * @return Le {@link SensitivityAnalysisStorage} pour le calculateur
     */
    public static SensitivityAnalysisStorage getSensitivityAnalysis(String name) {
        String cacheName = getContextDatabaseCacheKey(name);
        SensitivityAnalysisStorage result = sensitivityCache.get(cacheName);
        if (result == null) {
            result = new SensitivityAnalysisStorage(getContextDatabaseDirectory(), getSensitivityAnalysisDirectory(), name);
            sensitivityCache.put(cacheName, result);
        }
        return result;
    }
    
    /**
     * Retourne le storage pour le calculateur demandé.
     *
     * @param name le nom du calculateur souhaité
     * @return Le {@link SensitivityAnalysisStorage} pour le calculateur
     */
    public static SensitivityAnalysisStorage getCommunitySensitivityAnalysis(String name) {
        String cacheName = getCommunityDatabaseCacheKey(name);
        SensitivityAnalysisStorage result = sensitivityCache.get(cacheName);
        if (result == null) {
            result = new SensitivityAnalysisStorage(getCommunityDatabaseDirectory(), getCommunitySensitivityAnalysisDirectory(), name);
            sensitivityCache.put(cacheName, result);
        }
        return result;
    }

    /**
     * Retourne une nouvelle instance du calculateur. Compile le fichier si besoin
     *
     * @return retourne une nouvelle instance du calculateur
     * @throws IsisFishException s'il y a un problème d'instanciation
     */
    public SensitivityAnalysis getNewSensitivityAnalysisInstance() throws IsisFishException {
        Object result = getNewInstance();
        return (SensitivityAnalysis) result;
    }

    /**
     * Effectue un chekout VCS sur le répertoire des calculateurs.
     * 
     * @see VersionStorage#checkout(File, String)
     * @see #SENSITIVITY_ANALYSIS_PATH
     * 
     * @throws VCSException if an error occurs during checkout
     */
    public static void checkout() throws VCSException {
        checkout(IsisFish.config.getDatabaseDirectory(), SENSITIVITY_ANALYSIS_PATH);
    }

    /**
     * Retourne la liste des noms de tous les calculateurs disponibles en local qui
     * ne sont pas encore sur le serveur VCS.
     *
     * @return liste de noms de calculateurs
     */
    static public List<String> getNewSensitivityAnalysisNames() {
        List<String> result = getSensitivityAnalysisNames();
        result.removeAll(getRemoteSensitivityAnalysisNames());
        return result;
    }

    /**
     * Retourne la liste des noms de tous les calculateurs disponibles sur le
     * serveur VCS
     *
     * @return la liste des noms de tous les calculateurs disponibles sur le serveur
     *         VCS. Si le serveur n'est pas disponible la liste retournée est
     *         vide.
     */
    static public List<String> getRemoteSensitivityAnalysisNames() {
        File dir = getSensitivityAnalysisDirectory();
        return getRemoteStorageNames(dir);

    }

    /**
     * Retourne la liste des noms de tous les calculateurs disponibles sur le
     * serveur VCS qui ne sont pas encore en local.
     *
     * @return liste de noms de regions
     * @throws VCSException
     */
    static public List<String> getNewRemoteSensitivityAnalysisNames()
            throws VCSException {
        List<String> result = getRemoteSensitivityAnalysisNames();
        result.removeAll(getSensitivityAnalysisNames());
        return result;
    }

    /**
     * <b>Be ware this method require to instanciate a AnalysePlan, so
     * it would be better to call as often as possible.</b>
     *
     * @return the descript of the instanciate AnalysePlan
     * @see Doc
     * @see Docable
     */
    @Override
    public String getDescription() {
        String result = null;
        try {
            SensitivityAnalysis sensitivityAnalysis = getNewSensitivityAnalysisInstance();
            result = sensitivityAnalysis == null ? null : sensitivityAnalysis.getDescription();
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.description", this));
        }
        return result;
    }

} // SensitivityStorage
