/*
 * #%L
 * IsisFish
 * 
 * $Id: ScriptTreeModel.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.script.model;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.tree.TreeModelSupport;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.datastore.AnalysePlanStorage;
import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.datastore.FormuleStorage;
import fr.ifremer.isisfish.datastore.RuleStorage;
import fr.ifremer.isisfish.datastore.ScriptStorage;
import fr.ifremer.isisfish.datastore.SensitivityExportStorage;
import fr.ifremer.isisfish.datastore.SensitivityStorage;
import fr.ifremer.isisfish.datastore.SimulatorStorage;

/**
 * Tree model for scripts edition.
 * 
 * @author chatellier
 * @version $Revision: 3124 $
 * 
 * Last update : $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * By : $Author$
 */
public class ScriptTreeModel implements TreeModel {

    /** Class logger. */
    private static Log log = LogFactory.getLog(ScriptTreeModel.class);
    
    protected TreeModelSupport modelSupport;

    public List<File> rootFiles;

    /** Add node operation */
    public static final int OPERATION_ADD = 0;
    /** Modify node operation */
    public static final int OPERATION_MODIFY = 1;
    /** Delete Node operation */
    public static final int OPERATION_DELETE = 2;

    public ScriptTreeModel() {
        modelSupport = new TreeModelSupport(this);

        rootFiles = new ArrayList<File>();
        rootFiles.add(AnalysePlanStorage.getAnalysePlanDirectory());
        rootFiles.add(ExportStorage.getExportDirectory());
        rootFiles.add(FormuleStorage.getFormuleDirectory());
        rootFiles.add(RuleStorage.getRuleDirectory());
        rootFiles.add(ScriptStorage.getScriptDirectory());
        rootFiles.add(SensitivityExportStorage.getSensitivityExportDirectory());
        rootFiles.add(SensitivityStorage.getSensitivityDirectory());
        rootFiles.add(SimulatorStorage.getSimulatorDirectory());
    }

    /*
     * @see javax.swing.tree.TreeModel#addTreeModelListener(javax.swing.event.TreeModelListener)
     */
    @Override
    public void addTreeModelListener(TreeModelListener l) {
        modelSupport.addTreeModelListener(l);
    }

    /*
     * @see javax.swing.tree.TreeModel#getChild(java.lang.Object, int)
     */
    @Override
    public Object getChild(Object parent, int index) {

        Object result = null;

        if (parent == rootFiles) {
            result = rootFiles.get(index);
        }
        else {
            File parentFile = (File)parent;
            File[] filesArray = parentFile.listFiles();
            List<File> files = getVersionnableSortedFiles(filesArray);
            result = files.get(index);
        }

        return result;
    }

    /*
     * @see javax.swing.tree.TreeModel#getChildCount(java.lang.Object)
     */
    @Override
    public int getChildCount(Object parent) {
        
        int count = 0;
        if (parent == rootFiles) {
            count = rootFiles.size();
        }
        else {
            File parentFile = (File)parent;
            File[] filesArray = parentFile.listFiles();
            if ( filesArray != null) {
                List<File> files = getVersionnableSortedFiles(filesArray);
                count = files.size();
            }
        }
        return count;
    }

    /*
     * @see javax.swing.tree.TreeModel#getIndexOfChild(java.lang.Object, java.lang.Object)
     */
    @Override
    public int getIndexOfChild(Object parent, Object child) {
        int index = 0;
        if (parent == rootFiles) {
            index = rootFiles.indexOf(child);
        }
        else {
            File parentFile = (File)parent;
            File[] filesArray = parentFile.listFiles();
            List<File> files = getVersionnableSortedFiles(filesArray);
            index = files.indexOf(child);
        }
        return index;
    }

    /*
     * @see javax.swing.tree.TreeModel#getRoot()
     */
    @Override
    public Object getRoot() {
        return rootFiles;
    }

    /*
     * @see javax.swing.tree.TreeModel#isLeaf(java.lang.Object)
     */
    @Override
    public boolean isLeaf(Object node) {
        return getChildCount(node) == 0;
    }

    /*
     * @see javax.swing.tree.TreeModel#removeTreeModelListener(javax.swing.event.TreeModelListener)
     */
    @Override
    public void removeTreeModelListener(TreeModelListener l) {
        modelSupport.removeTreeModelListener(l);
    }

    /*
     * @see javax.swing.tree.TreeModel#valueForPathChanged(javax.swing.tree.TreePath, java.lang.Object)
     */
    @Override
    public void valueForPathChanged(TreePath path, Object newValue) {

    }

    /**
     * Filter input file array and return only sorted collection with only directories.
     * 
     * @param filesArray file array
     * @return sorted list
     */
    protected List<File> getVersionnableSortedFiles(File[] filesArray) {
        List<File> files = new ArrayList<File>();
        for (File fileArray : filesArray) {
            if (IsisFish.vcs.isVersionnableFile(fileArray)) {
                files.add(fileArray);
            }
        }
        Collections.sort(files);
        return files;
    }
    
    /**
     * Notify for file addition.
     * 
     * @param file added file
     */
    public void fileAdded(File file) {
        TreePath path = new TreePath(rootFiles);
        updateChildren(file, path, OPERATION_ADD);
    }
    
    /**
     * Notify for file modification.
     * 
     * @param file modified file
     */
    public void fileModified(File file) {
        TreePath path = new TreePath(rootFiles);
        updateChildren(file, path, OPERATION_MODIFY);
    }
    
    /**
     * Notify for file deletion.
     * 
     * TODO must be called "before" effective deletion
     * 
     * @param file deleted file
     */
    public void fileDeleted(File file) {
        TreePath path = new TreePath(rootFiles);
        updateChildren(file, path, OPERATION_DELETE);
    }

    /**
     * Notify for tree operation.
     * 
     * @param file
     * @param path path to update
     * @param operation operation {@link #OPERATION_ADD #OPERATION_DELETE #OPERATION_MODIFY}
     * @return {@code true} if path has been updated
     */
    protected boolean updateChildren(File file, TreePath path,
            int operation) {
        Object pathLastComponent = (Object) path.getLastPathComponent();

        // pour ses enfants
        boolean updated = false;
        int childCount = getChildCount(pathLastComponent);
        for (int childIndex = 0; !updated && childIndex < childCount; ++childIndex) {
            File child = (File) getChild(pathLastComponent, childIndex);

            TreePath childTreePath = path.pathByAddingChild(child);

            if (file.equals(child)) {

                // this update only node, not all path...
                switch (operation) {

                case OPERATION_ADD:
                    modelSupport.fireChildAdded(path, childIndex, child);
                    // expand path
                    //projectsAndTaskTable.expandPath(path);
                    break;

                case OPERATION_DELETE:
                    modelSupport.fireChildRemoved(path, childIndex, child);
                    break;

                case OPERATION_MODIFY:
                    modelSupport.fireChildChanged(path, childIndex, child);
                    break;

                default:
                    if (log.isErrorEnabled()) {
                        log.error("Unknow operation : " + operation);
                    }
                }

                if (log.isTraceEnabled()) {
                    log.trace(" updated : " + childTreePath);
                }
                updated = true;
            } else {
                updated = updateChildren(file, childTreePath, operation);

                if (updated) {
                    // ...and by recursion update all path
                    modelSupport.firePathChanged(path);
                }
            }
        }

        return updated;
    }
    
    /**
     * Find tree path for a file
     * 
     * @param file to search
     * @return tree path
     */
    public TreePath getTreePathFor(File file) {
        TreePath path = new TreePath(rootFiles);
        path = getRecursiveTreePath(file, path);
        return path;
    }

    /**
     * Find file in tree, and return tree path.
     * 
     * @param file
     * @param path path to update
     * @return tree path
     */
    protected TreePath getRecursiveTreePath(File file, TreePath path) {
        Object pathLastComponent = (Object) path.getLastPathComponent();
        TreePath resultTreePath = null;
        
        int childCount = getChildCount(pathLastComponent);
        for (int childIndex = 0; resultTreePath == null && childIndex < childCount; ++childIndex) {
            File child = (File) getChild(pathLastComponent, childIndex);

            TreePath childTreePath = path.pathByAddingChild(child);

            if (file.equals(child)) {
                resultTreePath = childTreePath;
            } else {
                resultTreePath = getRecursiveTreePath(file, childTreePath);
            }
        }

        return resultTreePath;
    }
}
