/*
 * #%L
 * IsisFish
 * 
 * $Id: RuleParametersTableModel.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.models.rule;

import static org.nuiton.i18n.I18n._;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.swing.table.AbstractTableModel;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.datastore.RuleStorage;
import fr.ifremer.isisfish.rule.Rule;

/**
 * Model de la table de definition de parametre d'une regle.
 * 
 * Columns :
 * <li>Parameter name</li>
 * <li>Parameter value</li>
 *
 * @author chatellier
 * @version $Revision: 3124 $
 * 
 * Last update : $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * By : $Author: chatellier $
 */
public class RuleParametersTableModel extends AbstractTableModel {

    /** Log. */
    private static Log log = LogFactory
            .getLog(RuleParametersTableModel.class);

    /** serialVersionUID. */
    private static final long serialVersionUID = 3169786638868209920L;

    /** Columns names. */
    public final static String[] COLUMN_NAMES = { _("isisfish.common.name"),
            _("isisfish.common.value") };

    protected Rule rule;
    protected List<String> ruleParametersNames;

    /**
     * Constructor with data.
     *  
     * @param rule rule to edit parameters
     */
    public RuleParametersTableModel(Rule rule) {
        super();
        this.rule = rule;

        //TODO map may be sorted
        ruleParametersNames = new ArrayList<String>();
        Map<String, Class<?>> ruleParametersNamesAndTypes = RuleStorage
                .getParameterNames(rule);
        for (String names : ruleParametersNamesAndTypes.keySet()) {
            ruleParametersNames.add(names);
        }
    }

    /*
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    @Override
    public int getColumnCount() {
        return COLUMN_NAMES.length;
    }

    /*
     * @see javax.swing.table.TableModel#getRowCount()
     */
    @Override
    public int getRowCount() {
        return ruleParametersNames.size();
    }

    /*
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {

        Object result = null;

        String name = ruleParametersNames.get(rowIndex);
        switch (columnIndex) {
        case 0:
            result = name;
            break;
        case 1:
            try {
                result = RuleStorage.getParameterValue(rule, name);
            } catch (IsisFishException e) {
                if (log.isErrorEnabled()) {
                    log.debug("Can't get parameters value", e);
                }
            }
            break;
        default:
            throw new IndexOutOfBoundsException("No such column " + columnIndex);
        }

        return result;
    }

    /*
     * @see javax.swing.table.TableModel#getColumnClass(int)
     */
    @Override
    public Class<?> getColumnClass(int columnIndex) {

        Class<?> result = null;

        switch (columnIndex) {
        case 0:
            result = String.class;
            break;
        case 1:
            result = Object.class;
            break;
        default:
            throw new IndexOutOfBoundsException("No such column " + columnIndex);
        }

        return result;
    }

    /*
     * @see javax.swing.table.TableModel#getColumnName(int)
     */
    @Override
    public String getColumnName(int columnIndex) {
        return COLUMN_NAMES[columnIndex];
    }

    /*
     * @see javax.swing.table.TableModel#isCellEditable(int, int)
     */
    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return columnIndex > 0;
    }

    /*
     * @see javax.swing.table.TableModel#setValueAt(java.lang.Object, int, int)
     */
    @Override
    public void setValueAt(Object value, int rowIndex, int columnIndex) {

        if (log.isDebugEnabled()) {
            log.debug("Cell edition (column " + columnIndex + ") = " + value);
        }

        String name = ruleParametersNames.get(rowIndex);
        switch (columnIndex) {
        case 1:
            try {
                RuleStorage.setParameterValue(rule, name, value);
            } catch (IsisFishException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't set parameter value", e);
                }
            }
            break;
        default:
            throw new IndexOutOfBoundsException("Can't edit column "
                    + columnIndex);
        }

    }
}
