/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.input;

import static org.nuiton.i18n.I18n._;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;

import jaxx.runtime.swing.navigation.NavigationTreeModel.NavigationTreeNode;

import org.apache.commons.beanutils.MethodUtils;
import org.apache.commons.lang.ClassUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaDAO;
import org.nuiton.topia.persistence.TopiaEntity;

import fr.ifremer.isisfish.IsisFishDAOHelper;
import fr.ifremer.isisfish.ui.SaveVerifier;
import fr.ifremer.isisfish.ui.sensitivity.SensitivityTabUI;
import fr.ifremer.isisfish.ui.util.ErrorHelper;

/**
 * InputSaveVerifier.
 *
 * @author letellier
 * @version $Revision: 1312 $
 *
 * Last update: $Date: 2008-08-28 10:21:07 +0200 (jeu, 28 aoû 2008) $
 * by : $Author: sletellier $
 */
public class InputSaveVerifier implements SaveVerifier {

    /** Class logger. */
    private static Log log = LogFactory.getLog(InputSaveVerifier.class);

    /** New button. Used to create new {@link #type} component. */
    protected JButton currentNewButton = null;
    /** Delete button. */
    protected JButton currentDeleteButton = null;
    /** Save button. */
    protected JButton currentSaveButton = null;
    /** Cancel button. */
    protected JButton currentCancelButton = null;

    protected ActionListener saveListener = new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
            topiaSave();
        }
    };

    protected ActionListener cancelListener = new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
            topiaCancel();
        }
    };

    protected ActionListener newListener = new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
            topiaCreate();
        }
    };

    protected ActionListener deleteListener = new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
            topiaRemove();
        }
    };

    /** Entity type for new creation. */
    protected String type = null;

    protected boolean editable = false;
    protected boolean changed = false;

    protected NavigationTreeNode currentNode = null;
    protected String currentOnglet = null;

    protected TopiaContext isisContext = null;
    
    /**
     * Map entity key to {@link TopiaEntity}.
     */
    protected HashMap<String, TopiaEntity> currentEntities = new HashMap<String, TopiaEntity>();
    protected HashSet<InputContentUI> currentPanels = new HashSet<InputContentUI>();

    protected InputUI rootUI;
    protected SensitivityTabUI sensUI;

    @Override
    public int checkEdit() {
        int response = JOptionPane.NO_OPTION;
        if (editable) {
            if (changed) {
                // ask user to close edition
                // still in edit mode, must warn user
                response = askUser(_("isisfish.message.page.modified"));
                if (response == JOptionPane.NO_OPTION) {
                    topiaCancel();
                } else if (response == JOptionPane.YES_OPTION) {
                    topiaSave();
                }
            }
        }
        return response;
    }

    /**
     * Ask user option to save non saved datas.
     * 
     * @param message message to display
     * @return user option
     */
    protected int askUser(String message) {
        int response = JOptionPane.showConfirmDialog(rootUI, message,
                _("isisfish.input.menu.commit"),
                JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);
        return response;
    }

    protected void topiaChanged() {
        changed = true;
        setPanelsActifs();
        setEnabled(currentSaveButton, true);
        setEnabled(currentCancelButton, true);
        setEnabled(currentNewButton, false);
        setEnabled(currentDeleteButton, false);
    }

    protected void noModif() {
        changed = false;
        setPanelsActifs();
        setEnabled(currentSaveButton, false);
        setEnabled(currentCancelButton, false);
        setEnabled(currentNewButton, true);
        setEnabled(currentDeleteButton, editable);
    }

    /**
     * Set component enabled state.
     * 
     * @param c component
     * @param enabled enabled state
     */
    protected void setEnabled(Component c, boolean enabled) {
        if (c != null) {
            c.setEnabled(enabled);
        }
    }

    /**
     * Delete one entity and commit the change, try to selected intelligently
     * other node in tree.
     * <p/>
     * Refresh all ui component where name match "input&lt;entity type without
     * package &gt;.*"
     */
    protected void topiaRemove() {
        if (log.isTraceEnabled()) {
            log.trace("remove called");
        }
        String msg = "";
        try {
            boolean doDelete;
            TopiaEntity topiaEntity = (TopiaEntity) currentNode
                    .getJAXXContextValue(rootUI);
            List<TopiaEntity> allWillBeRemoved = topiaEntity.getComposite();
            if (allWillBeRemoved.size() > 0) {
                String label = _("isisfish.message.delete.object", topiaEntity
                        .toString());
                String text = "";
                for (TopiaEntity e : allWillBeRemoved) {
                    text += ClassUtils.getShortClassName(e.getClass()) + " - "
                            + e.toString() + "\n";
                }
                int resp = showTextAreaConfirmationMessage(null, label, text,
                        _("isisfish.message.delete.entities"),
                        JOptionPane.YES_NO_OPTION);
                doDelete = resp == JOptionPane.YES_OPTION;
            } else {
                String text = _("isisfish.message.confirm.delete.object",
                        topiaEntity.toString());
                int resp = JOptionPane.showConfirmDialog(null, text,
                        _("isisfish.message.delete.entity"),
                        JOptionPane.YES_NO_OPTION);
                doDelete = resp == JOptionPane.YES_OPTION;
            }

            if (doDelete) {
                topiaEntity.delete();
                isisContext.commitTransaction();
                msg = _("isisfish.message.remove.finished");
            } else {
                msg = _("isisfish.message.remove.canceled");
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't remove entity: "
                                        + currentEntities.get(0), eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.removeentity"), eee);
        }
        rootUI.setStatusMessage(msg);
        // set noModif BEFORE path changed
        noModif();
        String path = currentNode.getParent().getContextPath();
        rootUI.setTreeModel();
        rootUI.setTreeSelection(path);
    }

    /**
     * Display a JOptionPane with a JTextArea as main component.
     * 
     * @param parent parent
     * @param labelMessage label message
     * @param textMessage text message into area
     * @param title
     * @param option
     * @return user response
     */
    protected int showTextAreaConfirmationMessage(Component parent,
            String labelMessage, String textMessage, String title, int option) {
        JLabel labelForMessage = new JLabel(labelMessage);
        JTextArea areaForMessage = new JTextArea(textMessage);
        areaForMessage.setEditable(false);
        areaForMessage.setAutoscrolls(true);
        JScrollPane spMessage = new JScrollPane(areaForMessage);
        spMessage.setPreferredSize(new Dimension(500, 100)); // don't remove popup is huge

        int response = JOptionPane.showConfirmDialog(parent, new Object[] {
                labelForMessage, spMessage }, title, option);
        return response;
    }

    protected void topiaCreate() {

        if (log.isDebugEnabled()) {
            log.debug("Create called for " + type);
        }

        try {
            String name = type + "_new";

            Method method = MethodUtils.getAccessibleMethod(
                    IsisFishDAOHelper.class, "get" + type + "DAO",
                    TopiaContext.class);
            TopiaDAO<TopiaEntity> dao = (TopiaDAO<TopiaEntity>) method.invoke(
                    null, isisContext);

            TopiaEntity entity = dao.create("name", name);
            entity.update();
            isisContext.commitTransaction();

            String path = currentNode.getParent().getContextPath() + "/"
                    + entity.getTopiaId();
            if (!editable) {
                path = currentNode.getContextPath() + "/" + entity.getTopiaId();
            }

            rootUI.setTreeModel();
            rootUI.setTreeSelection(path);

            rootUI.setStatusMessage(_("isisfish.message.creation.finished"));

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't create entity", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.createentity"), eee);
        }
    }

    /**
     * Save all non saved entities.
     * 
     * Change registered button states.
     * Commit opened topia context.
     */
    protected void topiaSave() {
        try {
            noModif();
            for (TopiaEntity t : currentEntities.values()) {
                t.update();
                if (log.isDebugEnabled()) {
                    log.debug("updating : " + t);
                }
            }
            isisContext.commitTransaction();
            rootUI.repaintNode(currentNode.getContextPath());
            rootUI.setStatusMessage(_("isisfish.message.save.finished"));
        } catch (TopiaException eee) {
            
            // EC-20100401 : ajouté pour avoir un context
            // correct apres un erreur, sinon, on ne peut plus rien faire
            // rien sauver d'autre...
            try {
                isisContext.rollbackTransaction();
                if (log.isErrorEnabled()) {
                    log.error("Can't save entity", eee);
                }
                ErrorHelper.showErrorDialog(_("isisfish.error.input.saveentity"), eee);
            } catch (TopiaException eee2) {
                if (log.isErrorEnabled()) {
                    log.error("Can't save or rollback entity", eee2);
                }
            }
        }
    }

    /**
     * Cancel all modification on entity (rollback), and force reload it.
     */
    protected void topiaCancel() {
        try {
            noModif();
            isisContext.rollbackTransaction();
            Map<String, TopiaEntity> canceledEntity = (Map<String, TopiaEntity>)currentEntities.clone();
            
            currentEntities.clear();
            for (Entry<String, TopiaEntity> currentEntity : canceledEntity.entrySet()) {
                TopiaEntity t = isisContext.findByTopiaId(currentEntity.getValue().getTopiaId());

                // TODO a quoi ca sert de recharger les entités ?
                // desynchronise la précédente
                // fix : org.hibernate.NonUniqueObjectException: a different object with the
                // same identifier value was already associated with the session
                //((TopiaContextImpl)isisContext).getHibernate().evict(t2);
                //t.setTopiaId(null);

                rootUI.repaintNode(currentNode.getContextPath());
                rootUI.setTreeSelection(currentNode.getContextPath());

                String key = currentEntity.getKey();
                addCurrentEntity(t, key);
            }

            // refresh all registred panel
            // to discard modification in UI
            refreshAll();
            rootUI.setStatusMessage(_("isisfish.message.cancel.finished"));
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't cancel modification in region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.cancelentity"), eee);
        }
    }

    protected void setPanelsActifs() {
        for (InputContentUI panel : currentPanels) {
            panel.setActif(editable);
        }
    }

    /**
     * Refresh all registered {@link InputContentUI} component.
     * 
     * Call {@link InputContentUI#refresh()} method on each component. 
     */
    public void refreshAll() {
        
        // chatellier 20090602 refresh() call addCurrentPanel
        // and cause ConcurentModificationException.
        Set<InputContentUI> panelsToRefresh = (HashSet<InputContentUI>)currentPanels.clone();
        for (InputContentUI panel : panelsToRefresh) {
            if (log.isDebugEnabled()) {
                log.debug("Verifier refresh ui : " + panel);
            }
            panel.refresh();
            // do not call refresh action buttons here
            panel.setActif(editable);
        }
    }

    /**
     * Add entity to check for modification.
     * 
     * The verifier register to entity using {@link TopiaEntity#addPropertyChangeListener(PropertyChangeListener)}.
     * So modification have to be done on current entity.
     * 
     * To check for embedded entity, add it too.
     * 
     * Remove all entity with key 
     * 
     * @param currentEntity entity to check
     * @param key specific key (defaut to topiaId)
     */
    public void addCurrentEntity(TopiaEntity currentEntity, String key) {
        if (currentEntity != null) {
            editable = true;
            isisContext = currentEntity.getTopiaContext();
            currentEntity
                    .addPropertyChangeListener(new PropertyChangeListener() {
                        @Override
                        public void propertyChange(PropertyChangeEvent evt) {
                            if (log.isDebugEnabled()) {
                                log.debug("PropertyChanged : "
                                        + evt.getPropertyName()
                                        + " New Value : " + evt.getNewValue());
                            }
                            topiaChanged();
                        }
                    });
            // chatellier 20090602 , pas sur du code suivant
            // plutot ne rien faire, si une autre entite de la meme
            // clé est ajoutée, elle sera ecrasée
            //TopiaEntity entity = getEntity(currentEntity.getClass());
            //if (entity != null) {
            //    currentEntities.remove(entity);
            //}

            this.currentEntities.put(key, currentEntity);
            setPanelsActifs();
        }
    }

    /**
     * Add entity to check for modification.
     * 
     * The verifier register to entity using {@link TopiaEntity#addPropertyChangeListener(PropertyChangeListener)}.
     * So modification have to be done on current entity.
     * 
     * To check for embedded entity, add it too.
     * 
     * Remove all entity with key 
     * 
     * @param currentEntity entity to check
     */
    public void addCurrentEntity(TopiaEntity currentEntity) {
        if (currentEntity != null) {
            addCurrentEntity(currentEntity, currentEntity.getTopiaId());
        }
    }

    public void reset() {
        removeAllEntity();
        removeAllPanels();
        this.currentCancelButton = null;
        this.currentDeleteButton = null;
        this.currentNewButton = null;
        this.currentSaveButton = null;
        this.currentOnglet = null;
        this.currentNode = null;
    }

    public void removeAllEntity() {
        currentEntities.clear();
        editable = false;
        noModif();
        setPanelsActifs();
    }

    public void addCurrentPanel(InputContentUI... panels) {
        for (InputContentUI ui : panels) {
            editable = !currentEntities.isEmpty();
            this.currentPanels.add(ui);
            if (rootUI == null) {
                ui.setSensitivity(true);
                ui.setLayer(true);
            }
            ui.refresh();
            // do not call refresh action buttons here
            ui.setActif(editable);
        }
    }

    public void removeAllPanels() {
        currentPanels.clear();
    }

    public boolean isEditable() {
        return editable;
    }

    public void setSaveButton(JButton saveButton) {
        setSaveButton(saveButton, true);
    }

    public void setSaveButton(JButton saveButton, Boolean listener) {
        if (listener) {
            // TODO what is it for (remove/readd) ?
            saveButton.removeActionListener(saveListener);
            saveButton.addActionListener(saveListener);
        }
        saveButton.setEnabled(changed);
        saveButton.setText(_("isisfish.common.save"));
        this.currentSaveButton = saveButton;
    }

    public void setNewButton(JButton saveButton, String name) {
        setNewButton(saveButton, name, true);
    }

    public void setNewButton(JButton newButton, String t, Boolean listener) {
        if (listener) {
            // TODO what is it for (remove/readd) ?
            newButton.removeActionListener(newListener);
            newButton.addActionListener(newListener);
        }
        newButton.setText(_("isisfish.common.new"));
        newButton.setEnabled(!changed);
        this.type = t;
        this.currentNewButton = newButton;
    }

    public void setCancelButton(JButton cancelButton) {
        cancelButton.removeActionListener(cancelListener);
        cancelButton.addActionListener(cancelListener);
        cancelButton.setText(_("isisfish.common.cancel"));
        cancelButton.setEnabled(changed);
        this.currentCancelButton = cancelButton;
    }

    public void setDeleteButton(JButton deleteButton) {
        setDeleteButton(deleteButton, true);
    }

    public void setDeleteButton(JButton deleteButton, boolean listener) {
        if (listener) {
            deleteButton.removeActionListener(deleteListener);
            deleteButton.addActionListener(deleteListener);
        }
        deleteButton.setEnabled(editable);
        deleteButton.setText(_("isisfish.common.remove"));
        this.currentDeleteButton = deleteButton;
    }

    public TopiaContext getIsisContext() {
        return isisContext;
    }

    public Collection<TopiaEntity> getCurrentEntities() {
        return currentEntities.values();
    }

    public <E extends TopiaEntity> E getEntity(Class<E> clazz, String key) {
        return (E) currentEntities.get(key);
    }

    public <E extends TopiaEntity> E getEntity(Class<E> clazz) {
        for (TopiaEntity te : currentEntities.values()) {
            if (clazz.isInstance(te)) {
                return (E) te;
            }
        }
        return null;
    }

    public String getCurrentOnglet() {
        return currentOnglet;
    }

    public NavigationTreeNode getCurrentNode() {
        return currentNode;
    }

    public void setIsisContext(TopiaContext isisContext) {
        this.isisContext = isisContext;
    }

    public void setEditable(boolean editable) {
        this.editable = editable;
    }

    public void setCurrentOnglet(String currentOnglet) {
        this.currentOnglet = currentOnglet;
    }

    public void setCurrentNode(NavigationTreeNode currentNode) {
        this.currentNode = currentNode;
    }

    protected void setRootPanel(InputUI inputUI) {
        this.rootUI = inputUI;
    }

    public SensitivityTabUI getSensPanel() {
        return sensUI;
    }

    public void setSensPanel(SensitivityTabUI sensUI) {
        this.sensUI = sensUI;
    }
}
