/*
 * #%L
 * IsisFish
 * 
 * $Id: SSHLauncherConfigAction.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, Code Lutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.config;

import static org.nuiton.i18n.I18n._;

import java.awt.Color;
import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;

import javax.swing.JComponent;
import javax.swing.JOptionPane;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.KeyPair;
import com.jcraft.jsch.Session;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.ui.input.InputAction;
import fr.ifremer.isisfish.util.ssh.InvalidPassphraseException;
import fr.ifremer.isisfish.util.ssh.SSHAgent;
import fr.ifremer.isisfish.util.ssh.SSHException;
import fr.ifremer.isisfish.util.ssh.SSHUserInfo;
import fr.ifremer.isisfish.util.ssh.SSHUtils;

/**
 * Action for SSHLauncherConfig UI.
 * 
 * @author chatellier
 * @version $Revision: 3124 $
 * 
 * Last update : $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * By : $Author$
 */
public class SSHLauncherConfigAction {

    /** Class logger. */
    private static Log log = LogFactory.getLog(InputAction.class);

    protected SSHLauncherConfigUI configUI;

    protected String currentSSHserver;
    protected String currentSSHUsername;
    protected File currentSSHKey;
    protected String currentSSHUserhome;
    protected String currentSSHDatapath;
    protected String currentSSHIsisHome;
    protected String currentSSHTempPath;
    protected String currentSSHJavaPath;
    protected String currentSSHPbsBinPath;
    protected String currentSSHPbsQsubOptions;
    protected String currentSSHMaxThreads;
    protected String currentSSHControlInterval;

    /**
     * Constructor with UI.
     * 
     * @param configUI config ui
     */
    public SSHLauncherConfigAction(SSHLauncherConfigUI configUI) {
        this.configUI = configUI;
    }

    /**
     * Reset values with default configuration values.
     */
    public void resetSSHConfiguration() {
        // server config
        configUI.getSshServerField().setText(IsisFish.config.getSimulatorSshServer());
        configUI.getUsernameField().setText(IsisFish.config.getSimulatorSshUsername());
        configUI.getSshKeyField().setText(IsisFish.config.getSSHPrivateKeyFilePath().getAbsolutePath());

        // caparmor config
        configUI.getSshUserhomeField().setText(IsisFish.config.getSimulatorSshUserHome());
        configUI.getSshDatapathField().setText(IsisFish.config.getSimulatorSshDataPath());
        configUI.getSshIsisHomeField().setText(IsisFish.config.getSimulatorSshIsisHome());
        configUI.getSshTemppathField().setText(IsisFish.config.getSimulatorSshTmpPath());
        configUI.getSshJavaPathField().setText(IsisFish.config.getSimulatorSshJavaPath());
        configUI.getSshPbsBinPathField().setText(IsisFish.config.getSimulatorSshPbsBinPath());
        configUI.getSshPbsQsubOptionsField().setText(IsisFish.config.getSimulatorSshPbsQsubOptions());
        configUI.getSshControlIntervalField().setText(String.valueOf(IsisFish.config.getSimulatorSshControlCheckInterval()));
        configUI.getSshMaxThreadsField().setText(String.valueOf(IsisFish.config.getSimulatorSshMaxThreads()));

        doCheck();
    }

    /**
     * Reset display.
     */
    protected void configurationChanged() {
        
        configUI.setConnected(false);
        configUI.getMessageLabel().setText("");
        configUI.getStatusFreeDatabaseLabel().setText(_("isisfish.simulator.ssh.configuration.freespace.label", "isis-database", "0"));
        configUI.getStatusFreeTempLabel().setText(_("isisfish.simulator.ssh.configuration.freespace.label", "isis-tmp", "0"));

    }

    /**
     * Check input format validity.
     */
    public void doCheck() {

        configurationChanged();

        // copy values
        currentSSHserver = configUI.getSshServerField().getText().trim();
        if (currentSSHserver.isEmpty()) {
            setColor(true, configUI.getSshServerField());
        } else {
            setColor(false, configUI.getSshServerField());
        }

        // copy values
        //currentSSHsftpServer = configUI.getSshSftpServerField().getText().trim();

        // can be empty (optionnal use)
        currentSSHUsername = configUI.getUsernameField().getText().trim();
        if (!currentSSHUsername.matches("\\w+")) {
            setColor(true, configUI.getUsernameField());
        } else {
            setColor(false, configUI.getUsernameField());
        }

        // if ssh key is set, must be a valid file
        currentSSHKey = new File(configUI.getSshKeyField().getText().trim());
        if (!currentSSHKey.isFile()) {
            setColor(true, configUI.getSshKeyField());
        } else {
            configUI.getSshKeyButton().setEnabled(false);
            configUI.getSshKeyField().setEnabled(false);
            setColor(false, configUI.getSshKeyField());
        }

        currentSSHDatapath = configUI.getSshDatapathField().getText().trim();
        if (currentSSHDatapath.isEmpty()) {
            setColor(true, configUI.getSshDatapathField());
        } else {
            setColor(false, configUI.getSshDatapathField());
        }

        currentSSHIsisHome = configUI.getSshIsisHomeField().getText().trim();
        if (currentSSHIsisHome.isEmpty()) {
            setColor(true, configUI.getSshIsisHomeField());
        } else {
            setColor(false, configUI.getSshIsisHomeField());
        }

        currentSSHTempPath = configUI.getSshTemppathField().getText().trim();
        if (currentSSHTempPath.isEmpty()) {
            setColor(true, configUI.getSshTemppathField());
        } else {
            setColor(false, configUI.getSshTemppathField());
        }

        currentSSHJavaPath = configUI.getSshJavaPathField().getText().trim();
        if (currentSSHJavaPath.isEmpty()) {
            setColor(true, configUI.getSshJavaPathField());
        } else {
            setColor(false, configUI.getSshJavaPathField());
        }

        currentSSHPbsBinPath = configUI.getSshPbsBinPathField().getText()
                .trim();
        if (currentSSHPbsBinPath.isEmpty()) {
            setColor(true, configUI.getSshPbsBinPathField());
        } else {
            setColor(false, configUI.getSshPbsBinPathField());
        }

        // currentSSHPbsQsubOptions
        currentSSHPbsQsubOptions = configUI.getSshPbsQsubOptionsField().getText().trim();

        // currentSSHMaxThreads
        currentSSHMaxThreads = configUI.getSshMaxThreadsField().getText().trim();
        if (!currentSSHMaxThreads.matches("\\d+")) {
            setColor(true, configUI.getSshMaxThreadsField());
        } else {
            setColor(false, configUI.getSshMaxThreadsField());
        }
        
        // currentSSHControlInterval
        currentSSHControlInterval = configUI.getSshControlIntervalField()
                .getText().trim();
        if (!currentSSHControlInterval.matches("\\d+")) {
            setColor(true, configUI.getSshControlIntervalField());
        } else {
            setColor(false, configUI.getSshControlIntervalField());
        }
    }

    /**
     * Set values in config and force configuration save.
     */
    public void saveSSHConfiguration() {
        IsisFish.config.setSimulatorSshServer(currentSSHserver);
        //IsisFish.config.setSimulatorSshSftpServer(currentSSHsftpServer);
        IsisFish.config.setSimulatorSshUsername(currentSSHUsername);
        IsisFish.config.setSSHPrivateKeyFilePath(currentSSHKey);

        IsisFish.config.setSimulatorSshUserHome(currentSSHUserhome);
        IsisFish.config.setSimulatorSshDataPath(currentSSHDatapath);
        IsisFish.config.setSimulatorSshIsisHome(currentSSHIsisHome);
        IsisFish.config.setSimulatorSshTmpPath(currentSSHTempPath);
        IsisFish.config.setSimulatorSshJavaPath(currentSSHJavaPath);
        IsisFish.config.setSimulatorSshPbsBinPath(currentSSHPbsBinPath);
        IsisFish.config.setSimulatorSshPbsQsubOptions(currentSSHPbsQsubOptions);
        IsisFish.config.setSimulatorSshControlCheckInterval(Integer.parseInt(currentSSHControlInterval));
        IsisFish.config.setSimulatorSshMaxThreads(Integer.parseInt(currentSSHMaxThreads));

        IsisFish.config.saveForUser();
        configUI.dispose();
    }

    /**
     * Close frame.
     */
    public void cancelSSHConfiguration() {
        configUI.dispose();
    }

    /**
     * Realise une connexion ssh et teste les données.
     */
    public void testSSHConfiguration() {
        JSch jsch = new JSch();

        String host = currentSSHserver;
        int port = 22; // by default, 22
        String sPort = null;

        try {
            if (host.indexOf(':') > 0) {
                sPort = host.substring(host.indexOf(':') + 1);
                port = Integer.parseInt(sPort);
                host = host.substring(0, host.indexOf(':'));
            }

            // add ssh key
            boolean sshKeyUsed = false;
            if (currentSSHKey.canRead()) {
                if (log.isInfoEnabled()) {
                    log.info(_("Ssh key found '%s' will be used to connect to",
                            currentSSHKey.getAbsoluteFile(), host));
                }
                jsch.addIdentity(currentSSHKey.getAbsolutePath());
                sshKeyUsed = true;
            } else {
                if (log.isInfoEnabled()) {
                    log.info(_("Can't read ssh key : %s", currentSSHKey));
                }
            }

            Session session = jsch.getSession(currentSSHUsername, host, port);

            // username and password will be given via UserInfo interface.
            SSHUserInfo ui = new SSHUserInfo();
            if (sshKeyUsed) {
                String passphrase = null;
                passphrase = SSHAgent.getAgent().getPassphrase(currentSSHKey);
                ui.setPassphrase(passphrase);
                setTestMessage(_("isisfish.simulator.ssh.configuration.connectingpk"), false);
            } else {
                setTestMessage(_("isisfish.simulator.ssh.configuration.connecting"), false);
            }
            session.setUserInfo(ui);
            session.connect(10000); // timeout

            setTestMessage(_("isisfish.simulator.ssh.configuration.connectionok"), false);

            // get user home
            currentSSHUserhome = getUserHomeDirectory(session);
            configUI.getSshUserhomeField().setText(currentSSHUserhome);
            // get space disk
            String databaseSize = getFolderSize(session, currentSSHDatapath);
            String temppathSize = getFolderSize(session, currentSSHTempPath);
            configUI.getStatusFreeDatabaseLabel().setText(_("isisfish.simulator.ssh.configuration.freespace.label", currentSSHDatapath, databaseSize));
            configUI.getStatusFreeTempLabel().setText(_("isisfish.simulator.ssh.configuration.freespace.label", currentSSHTempPath, temppathSize));
            session.disconnect();

            configUI.setConnected(true);
        } catch (NumberFormatException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't connect", e);
            }
            setTestMessage(_("isisfish.error.simulation.remote.wrongportvalue",
                    sPort), true);
        } catch (JSchException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't connect", e);
            }
            setTestMessage(_(
                    "isisfish.simulator.ssh.configuration.connectionerror", e
                            .getMessage()), true);
        } catch (SSHException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't connect", e);
            }
            setTestMessage(_(
                    "isisfish.simulator.ssh.configuration.connectionerror", e
                            .getMessage()), true);
        } catch (InvalidPassphraseException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't connect", e);
            }
            setTestMessage(
                    _("isisfish.simulator.ssh.configuration.invalidpassphrase"),
                    true);
        }
    }

    /**
     * Get user home directory with an opened session.
     * 
     * @param session opened session
     * @return 'pwd' result
     */
    protected String getUserHomeDirectory(Session session) throws SSHException {

        String command = "pwd";

        Writer output = new StringWriter();
        int exit = SSHUtils.exec(session, command, output);

        if (exit != 0) {
            throw new SSHException(_("Command '%s' fail to execute", command));
        }

        String out = output.toString();
        return out;
    }
    
    /**
     * Get user home directory with an opened session.
     * 
     * @param session opened session
     * @return 'pwd' result
     */
    protected String getFolderSize(Session session, String path) throws SSHException {

        String command = "du -hs \"" + path + "\"";

        Writer output = new StringWriter();
        SSHUtils.exec(session, command, output);

        // la sortie est "35Go   isisdatabase-3"
        String out = output.toString();
        out = out.split("\\s+")[0];
        return out;
    }

    /**
     * Clear database.
     */
    public void clearDatabase() {
        clearFolder(currentSSHDatapath);
    }

    /**
     * Clear temp directory.
     */
    public void clearTempDirectory() {
        clearFolder(currentSSHTempPath);
    }

    /**
     * Clear specified directory.
     * 
     * @param path path to clear
     */
    protected void clearFolder(String path) {

        // prevent, for home deletion, can happens :D
        if (StringUtils.isEmpty(path) || path.trim().equals(".")) {
            throw new IllegalArgumentException("Can't delete directory : " + path);
        }

        int response = JOptionPane.showConfirmDialog(configUI, _("isisfish.simulator.ssh.configuration.freespace.confirmdelete", path),
                _("isisfish.common.confirm"), JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
        
        if (response == JOptionPane.YES_OPTION) {
            JSch jsch = new JSch();
    
            String host = currentSSHserver;
            int port = 22; // by default, 22
            String sPort = null;
    
            try {
                if (host.indexOf(':') > 0) {
                    sPort = host.substring(host.indexOf(':') + 1);
                    port = Integer.parseInt(sPort);
                    host = host.substring(0, host.indexOf(':'));
                }
    
                // add ssh key
                boolean sshKeyUsed = false;
                if (currentSSHKey.canRead()) {
                    if (log.isInfoEnabled()) {
                        log.info(_("Ssh key found '%s' will be used to connect to",
                                currentSSHKey.getAbsoluteFile(), host));
                    }
                    jsch.addIdentity(currentSSHKey.getAbsolutePath());
                    sshKeyUsed = true;
                } else {
                    if (log.isInfoEnabled()) {
                        log.info(_("Can't read ssh key : %s", currentSSHKey));
                    }
                }
    
                Session session = jsch.getSession(currentSSHUsername, host, port);
    
                // username and password will be given via UserInfo interface.
                SSHUserInfo ui = new SSHUserInfo();
                if (sshKeyUsed) {
                    String passphrase = null;
                    passphrase = SSHAgent.getAgent().getPassphrase(currentSSHKey);
                    ui.setPassphrase(passphrase);
                }
                session.setUserInfo(ui);
                session.connect(10000); // timeout
    
                String command = "rm -rf \"" + path + "\"";
    
                if (log.isDebugEnabled()) {
                    log.debug("Removing folder with command : " + command);
                }
                Writer output = new StringWriter();
                int exit = SSHUtils.exec(session, command, output);
    
                // get space disk
                String databaseSize = getFolderSize(session, currentSSHDatapath);
                String temppathSize = getFolderSize(session, currentSSHTempPath);
                configUI.getStatusFreeDatabaseLabel().setText(_("isisfish.simulator.ssh.configuration.freespace.label", currentSSHDatapath, databaseSize));
                configUI.getStatusFreeTempLabel().setText(_("isisfish.simulator.ssh.configuration.freespace.label", currentSSHTempPath, temppathSize));
                session.disconnect();
                
                session.disconnect();
                if (exit != 0) {
                    throw new SSHException(_("Command '%s' fail to execute", command));
                }
            }
            catch (Exception e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't connect", e);
                }
                setTestMessage(_("isisfish.simulator.ssh.configuration.connectionerror", e.getMessage()), true);
            }
        }
        else {
            if (log.isInfoEnabled()) {
                log.info("Delete action canceled by user");
            }
        }
    }

    /**
     * Generate new SSH key.
     */
    protected void generateSSHKey() {
        if (currentSSHKey.exists()) {
            throw new IllegalArgumentException("Can't overwrite ssh key");
        }

        try {
            // make parent dir
            if (currentSSHKey.getParentFile() != null
                    && !currentSSHKey.getParentFile().exists()) {
                currentSSHKey.getParentFile().mkdirs();
            }
            JSch jsch = new JSch();
            KeyPair kpair = KeyPair.genKeyPair(jsch, KeyPair.RSA, 2048);
            //kpair.setPassphrase(passphrase);
            kpair.writePrivateKey(currentSSHKey.getAbsolutePath());
            kpair.writePublicKey(currentSSHKey.getAbsolutePath() + ".pub",
                    currentSSHUsername + "@forIsisFish");
            if (log.isInfoEnabled()) {
                log.info("Finger print: " + kpair.getFingerPrint());
            }
            kpair.dispose();
        } catch (JSchException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't make ssh key", e);
            }
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't make ssh key", e);
            }
        }

        // refresh ckeck
        doCheck();
    }

    /**
     * Set color depending of field validity.
     * 
     * 
     * @param invalid valid field
     * @param component component to set color
     */
    protected void setColor(boolean invalid, JComponent component) {
        component.setForeground(invalid ? Color.RED : Color.BLACK);
    }

    /**
     * Set message in message field with status color.
     * 
     * @param message message to display
     * @param error error status
     */
    protected void setTestMessage(String message, boolean error) {
        configUI.getMessageLabel().setForeground(
                error ? Color.RED : Color.GREEN.darker());
        configUI.getMessageLabel().setText(message);
    }
}
