/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator.sensitivity.domain;

/**
 * Domaine continu portant sur variable pouvant prendre diférente valeur par
 * rapport à une valeur initiale.
 * 
 * Composé d'un borne min et max et d'un opération.
 * 
 * @author chatellier
 * @version $Revision: 1.0 $
 * 
 * Last update : $Date: 24 févr. 2009 $ By : $Author: chatellier $
 */
public class EquationContinuousDomain extends
        ContinuousDomain<Double,Double> {

    /** serialVersionUID. */
    private static final long serialVersionUID = -2037768174807839046L;

    /** Variable name (must be java valid identifier) */
    protected String variableName;

    /** Operation. (=,+,-,/) */
    protected String operator;

    /** Coefficient in percent */
    protected Double coefficient;

    /** Reference value*/
    protected Double referenceValue;
    
    /** Value used to create the equation */
    protected Double value;

    /**
     * Empty constructor.
     */
    public EquationContinuousDomain() {

    }

    @Override
    public String toString() {
        String result = "variableName=" + variableName + " operator=" + operator +
                " coefficient=" + coefficient + " referenceValue=" + referenceValue +
                " value=" + value;
        return result;
    }

    /**
     * Get variable name.
     * 
     * @return the variableName
     */
    public String getVariableName() {
        return variableName;
    }

    /**
     * Set variable name.
     * 
     * @param variableName
     *            the variableName to set
     */
    public void setVariableName(String variableName) {
        this.variableName = variableName;
    }

    /**
     * Get operator.
     * 
     * @return the operator
     */
    public String getOperator() {
        return operator;
    }

    /**
     * Set operator.
     * 
     * @param operator
     *            the operator to set
     */
    public void setOperator(String operator) {
        this.operator = operator;
    }

    /**
     * Get coefficient.
     * 
     * @return the coefficient
     */
    public Double getCoefficient() {
        return coefficient;
    }

    /**
     * Set coefficient.
     * 
     * @param coefficient the coefficient to set
     */
    public void setCoefficient(Double coefficient) {
        this.coefficient = coefficient;
    }

    /**
     * Get reference value.
     * 
     * @return the reference value
     */
    public Double getReferenceValue() {
        return referenceValue;
    }

    /**
     * Set reference value.
     * 
     * @param referenceValue the reference value.to set
     */
    public void setReferenceValue(Double referenceValue) {
        this.referenceValue = referenceValue;
    }
    
    /**
     * {@inheritDoc}.
     * 
     * In equation continuous domain, always return 0
     */
    @Override
    public double getMinBound() {
        return 0;
    }

    /**
     * {@inheritDoc}.
     * 
     * In equation continuous domain, always return 1
     */
    @Override
    public double getMaxBound() {
        return 1;
    }

    /**
     * {@inheritDoc}.
     * 
     * In equation continuous domain, value appear in continuous domain values
     * only. Here : return identifier.
     */
    @Override
    public Double getValueForIdentifier(Double identifier) {
        
        this.value = identifier;
        
        if (operator.equals("+")) {
            return  ((Double) (referenceValue + ((referenceValue + coefficient) * ((2 * value - 1)))));
        } else if (operator.equals("-")) {
            return ((Double) (referenceValue + ((referenceValue - coefficient) * ((2 * value - 1)))));
        } else if (operator.equals("*")) {
            return ((Double) (referenceValue + ((referenceValue * coefficient) * ((2 * value - 1)))));
        } else if (operator.equals("/")) {
            return ((Double) (referenceValue + ((referenceValue / coefficient) * ((2 * value - 1)))));
        }
        return identifier;
    }
    
    /**
     * Get value used to create the equation.
     * 
     * @return the value
     */
    public Double getValue() {
        return this.value;
    }

    /**
     * Set value used to create the equation.
     * 
     * @param value
     *            the value to set
     */
    public void setValue(Double value) {
        this.value = value;
    }
    
    public EquationContinuousDomain clone(){
        EquationContinuousDomain cloned = new EquationContinuousDomain();
        cloned.setCardinality(this.cardinality);
        cloned.setMaxBound(this.maxBound);
        cloned.setMinBound(this.minBound);
        cloned.setCoefficient(this.coefficient);
        cloned.setOperator(this.operator);
        cloned.setValue(this.value);
        cloned.setReferenceValue(this.referenceValue);
        cloned.setVariableName(this.variableName);
        return cloned;
    }
    
}
