/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, Code Lutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.mexico;

import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xml.serialize.OutputFormat;
import org.apache.xml.serialize.XMLSerializer;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.nuiton.math.matrix.MatrixFactory;
import org.nuiton.math.matrix.MatrixIterator;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.persistence.TopiaEntity;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.mexico.xml.DesignPlanXMLVisitor;
import fr.ifremer.isisfish.mexico.xml.DomXMLParser;
import fr.ifremer.isisfish.simulator.sensitivity.DesignPlan;
import fr.ifremer.isisfish.util.ConverterUtil;

/**
 * Mexico helper class.
 *
 * @author chatellier
 * @version $Revision: 3124 $
 * 
 * @since 3.2.0.4
 * 
 * Last update : $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * By : $Author: chatellier $
 */
public class MexicoHelper {

    /** Class logger. */
    private static Log log = LogFactory.getLog(MexicoHelper.class);

    /**
     * Get xml representation of a design plan.
     * 
     * @param designPlan design plan
     * @return xml design plan representation
     */
    public static String getDesignPlanAsXML(DesignPlan designPlan) {

        DesignPlanXMLVisitor visitor = new DesignPlanXMLVisitor();
        designPlan.accept(visitor);
        String designPlanXml = visitor.getXML();

        // apply beautiful xml indented format
        designPlanXml = MexicoHelper.formatXML(designPlanXml);

        return designPlanXml;

    }

    /*
     * Get xml representation of a scenario.
     * 
     * @param scenarios sensitivity scenarios
     * @return xml sensitivity scenarios representation
     * 
     * TODO : non finished and untested
     *
    public static String getSensitivityScenarionsAsXML(
            SensitivityScenarios scenarios) {

        SensitivityScenariosXMLVisitor visitor = new SensitivityScenariosXMLVisitor();
        scenarios.accept(visitor);
        String sensitivityScenariosXml = visitor.getXML();
        return sensitivityScenariosXml;

    }*/

    /**
     * Parse xmlFile with sax, and return a {@link DesignPlan}.
     * 
     * @param xmlFile file path to parse
     * @param topiaContext database context
     * @return DesignPlan
     * @throws IOException 
     */
    public static DesignPlan getDesignPlanFromXML(File xmlFile, TopiaContext topiaContext)
            throws IOException {
        DesignPlan designPlan = null;
        try {
            SAXReader reader = new SAXReader();
            // don't use reader.read(String);
            // don't work on windows because of : in path
            // Document doc = reader.read(xmlFile);
            reader.setEncoding("utf-8");
            Document doc = reader.read(xmlFile);
            Element root = doc.getRootElement();
            designPlan = DomXMLParser.parseDesignPlan(root, topiaContext);
        } catch (DocumentException e) {
            throw new IOException(e);
        }
        return designPlan;
    }

    /*
     * Parse xmlFile with sax, and return a {@link SensitivityScenarios}.
     * 
     * @param xmlFile file path to parse
     * @return SensitivityScenarios
     * @throws IOException
     * 
     * TODO : non finished and untested
     *
    public static SensitivityScenarios getSensitivityScenariosFromXML(
            String xmlFile) throws IOException {
        SensitivityScenarios scenarios = null;
        try {
            SAXReader reader = new SAXReader();
            Document doc = reader.read(xmlFile);
            Element root = doc.getRootElement();
            scenarios = DomXMLParser.parseSensitivityScenarios(root);
        } catch (DocumentException e) {
            throw new IOException(e);
        }
        return scenarios;
    }*/
    
    /**
     * Format xml string.
     * 
     * @param unformattedXml non formatted xml string (must be valid xml)
     * @return xml, formatted and indented
     * 
     * @throws IsisFishRuntimeException
     * @throws IllegalArgumentException if input xml is not valid
     * 
     * @deprecated use standard java xml api instead of xerces
     */
    @Deprecated
    public static String formatXML(String unformattedXml) {

        try {
            // parseXmlFile
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder db = dbf.newDocumentBuilder();
            InputSource is = new InputSource(new StringReader(unformattedXml));
            org.w3c.dom.Document document = db.parse(is);

            // format
            OutputFormat format = new OutputFormat(document);
            format.setLineWidth(65);
            format.setIndenting(true);
            format.setIndent(2);
            Writer out = new StringWriter();
            XMLSerializer serializer = new XMLSerializer(out, format);
            serializer.serialize(document);

            return out.toString();
        } catch (IOException e) {
            throw new IsisFishRuntimeException(e);
        } catch (ParserConfigurationException e) {
            throw new IllegalArgumentException(e);
        } catch (SAXException e) {
            throw new IllegalArgumentException(e);
        }
    }
    
    /**
     * Transform matrix into XML mexico format.
     * 
     * Format is :
     * <pre>
     *   <mx name="une matrice">
     *    <dimension name="classe" size="2">
     *      <label>jeune</label>
     *      <label>vieux</label>
     *    </dimension>
     *    <dimension name="x" size="3">
     *      <label>0</label>
     *      <label>1</label>
     *      <label>2</label>
     *    </dimension>
     *    <d>2.0</d>
     *    <d>3.1415</d>
     *    <d>2.0</d>
     *    <d>3.1415</d>
     *    <d>2.0</d>
     *    <d>3.1415</d>
     *  </mx>
     * </pre>
     * 
     * @param matrix matrix
     * @return matrix as xml or {@code null} if matrix is null
     * @since 3.3.0.0
     */
    public static String getMatrixAsXML(MatrixND matrix) {
        
        if (matrix == null) {
            return null;
        }

        StringBuffer buffer = new StringBuffer();

        // matrix name
        buffer.append("<mx name=\"" + matrix.getName() + "\">");
        
        // matrix dimensions
        for (int dimIndex = 0 ; dimIndex < matrix.getDim().length ; dimIndex++) {
            List<?> semantics = matrix.getSemantic(dimIndex);
            buffer.append("<dimension name=\"" + matrix.getDimensionName(dimIndex) + "\" size=\"" + semantics.size() + "\">");
            for (Object semantic : semantics) {
                /*if (semantic == null) {
                    buffer.append("<label/>");
                }
                else if (semantic instanceof TopiaEntity) {
                    TopiaEntity semanticTE = (TopiaEntity)semantic;
                    buffer.append("<label>" + semanticTE.getTopiaId() + "</label>");
                }
                else {
                    buffer.append("<label>" + semantic.toString() + "</label>");
                }*/
                
                buffer.append("<label>");
                appendString(buffer, semantic);
                buffer.append("</label>");
            }
            buffer.append("</dimension>");
        }

        // matrix data
        for (MatrixIterator mi = matrix.iterator(); mi.next();) {
            // d for double
            buffer.append("<d>" + mi.getValue() + "</d>");
        }

        buffer.append("</mx>");

        return buffer.toString();
    }

    /**
     * Append object type and value in stringbuffer.
     * 
     * Append it as :
     * <pre>fqn(value)</pre>
     * 
     * @param buffer buffer to append to
     * @param o value to append
     * @return stringbuffer
     */
    protected static StringBuffer appendString(StringBuffer buffer, Object o) {
        if (o == null) {
            buffer.append("null()");
        } else {
            String qualifiedName = getQualifiedName(o);
            buffer.append(qualifiedName).append("(");
            ConvertUtilsBean beanUtils = ConverterUtil.getConverter(null);
            buffer.append(beanUtils.convert(o));
            buffer.append(")");
        }
        return buffer;
    }

    /**
     * Parse a dom element (mx) as a {@link MatrixND}.
     * 
     * Format is :
     * <pre>
     *   <mx name="une matrice">
     *    <dimension name="classe" size="2">
     *      <label>jeune</label>
     *      <label>vieux</label>
     *    </dimension>
     *    <dimension name="x" size="3">
     *      <label>0</label>
     *      <label>1</label>
     *      <label>2</label>
     *    </dimension>
     *    <d>2.0</d>
     *    <d>3.1415</d>
     *    <d>2.0</d>
     *    <d>3.1415</d>
     *    <d>2.0</d>
     *    <d>3.1415</d>
     *  </mx>
     * </pre>
     * 
     * @param mxElement dom element
     * @param context topia context
     * @return matrix nd
     */
    public static MatrixND getMatrixFromXml(Element mxElement, TopiaContext context) {

        String name = mxElement.attributeValue("name");

        // get dimension names and semantics
        List<String> dimNames = new ArrayList<String>();
        List<List<?>> semantics = new ArrayList<List<?>>();

        List<Element> dimensionElements = mxElement.elements("dimension");
        for (Element dimensionElement : dimensionElements) {
            String dimName = dimensionElement.attributeValue("name");
            dimNames.add(dimName);

            // parse sub semantics
            List<Object> semantic = new ArrayList<Object>();
            List<Element> labelElements = dimensionElement.elements("label");
            for (Element labelElement : labelElements) {
                String content = labelElement.getText();
                Object value = null;

                if (content != null) {
                    content = content.trim();

                    Pattern matrixPattern = Pattern.compile("^(.*)\\((.*)\\)$");
                    Matcher matcher = matrixPattern.matcher(content);

                    if (matcher.find()) {
                        String objectType = matcher.group(1);
                        String objectString = matcher.group(2);

                        if (log.isDebugEnabled()) {
                            log.debug("Looking for object : " + objectType + ":" + objectString);
                        }

                        if (!"null".equals(objectType)) {
                            ConvertUtilsBean beanUtils = ConverterUtil.getConverter(context);
                            try {
                                value = beanUtils.convert(objectString, Class.forName(objectType));
                            } catch (Exception e) {
                                // if can't create object, put String representation as semantics
                                value = objectType + "(" + objectString + ")";
                                if (log.isWarnEnabled()) {
                                    log.warn("Can't parse '" + content + "' as valid semantic");
                                }
                            }
                        }
                    }
                    else {
                        if (log.isWarnEnabled()) {
                            log.warn("Can't parse '" + content + "' as valid semantic");
                        }
                    }
                }
                // always add value even if value is null
                semantic.add(value);
            }
            semantics.add(semantic);
        }

        MatrixND result = MatrixFactory.getInstance().create(name,
                semantics.toArray(new List<?>[semantics.size()]),
                dimNames.toArray(new String[dimNames.size()]));

        MatrixIterator iterator = result.iterator();
        // TODO it's d for double here, can be int...
        List<Element> values = mxElement.elements("d");
        for (Element value : values) {
            iterator.next();
            String text = value.getText().trim();
            double doubleValue = Double.parseDouble(text);
            iterator.setValue(doubleValue);
        }

        return result;
    }

    /**
     * Return object fully qualified name excepted for {@link TopiaEntity}.
     * 
     * @param o object to get fqn
     * @return fqn for mexico file format
     */
    protected static String getQualifiedName(Object o) {
        String qualifiedName;
        if (o instanceof TopiaEntity) {
            qualifiedName = TopiaEntity.class.getName();
        }
        else {
            qualifiedName = o.getClass().getName();
        }
        return qualifiedName;
    }
}
