/*
 * #%L
 * IsisFish
 * 
 * $Id: VersionStorage.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.datastore;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;
import org.nuiton.util.ListenerSet;
import org.nuiton.util.Version;

import fr.ifremer.isisfish.IsisConfig;
import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.vcs.VCS;
import fr.ifremer.isisfish.vcs.VCSException;

/**
 * Classe permettant de géré l'interaction avec le VSC.
 *
 * @author poussin
 * @version $Revision: 3124 $
 *
 * Last update: $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * by : $Author: chatellier $
 */
public abstract class VersionStorage {

    /** Logger for this class */
    private static final Log log = LogFactory.getLog(VersionStorage.class);

    /** Listener set. Manage single reference and weak reference. */
    protected static ListenerSet<StorageChangeListener> changeListeners = new ListenerSet<StorageChangeListener>();

    protected File root = null;

    protected File file = null;

    /**
     * Par exemple si on a /home/poussin/isis-database comme root
     * il faut que file soit un fichier ou  un sous fichiers dans root
     *
     * @param root le repertoire racine de mise en VCS (HOME VCS)
     * @param file le fichier a gérer.
     */
    protected VersionStorage(File root, File file) {
        this.root = root;
        this.file = file;
    }

    /**
     * Return context root directory to use for all data files depending on
     * context (simulation/no simulation).
     * 
     * In simulation context, must look for files in simulation directory
     * instead of isis database.
     * 
     * TODO better place in other storage, but needed for region anad java scrits
     */
    protected static File getContextDatabaseDirectory() {
        return IsisFish.config.getContextDatabaseDirectory();
    }

    /**
     * Get cache storage key to use depending on context storage used.
     * 
     * Two simulation must have their own cache.
     * 
     * @return context cache key
     * 
     * TODO better place in other storage, but needed for region anad java scrits
     */
    protected static String getContextDatabaseCacheKey(String key) {
        String result = getContextDatabaseDirectory().getAbsolutePath() + key;
        return result;
    }

    /**
     * Get {@link VCS}.
     * 
     * @return VCS
     */
    protected static VCS getVCS() {
        return IsisFish.vcs;
    }

    /**
     * Permet de demander la preparation des fichiers pour etre envoyé vers le VCS.
     */
    protected abstract void prepare();

    /**
     * Get VCS root directory.
     * 
     * @return the root.
     */
    public File getRoot() {
        return this.root;
    }

    /**
     * Get file.
     * 
     * @return the file.
     */
    public File getFile() {
        return this.file;
    }

    /**
     * Indique s'il a deja ete ajouté au VCS. Si file est un repertoire
     * (ex pour Region) alors vrai meme si tous les fichiers du repertoire
     * ne sont pas sur le VCSNone
     *
     * @return {@code true} si deja dans le VCS
     * @throws VCSException
     */
    public boolean isOnRemote() throws VCSException {
        return getVCS().isOnRemote(getFile());
    }

    /**
     * Permet de savoir si un fichier doit etre géré par le vcs ou non.
     * L'implantation par defaut exclus les répertoires VCS, il faut
     * toujours appeler le super si on surcharge la methode.
     *
     * @param file le fichier a tester
     * @return {@code true} si le fichier est versionné
     */
    protected boolean isVersionnableFile(File file) {

        // Dans le cas ou on essaye de savoir si file
        // est versionnable dans le storage courant
        // mais le VCS est toujours celui d'ISIs en static,
        // si le storage est alleur que ce VCS
        // cela ne fonctionne pas.

        // FIXME il ne faurdrait pas que le VCS soit
        // statique , mais contextuel au this.file
        // du storage

        boolean result = true;
        
        // Ca fait une erreur de fichier non versionne
        // si on essaye de l'appeler sur le working directory
        // dans ce cas, on retourne la valeur par defaut "true"
        if (!getVCS().getLocalRepository().equals(this.file)) {
            result = getVCS().isVersionnableFile(file);
        }
        return result;
    }

    /**
     * Donne la liste de tous les fichiers que le VCS doit gérer.
     * Par defaut parcours tous les repertoires et ajouter tous les fichiers
     * et repertoire trouvé. Il est possible d'exclure des fichiers et/ou
     * repertoire en surchargeant {@link #isVersionnableFile(File)}
     *
     * @param current le fichier
     * @param result la liste des fichiers à traiter
     * @return La liste des fichiers a géré par le CVS pour un add, remove
     *         update
     */
    protected List<File> getFiles(File current, List<File> result) {
        if (isVersionnableFile(current)) {
            result.add(current);
            if (current.exists() && current.isDirectory()) {
                for (File child : current.listFiles()) {
                    getFiles(child, result);
                }
            }
        }
        return result;
    }

    /**
     * Donne la liste de tous les fichiers à gérer par le VCS.
     * Ceci inclu la liste des répertoires pour aller de ce storage
     * jusqu'a la racine VCSNone si withParent est vrai
     *
     * @param withParent si vrai inclu les parents
     * @return la liste des fichiers trouvés
     */
    protected List<File> getFiles(boolean withParent) {
        List<File> result = new ArrayList<File>();

        // on ajoute tous les directories jusqu'a root
        File current = getFile();
        while (withParent && current != null && !current.equals(getRoot())) {
            current = current.getParentFile();
            if (current != null) {
                // insere les parents avant tous les autres
                result.add(0, current);
            }
        }
        getFiles(getFile(), result);
        return result;
    }

    /**
     * Permet d'ajouter ce storage dans le VCSNone. Cela prend effet immediatement
     * (un commit est fait).
     *
     * @param msg le message indiquant le but du fichier
     * @throws VCSException si pb pendant l'op
     *
     */
    public void add(String msg) throws VCSException {
        prepare();

        // parent folder must be added too
        List<File> files = getFiles(true);

        // we can sure remove module directory and root directory
        files.remove(IsisFish.config.getDatabaseDirectory());

        if (log.isDebugEnabled()) {
            log.debug("files to add: " + files);
        }
        getVCS().add(files, msg);
    }

    /**
     * Permet de supprimer un fichier ou répertoire versionné ou non.
     *
     * @param vcsDelete si vrai alors le fichier sera aussi supprimé sur le
     *                  vcs si elle existait. Cela prend effet immediatement (un commit est fait)
     * @throws StorageException if delete operation fail
     */
    public void delete(boolean vcsDelete) throws StorageException {
        if (vcsDelete) {
            
            // parent folder must not be deleted
            List<File> files = getFiles(false);

            if (log.isDebugEnabled()) {
                log.debug("About to delete : " + files);
            }

            try {
                getVCS().delete(files, _("isisfish.versionStorage.removed"));
            } catch (VCSException eee) {
                throw new StorageException(
                        _("isisfish.error.delete.vcs.files"), eee);
            }
        }
        //TODO There is a bug to fix ? some files are not deleted!
        // due to h2, storage need to be closed before deletion
        if (getFile().isDirectory()) {
            FileUtil.deleteRecursively(getFile());
        } else {
            getFile().delete();
        }
    }

    /**
     * Permet d'envoyer des modifications faite en local sur le VCS.
     *
     * @param msg le message indiquant le type des modifications
     * @throws VCSException si pb pendant l'op
     */
    public void commit(String msg) throws VCSException {
        // on appelle en fait add, car il y a peut-etre des nouveaux fichiers
        // a ajouter au VCS et ca doit etre fait automatiquement
        add(msg);
    }

    /**
     * Permet de mettre a jour le fichier local en fonction de ce qu'il y
     * a sur le VCS
     *
     * @throws VCSException si pb pendant l'op
     */
    public void update() throws VCSException {
        prepare();
        getVCS().update(getFile(), true);
    }

    public boolean isUpToDate() throws VCSException {
        prepare();
        return getVCS().isUpToDate(getFile());
    }

    /**
     * Permet de ramener tout un répertoire du VCS. Utile seulement pour le
     * premier lancement pour scipts et exports.
     *
     * @param destDir le repertoire parent
     * @param module le repertoire qui peut etre scripts ou exports
     * @throws VCSException si pb pendant l'opération
     */
    public static void checkout(File destDir, String module)
            throws VCSException {
        // Si on utilise pas le bon tag on change de tag
        Version tag = IsisConfig.getApiVersion();
        if (!getVCS().isTag(tag)) {
            // pas de tag pour cette version, on checkout le trunk
            tag = null;
        }

        File file = new File(destDir, module);
        getVCS().update(file, true);
    }

    /**
     * Retourne la liste des noms de toutes les storages disponible en local
     *
     * @param directory le répertoire dans lequel vie l'ensemble des storage
     * @return la liste des noms de toutes les storages disponible en local
     */
    public static List<String> getStorageNames(File directory) {
        List<String> result = new ArrayList<String>();

        if (directory.exists()) {
            for (File f : directory.listFiles()) {
                if (getVCS().isVersionnableFile(f)) {
                    result.add(f.getName());
                }
            }
        }
        Collections.sort(result);

        return result;
    }

    /**
     * Retourne la liste des noms de tous les storages disponibles sur le
     * serveur VCSNone
     *
     * @param directory le répertoire sur le VCSNone ou doivent se trouver
     *                  les storages (regions, simulations)
     * @return la liste des noms de tous les storages disponibles sur le
     *         serveur VCSNone. Si le serveur n'est pas disponible la liste retournée
     *         est vide.
     */
    public static List<String> getRemoteStorageNames(File directory) {
        List<String> result = null;
        try {
            result = getVCS().getFileList(directory);
        } catch (VCSException e) {
            if (log.isWarnEnabled()) {
                log.warn("Error during connection to VCS server", e);
            }
            result = new ArrayList<String>();
        }
        Collections.sort(result);
        return result;
    }

    /**
     * Fire a change event to all registred listeners.
     * 
     * For example, to notify, that a {@link JavaSourceStorage} file has been added...
     * 
     * @param event event to fire
     */
    protected static void fireDataChanged(StorageChangeEvent event) {
        try {
            changeListeners.fire("versionDataChanged", event);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't fire data change event", e);
            }
        }
    }

    /**
     * Add listener to be notified on change to storage.
     * 
     * @param listener listener to add
     */
    public static void addStorageChangeListener(StorageChangeListener listener) {
        changeListeners.add(listener);
    }

    /**
     * Remove listener for storage change notification.
     * 
     * @param listener listener to remove
     */
    public static void removeStorageChangeListener(
            StorageChangeListener listener) {
        changeListeners.remove(listener);
    }
}
