/*
 * #%L
 * IsisFish
 * 
 * $Id: ScriptStorage.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2005 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.datastore;

import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.util.DocHelper;
import fr.ifremer.isisfish.util.Doc;
import fr.ifremer.isisfish.util.Docable;
import fr.ifremer.isisfish.vcs.VCSException;

import static org.nuiton.i18n.I18n._;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.util.List;
import java.util.Map;

/**
 * Gestion des fichers CVS de type scripts (appartenant au module scripts).
 * 
 * Created: 18 août 2005 15:07:36 CEST
 *
 * @author Grégoire DESSARD <dessard@codelutin.com>
 * @version $Revision: 3124 $
 *
 * Last update: $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * by : $Author: chatellier $
 */
public class ScriptStorage extends JavaSourceStorage implements Docable  {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(ScriptStorage.class);
    
    public static final String SCRIPT_PATH = "scripts";

    /** Template freemarker pour les scripts. */
    public static final String SCRIPT_TEMPLATE = "templates/script/script.ftl";
    
    @SuppressWarnings("unchecked")
    static private Map<String, ScriptStorage> scriptsCache =
            (Map<String, ScriptStorage>) new ReferenceMap();

    /**
     * 
     * Constructeur
     * 
     * @param rootSrc le repertoire root de stockage
     * @param directory le repertoire des scripts
     * @param name le nom du script
     */
    protected ScriptStorage(File rootSrc, File directory, String name) {
        super(rootSrc, directory, name);
    }
    
    static public File getScriptDirectory() {
        File result = new File(getContextDatabaseDirectory(), SCRIPT_PATH);
        result.mkdirs();
        return result;
    }
    
    /**
     * Retourne une nouvelle instance de la regle. Compile le fichier si besoin
     * 
     * @return une nouvelle instance
     * @throws IsisFishException
     */
    public Object getNewScriptInstance() throws IsisFishException {
        return getNewInstance();
    }
    
    /**
     * Retourne le storage pour la regle demandée
     * 
     * @param name le nom de la regle souhaitée
     * @return Le storage pour la regle
     */
    static public ScriptStorage getScript(String name) {
        String cacheName = getContextDatabaseCacheKey(name);
        ScriptStorage result = scriptsCache.get(cacheName);
        if (result == null) {
            result = new ScriptStorage(getContextDatabaseDirectory(),
                    getScriptDirectory(), name);
            scriptsCache.put(cacheName, result);
        }
        return result;
    }

    static public void checkout() throws VCSException {
        checkout(IsisFish.config.getDatabaseDirectory(), SCRIPT_PATH);
    } 
    
    /**
     * Retourne la liste des noms de toutes les régions disponible en local
     * 
     * @return la liste des noms de toutes les régions disponible en local
     */
    static public List<String> getScriptNames() {
        File dir = getScriptDirectory();
        return getStorageNames(dir);
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible en local qui
     * ne sont pas encore sur le serveur CVS
     * 
     * @return liste de noms de regions
     */
    static public List<String> getNewScriptNames() {
        List<String> result = getScriptNames();
        result.removeAll(getRemoteScriptNames());
        return result;
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible sur le
     * serveur CVS
     * 
     * @return la liste des noms de toutes les régions disponible sur le serveur
     *         CVS. Si le serveur n'est pas disponible la liste retournée est
     *         vide.
     */
    static public List<String> getRemoteScriptNames() {
        File dir = getScriptDirectory();
        return getRemoteStorageNames(dir);

    }

    /**
     * Retourne la liste des noms de toutes les régions disponible sur le
     * serveur CVS qui ne sont pas encore en local
     * 
     * @return liste de noms de regions
     */
    static public List<String> getNewRemoteScriptNames() {
        List<String> result = getRemoteScriptNames();
        result.removeAll(getScriptNames());
        return result;
    }

    /**
     * @return the @Doc of the underlied rule class
     * @see DocHelper
     * @see Doc
     * @see Docable
     */
    public Doc getClassDoc() {
        Doc result=null;
        try {
            Class<?> klazz = getCodeClass();
            result = DocHelper.getClassDoc(klazz);
        } catch (IsisFishException e) {
            log.warn(_("isisfish.error.not.found.code",this));
        }
        return result;
    }

    /**
     * @param fieldName name of the field to inspect
     * @return the @Doc of the given field of the underlied Rule class
     * @see DocHelper
     * @see Doc
     * @see Docable
     */
    public Doc getFieldDoc(String fieldName) {
        Doc result=null;
        try {
            Class<?> klazz = getCodeClass();
            result = DocHelper.getFieldDoc(klazz,fieldName);
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.field",fieldName,this));
        }
        return result;
    }


    /**
     * <b>Be ware this method require to instanciate a Rule, so
     * it would be better to call as often as possible.</b>
     *
     * @return the descript of the instanciate AnalysePlan
     * @see Doc
     * @see Docable
     */
    public String getDescription() {
        throw new IsisFishRuntimeException(_("isisfish.error.not.support.class",this));
    }
}
